/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/
#include <QRegExp>

#include "CCentralResources.h"
#include "CMainWindow.h"
#include "CGarminDBMap.h"
#include "CGarminDBWpt.h"
#include "CGarminDBTrack.h"
#include "CGarminDBRoute.h"
#include "CGarminLiveLog.h"
#include "CDlgDevice.h"
#include "CDlgDeviceNag.h"

#include <config.h>

#include <IDevice.h>

#include <qglobal.h>
#include <QtGui>

#define FT2M(x) (int)((double)x * 0.3048 + 0.5)
#define XSTR(x) STR(x)
#define STR(x) #x

#define TIME_OFFSET (631018800 + 12*3600)

CCentralResources * gpResources = 0;

/**
  @param progress the progress as integer from 0..100, if -1 no progress bar needed.
  @param ok if this pointer is 0 no ok button needed, if non zero set to 1 if ok button pressed
  @param cancel if this pointer is 0 no cancel button needed, if non zero set to 1 if cancel button pressed
  @param title dialog title as C string
  @param msg dialog message C string to display
  @param self void pointer as provided while registering the callback
*/
void GUICallback(int progress, int * ok, int * cancel, const char * title, const char * msg, void * self)
{
    CCentralResources * parent = static_cast<CCentralResources *>(self);
    CCentralResources::dlgdata_t& dd = parent->dlgData;

    if(progress != -1) {
        quint32 togo, hour, min, sec;
        QString message;

        if(dd.dlgProgress == 0) {
            dd.canceled     = false;
            dd.dlgProgress  = new QProgressDialog(QString(title),0,0,100,&parent->main,
                Qt::WindowStaysOnTopHint);
            dd.timeProgress.start();
            if(cancel) {
                QPushButton * butCancel = new QPushButton(QObject::tr("Cancel"),dd.dlgProgress);
                parent->connect(butCancel, SIGNAL(clicked()), parent, SLOT(slotCancel()));
                dd.dlgProgress->setCancelButton(butCancel);
            }
        }

        if(title) dd.dlgProgress->setWindowTitle(QString(title));

        togo = (quint32)((100.0 * (double)dd.timeProgress.elapsed() / (double)progress) + 0.5);
        togo = (quint32)((double)(togo - dd.timeProgress.elapsed()) / 1000.0 + 0.5);

        hour = (togo / 3600);
        min  = (togo - hour * 3600) / 60;
        sec  = (togo - hour * 3600 - min * 60);

        message.sprintf(QObject::tr("\n\nEstimated finish: %02i:%02i:%02i [hh:mm:ss]").toUtf8(),hour,min,sec);

        dd.dlgProgress->setLabelText(QString(msg) + message);
        dd.dlgProgress->setValue(progress);

        if(progress == 100 && dd.dlgProgress) {
            delete dd.dlgProgress;
            dd.dlgProgress = 0;
        }

        if(cancel) {
            *cancel = dd.canceled;
        }

        qApp->processEvents();

    }
    else {
        if(ok && cancel) {
            QMessageBox::StandardButtons res = QMessageBox::question(&parent->main,QString(title),QString(msg),QMessageBox::Ok|QMessageBox::Cancel,QMessageBox::Cancel);
            *ok     = res == QMessageBox::Ok;
            *cancel = res == QMessageBox::Cancel;
        }
        else if(ok && !cancel) {
            QMessageBox::question(&parent->main,QString(title),QString(msg),QMessageBox::Ok,QMessageBox::Ok);
            *ok     = true;
        }
        else if(!ok && cancel) {
            QMessageBox::question(&parent->main,QString(title),QString(msg),QMessageBox::Cancel,QMessageBox::Cancel);
            *cancel     = true;
        }
        else if(!ok && !cancel) {
            //kiozen - that doesn't work nicely
            //             QMessageBox * dlg = new QMessageBox(&parent->main);
            //             dlg->setWindowTitle(QString(title));
            //             dlg->setText(QString(msg));
            //             dlg->setStandardButtons(QMessageBox::NoButton);
            //             dlg->setIcon(QMessageBox::Information);
            //             dlg->show();
            //             qApp->processEvents(QEventLoop::AllEvents, 1000);
            //             sleep(3); // sleep for 3 seconds
            //             delete dlg;
        }
    }
}


CCentralResources::CCentralResources(CMainWindow& parent)
: QObject(&parent)
, main(parent)
, m_useHttpProxy(false)
, m_httpProxyPort(8080)
, m_eBrowser(eFirefox)
, cmdFirefox("firefox \"%s\" &")
, cmdKonqueror("kfmclient exec \"%s\"")
, time_offset(TIME_OFFSET)
, uploadMap(true)
, uploadWpt(true)
, uploadRte(true)
, downloadMap(true)
, downloadWpt(true)
, downloadTrk(true)
, doDriverNag(true)
, flipMouseWheel(false)
{
    gpResources = this;

    try
    {
        m_pLiveLog  = new CGarminLiveLog(main.tabWidget);
        m_pMapDB    = new CGarminDBMap(main.tabWidget);
        m_pWptDB    = new CGarminDBWpt(main.tabWidget);
        m_pTrackDB  = new CGarminDBTrack(main.tabWidget, main.rightSplitter);
        m_pRouteDB  = new CGarminDBRoute(main.tabWidget);
        m_pCanvas   = new CCanvas(&main);
    }
    catch(const QString& msg) {
        qDebug() << msg;
    }
    // reset and set sticky element. That's a good idea?
    m_pMapDB->clear();
    m_pWptDB->clear();
    m_pTrackDB->clear();

    main.rightSplitter->insertWidget(0,m_pCanvas);

    m_pFSM = new CFunctionStateMachine(main);
    main.leftSplitter->addWidget(m_pFSM);

    QSettings cfg;
    QString family  = cfg.value("environment/mapfont/family","Arial").toString();
    int size        = cfg.value("environment/mapfont/size",8).toInt();
    m_mapfont = QFont(family,size);

    m_doMetric        = cfg.value("environment/doMetric",true).toBool();
    m_offsetUTC       = cfg.value("environment/UTCOffset",100).toInt();
    m_offsetUTCfract  = cfg.value("environment/UTCOffsetFract",0).toInt();

    if(m_offsetUTC == 100) {
        QMessageBox::information(0,tr("Time offset ...")
            ,tr("QLandkarte assumes that your hardware clock is "
            "set to UTC time. You must setup the correct "
            "time offset in 'Setup->Config'")
            ,QMessageBox::Ok,QMessageBox::NoButton);
    }
    else {
        setUTCOffset(m_offsetUTC, m_offsetUTCfract);
    }

    flipMouseWheel    = cfg.value("environment/flipMouseWheel",flipMouseWheel).toBool();

    m_useHttpProxy    = cfg.value("network/useProxy",m_useHttpProxy).toBool();
    m_httpProxy       = cfg.value("network/proxy/url",m_httpProxy).toString();
    m_httpProxyPort   = cfg.value("network/proxy/port",m_httpProxyPort).toUInt();

    m_eBrowser        = (bowser_e)cfg.value("network/browser",m_eBrowser).toInt();
    cmdOther          = cfg.value("network/browser/other","my command \"%s\"").toString();

    doDriverNag       = cfg.value("device/nagscreen",doDriverNag).toBool();
    m_DeviceName      = cfg.value("device/name",m_DeviceName).toString();

    uploadMap         = cfg.value("device/upload/maps",true).toBool();
    uploadWpt         = cfg.value("device/upload/waypoints",true).toBool();
    uploadRte         = cfg.value("device/upload/routes",true).toBool();
    downloadMap       = cfg.value("device/download/maps",true).toBool();
    downloadWpt       = cfg.value("device/download/waypoints",true).toBool();
    downloadTrk       = cfg.value("device/download/tracks",true).toBool();

    serialPort        = cfg.value("device/port","/dev/ttyS0").toString();

    emit sigProxyChanged();

    altPluginPath.clear();
    QStringList env = QProcess::systemEnvironment();
    env = env.filter(QRegExp("QLANDKARTE_LIBDIR=.*"));
    if(!env.isEmpty()) {
        altPluginPath = env[0].section('=',1,1);
        qDebug() << "Use alt. plugin path:" << altPluginPath;
    }
}


CCentralResources::~CCentralResources()
{
    QSettings cfg;
    cfg.setValue("environment/mapfont/family",m_mapfont.family());
    cfg.setValue("environment/mapfont/size",m_mapfont.pointSize());
    cfg.setValue("environment/doMetric",m_doMetric);
    cfg.setValue("environment/UTCOffset",m_offsetUTC);
    cfg.setValue("environment/UTCOffsetFract",m_offsetUTCfract);
    cfg.setValue("environment/flipMouseWheel",flipMouseWheel);

    cfg.setValue("network/useProxy",m_useHttpProxy);
    cfg.setValue("network/proxy/url",m_httpProxy);
    cfg.setValue("network/proxy/port",m_httpProxyPort);

    cfg.setValue("network/browser",m_eBrowser);
    cfg.setValue("network/browser/other",cmdOther);

    cfg.setValue("device/nagscreen",doDriverNag);
    cfg.setValue("device/name",m_DeviceName);
    cfg.setValue("device/upload/maps",uploadMap);
    cfg.setValue("device/upload/waypoints",uploadWpt);
    cfg.setValue("device/upload/routes",uploadRte);
    cfg.setValue("device/download/maps",downloadMap);
    cfg.setValue("device/download/waypoints",downloadWpt);
    cfg.setValue("device/download/tracks",downloadTrk);

    cfg.setValue("device/port",serialPort);

    gpResources = 0;
}


void CCentralResources::slotCancel()
{
    dlgData.canceled  = true;
}


const QString CCentralResources::distance2str(double d)
{
    QString str;
    if(doMetric()) {
        if(d > 1000) {
            str = QString("%1 km").arg(d/1000,0,'f',2);
        }
        else {
            str = QString("%1 m").arg(d,0,'f',0);
        }
    }
    else {
        if(d > 1609.344) {
            str = QString("%1 ml").arg(d * 0.6213699E-3,0,'f',2);
        }
        else {
            str = QString("%1 ft").arg(d / 0.305,0,'f',0);
        }
    }
    return str;
}


double CCentralResources::str2distance(const QString& s)
{
    QString str(s);
    if(doMetric()) {
        if(str.contains(" km")) {
            return (1000 * str.remove(" km").toDouble());
        }
        else if (str.contains(" m")) {
            return str.remove(" m").toDouble();
        }
    }
    else {
        if(str.contains(" ml")) {
            return (str.remove(" ml").toDouble() / 0.6213699E-3);
        }
        else if (str.contains(" ft")) {
            return (str.remove(" ft").toDouble() * 0.305);
        }
    }
    return -1;
}


const QString CCentralResources::altitude2str(double a)
{
    QString str;
    if(doMetric()) {
        str.sprintf("%1.0f m",a);
    }
    else {
        str.sprintf("%1.0f ft",a * 3.28084);
    }
    return str;
}


double CCentralResources::str2altitude(const QString& s)
{
    QString str(s);
    if(doMetric()) {
        return str.remove(" m").toDouble();
    }
    else {
        return (str.remove(" ft").toDouble() * 3.28084);
    }
    return -1;
}


const QString CCentralResources::speed2str(double s)
{
    QString str;
    if(doMetric()) {
        if (s < 10.0) {
            str.sprintf("%1.2f km/h",s);
        }
        else {
            str.sprintf("%1.0f km/h",s);
        }
    }
    else {
        str.sprintf("%1.2f ml/h",s * 0.6213699);
    }
    return str;
}


double CCentralResources::str2speed(const QString& s)
{
    QString str(s);
    if(doMetric()) {
        return str.remove(" km/h").toDouble();
    }
    else {
        return (str.remove(" ml/h").toDouble() / 0.6213699);
    }
    return -1;
}


void CCentralResources::upload()
{
    if(uploadMap) {
        m_pMapDB->uploadSelectedTiles(false);
    }

    if(uploadWpt) {
        m_pWptDB->uploadWaypoints();
    }

    if(uploadRte) {
        m_pRouteDB->uploadRoutes();
    }

}


void CCentralResources::download()
{
    if(downloadMap) {
        m_pMapDB->downloadMapInfo();
    }

    if(downloadWpt) {
        m_pWptDB->downloadWaypoints();
    }

    if(downloadTrk) {
        m_pTrackDB->downloadTracks();
    }
}


void CCentralResources::setUTCOffset(int offset, int fract)
{
    time_offset = TIME_OFFSET + offset * 3600 + fract * 60;
}


void CCentralResources::openLink(const QString& link)
{
    QString cmd;
    if(m_eBrowser == eFirefox) {
        cmd.sprintf(cmdFirefox.toAscii(),link.toAscii().data());
    }
    else if(m_eBrowser == eKonqueror) {
        cmd.sprintf(cmdKonqueror.toAscii(),link.toAscii().data());
    }
    else if(m_eBrowser == eOther) {
        cmd.sprintf(cmdOther.toAscii(),link.toAscii().data());
    }
    else {
        return;
    }

    system(cmd.toAscii());

}


bool CCentralResources::getHttpProxy(QString& url, quint16& port)
{
    url  = m_httpProxy;
    port = m_httpProxyPort;
    return m_useHttpProxy;
}


void CCentralResources::setCurrentToolBoxWidget(QWidget * w)
{
    main.tabWidget->setCurrentWidget(w);
}


void CCentralResources::convertHeight(QString& str)
{
    bool ok;
    quint32 val = str.toUInt(&ok,10);
    if(ok) {
        if(m_doMetric) {
            str = QString("%1m").arg(FT2M(val));
        }
        else {
            str = QString("%1ft").arg(val);
        }
    }
}


Garmin::IDevice * CCentralResources::device()
{
    Garmin::IDevice * dev = 0;
    Garmin::IDevice * (*func)(const char*);

    while(1) {

        if(m_DeviceName.isEmpty()) {
            doDriverNag = true;
            CDlgDevice dlg(0);
            if(dlg.exec() == QDialog::Rejected) {
                return 0;
            }
        }
#if defined(QT_ARCH_WINDOWS)
        QString path        = altPluginPath.isEmpty() ? XSTR(QL_LIBDIR) : altPluginPath;
        QString libname, funcname;
        if ( m_DeviceName == "NMEATcp" ) {
            libname     = QString("%1/NMEATcp.dll").arg(path);
            funcname    = QString("init%1").arg(m_DeviceName);
        }
        else if ( m_DeviceName == "infoAboutMyGarmin" ) {
            libname     = QString("%1/whatGarmin.dll").arg(path);
            funcname    = QString("initwhatGarmin");
        }
        else {
            libname     = QString("%1/GPSMap60CSx.dll").arg(path);
            funcname    = QString("init%1").arg(m_DeviceName);
        }
        //        QMessageBox::information(0,"path=",path);
        //        QMessageBox::information(0,"libname=",libname);
        //        QMessageBox::information(0,"funcname=",funcname);
#else
#if defined(QT_ARCH_MACOSX)
        // MacOS X: plug-ins are stored in the bundle folder
        QString path        = altPluginPath.isEmpty() ?
            QCoreApplication::applicationDirPath().replace(QRegExp("MacOS$"), "Resources/plugins") :
        altPluginPath;
#else
        // UNIX: first find the plug-in in the folder defined in the environment
        // or in the build folder
        QString path        = altPluginPath.isEmpty() ?
            QCoreApplication::applicationDirPath() + "/plugins" : altPluginPath;
#endif
        QString libname     = QString("%1/lib%2" SOEXT).arg(path).arg(m_DeviceName);
        QString funcname    = QString("init%1").arg(m_DeviceName);
#endif

        func = (Garmin::IDevice * (*)(const char*))QLibrary::resolve(libname,funcname.toAscii());

#if !defined(QT_ARCH_WINDOWS) && !defined(QT_ARCH_MACOSX)
        // UNIX: not found, fall back to standard install folder
        if (func == 0) {
            path    = XSTR(QL_LIBDIR);
            libname = QString("%1/lib%2" SOEXT).arg(path).arg(m_DeviceName);
            func    = (Garmin::IDevice * (*)(const char*))QLibrary::resolve(libname,funcname.toAscii());
        }
#endif

        if(func == 0) {
            QMessageBox::warning(0,tr("Error ..."),tr("Failed to load driver."),QMessageBox::Ok,QMessageBox::NoButton);
            m_DeviceName.clear();
            continue;
        }
        dev = func(INTERFACE_VERSION);
        if(dev == 0) {
            QMessageBox::warning(0,tr("Error ..."),tr("Driver version mismatch."),QMessageBox::Ok,QMessageBox::NoButton);
            m_DeviceName.clear();
            continue;
        }
        break;
    }

    if(doDriverNag) {
        CDlgDeviceNag dlg;
        if(dlg.exec() == QDialog::Rejected) {
            return 0;
        }
    }

    dev->setPort(serialPort.toLatin1());
    dev->setGuiCallback(GUICallback,this);

    return dev;
}
