/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/

#include "gmapsupp.h"
#include "GarminTypedef.h"

#include <QtCore>

#undef DEBUG_SHOW_SECT_DESC

struct gms_subfile_part_t
{
    gms_subfile_part_t() : offset(0), size(0), nBlocks(0), nFATBlocks(0){}
    /// file offset of subfile part
    quint32 offset;
    /// size of the subfile part
    quint32 size;
    /// number of blocks used by data
    quint32 nBlocks;
    /// number of FAT blocks used by subfile desc.
    quint32 nFATBlocks;
};


struct gms_subfile_desc_t
{
    gms_subfile_desc_t() : pRawData(0) {};
    gms_subfile_desc_t(quint8 * pRawData) : pRawData(pRawData) {}
    /// internal filename
    QString name;
    /// location information of all parts
    QMap<QString,gms_subfile_part_t> parts;
    /// pointer to file buffer
    quint8 * pRawData;
};

#pragma pack(1)
/// special img fileheader for gmapsupp files
struct gms_imghdr_t
{
    quint8  xorByte;            ///< 0x00000000
    quint8  byte0x00000001_0x00000009[9];
    quint8  upMonth;              ///< 0x0000000A
    quint8  upYear;               ///< 0x0000000B
    quint8  byte0x0000000C_0x0000000F[4];
    char    signature[7];       ///< 0x00000010 .. 0x00000016
    quint8  byte0x00000017_0x00000040[42];
    char    identifier[7];      ///< 0x00000041 .. 0x00000047
    quint8  byte0x00000048;
    char    desc1[20];          ///< 0x00000049 .. 0x0000005C
    quint8  byte0x0000005D_0x00000060[4];
    quint8  e1;                 ///< 0x00000061
    quint8  e2;                 ///< 0x00000062
    quint16 nBlocks1;           ///< 0x00000063 .. 0x00000064
    char    desc2[31];          ///< 0x00000065 .. 0x00000083
    quint8  byte0x00000084_0x000001C9[0x146];
    quint16 nBlocks2;           ///< 0x000001CA .. 0x000001CB
    quint8  byte0x0000001CC_0x000001FD[0x32];
    quint16 terminator;         ///< 0x000001FE .. 0x000001FF
    quint8  byte0x00000200_0x0000040B[0x20C];
    quint32 dataoffset;         ///< 0x0000040C .. 0x0000040F
    quint8  byte0x00000410_0x00000FFF[0xBF0];

    quint32 blocksize(){return 1 << (e1 + e2);}
};

struct gms_mps_map_t
{
    quint8  tok;                ///< should be 0x4c
    quint16 size;               ///< total entry size will be size + sizeof(tok) + sizeof(size)
    quint16 product;            ///< product code as it can be found in the registry
    quint16 dummy;              ///< country / char. set?
    quint32 mapId;              ///< same id as in tdb and map filename
    char    name1[];
    /*
        There are several 0 terminated strings:
        "map name"
        "tile name"
    */
};
/*
    quint32 ???                 ///< the map id as it is used in the FAT table
    quint32 ???                 ///< always 0x00000000 (end token?)
*/

#pragma pack(0)


void initFATBlock(FATblock_t * pFAT)
{
    pFAT->flag = 0x01;
    memset(pFAT->name,0x20,sizeof(pFAT->name));
    memset(pFAT->type,0x20,sizeof(pFAT->type));
    memset(pFAT->byte0x00000012_0x0000001F,0x00,sizeof(pFAT->byte0x00000012_0x0000001F));
}

void mkGMapSupp(QList<gms_map_def_t> & files, const QSet<QString>& maps, const QSet<QString>& keys, QByteArray& result)
{
    unsigned i;
    char tmpstr[64];
    // data arrays of all mapfiles to combine
    QVector<QByteArray> inputData;
    // all subfile descriptors by internal filename
    QMap<QString,gms_subfile_desc_t> subfiles;
    // largest block size found in subfiles
    size_t largestBlocksize = 0;
    // over all block count
    quint16 blockcnt = 0;
    // the MAPSOURC.MPS section
    QByteArray mps;

    result.clear();

    QList<gms_map_def_t>::iterator file = files.begin();
    while(file != files.end()){
        // read file data as one big chunk
        QFile f(file->filename);
        if(!f.open(QIODevice::ReadOnly)){
            throw QObject::tr("Failed to open ") + file->filename;
        }
        inputData << f.readAll();
        // get the actual file size
        const quint64 fsize = f.size();
        f.close();

        // setup base pointer to byte array
        quint8 * const pRawData = (quint8*)inputData.last().data();

        // descramble data, if necessary
        if(*pRawData != '\0'){
            quint8 hash = *pRawData;
            quint8 *p   =  pRawData;
            size_t cnt  = 0;

            do{
                *p = (*p) ^ hash;
                ++p;++cnt;
            } while(cnt != fsize);
        }

        // test for garmin map file
        hdr_img_t * imghdr = (hdr_img_t *)pRawData;
        if(strncmp(imghdr->signature,"DSKIMG",7) != 0){
            throw QObject::tr("Bad file format: ") + file->filename;
        }
        if(strncmp(imghdr->identifier,"GARMIN",7) != 0){
            throw QObject::tr("Bad file format: ") + file->filename;
        }

        // The block size is important to generate the output file.
        // This should use the largest blocksize of all subfiles.
        const size_t blocksize = imghdr->blocksize();
        if(largestBlocksize < blocksize) largestBlocksize = blocksize;

        // read FAT
        const FATblock_t * pFAT = (const FATblock_t *)(pRawData + sizeof(hdr_img_t));
        size_t dataoffset = sizeof(hdr_img_t);

        // skip dummy blocks at the beginning
        while(dataoffset < fsize){
            if(pFAT->flag != 0x00){
                break;
            }
            dataoffset += sizeof(FATblock_t);
            ++pFAT;
        }

        while(dataoffset < fsize){
            if(pFAT->flag != 0x01){
                break;
            }

            // start of new subfile part
            /*
                It is taken for granted that the single subfile parts are not
                fragmented within the file. Thus it is not really neccessary to
                store and handle all block sequence numbers. Just the first one
                will give us the offset. This also implies that it is not necessary
                to care about FAT blocks with a non-zero part number.
            */
            if(pFAT->part == 0){
                memcpy(tmpstr,pFAT->name,sizeof(pFAT->name));
                tmpstr[sizeof(pFAT->name)] = 0;

                if(!subfiles.contains(tmpstr)){
                    file->intname = tmpstr;
                    subfiles[tmpstr] = gms_subfile_desc_t(pRawData);
                }

                gms_subfile_desc_t& subfile = subfiles[tmpstr];
                subfile.name = tmpstr;

                memcpy(tmpstr,pFAT->type,sizeof(pFAT->type));
                tmpstr[sizeof(pFAT->type)] = 0;

                gms_subfile_part_t& part = subfile.parts[tmpstr];
                part.size   = pFAT->size;
                part.offset = pFAT->blocks[0] * blocksize;

            }
            dataoffset += sizeof(FATblock_t);
            ++pFAT;
        }

        if((dataoffset == sizeof(imghdr)) || (dataoffset >= fsize)){
            throw QObject::tr("Failed to read file structure: ") + file->filename;
        }

#ifdef DEBUG_SHOW_SECT_DESC
    {
        QMap<QString,gms_subfile_desc_t>::const_iterator subfile = subfiles.begin();
        while(subfile != subfiles.end()){
            qDebug() << "--- subfile" << subfile->name << "---";
            QMap<QString,gms_subfile_part_t>::const_iterator part = subfile->parts.begin();
            while(part != subfile->parts.end()){
                qDebug() << part.key() << hex << part->offset << part->size;
                ++part;
            }
            ++subfile;
        }
    }
#endif //DEBUG_SHOW_SECT_DESC
        ++file;
    }

    if(subfiles.isEmpty()){
        return;
    }

    // generate MAPSOURCE.MPS section
    QDataStream s(&mps,QIODevice::WriteOnly);
    s.setByteOrder(QDataStream::LittleEndian);

    // write descriptors for tile files first
    file = files.begin();
    while(file != files.end()){
        s << (quint8)'L';
        s << (quint16)(16 + ((file->mapname.size() + 1)<<1) + file->tilename.size() + 1);
        s << (quint32)0x02180001; // ???
        s << (quint32)file->id;
        s.writeRawData(file->mapname.toAscii(),file->mapname.size() + 1);
        s.writeRawData(file->tilename.toAscii(),file->tilename.size() + 1);
        s.writeRawData(file->mapname.toAscii(),file->mapname.size() + 1);

        // ??? wow. :-/
        if(file->intname[0].isDigit()){
            s << (quint32)file->intname.toInt(0);
        }
        else{
            s << (quint32)file->intname.mid(1).toInt(0,16);
        }

        s << (quint32)0;
        ++file;
    }

    // next the keys for locked maps
    QSet<QString>::const_iterator key = keys.begin();
    while(key != keys.end()){
        s << (quint8)'U' << (quint16)26;
        s.writeRawData(key->toAscii(),26);
        ++key;
    }

    // finally the used maps
    QSet<QString>::const_iterator map = maps.begin();
    while(map != maps.end()){
        s << (quint8)'F';
        s << (quint16)(map->size() + 1 + 4);
        s << (quint32)0x02180001; // ???
        s.writeRawData(map->toAscii(),map->size() + 1);
        ++map;
    }

    QDateTime date = QDateTime::currentDateTime();

    // create gmapsupp.img file header
    gms_imghdr_t imghdr;
    memset(&imghdr,0,sizeof(gms_imghdr_t));
    imghdr.upMonth    = date.date().month();
    imghdr.upYear     = date.date().year() - 1900;
    memset(&imghdr.desc1,0x20,sizeof(imghdr.desc1));
    memset(&imghdr.desc2,0x20,sizeof(imghdr.desc2) - 1);
    strncpy(imghdr.signature,"DSKIMG",8);
    strncpy(imghdr.identifier,"GARMIN",8);
    imghdr.e1 = 0x09;
    size_t bs = largestBlocksize >> 9;
    while(bs != 1){
        ++imghdr.e2;
        bs >>= 1;
    }

    imghdr.terminator = 0xAA55;
    quint8 * p = (quint8*)&imghdr;

    *(p + 0x0E)               = 0x01;
    *(p + 0x17)               = 0x02;
    *(quint16*)(p + 0x18)     = 0x0020;
    *(quint16*)(p + 0x1A)     = 0x0020;
    *(quint16*)(p + 0x1C)     = 0x03C7;
    *(quint16*)(p + 0x5D)     = 0x0020;
    *(quint16*)(p + 0x5F)     = 0x0020;


    // date stuff
    *(quint16*)(p + 0x39)     = date.date().year();
    *(p + 0x3B)               = date.date().month();
    *(p + 0x3C)               = date.date().day();
    *(p + 0x3D)               = date.time().hour();
    *(p + 0x3E)               = date.time().minute();
    *(p + 0x3F)               = date.time().second();
    *(p + 0x40)               = 0x08;  ///< this has to be 0x08 ???


    *(quint16*)(p + 0x1C4)    = 0x0020;
/*  // not really needed ???
    *(p + 0x1C0)               = 0x01;
    *(p + 0x1C3)               = 0x15;
    *(p + 0x1C4)               = 0x10;
    *(p + 0x1C5)               = 0x00;
*/

    memcpy(imghdr.desc1,"QLandkarte",10);

    //qDebug() << "e1:" << hex << imghdr.e1 << "e2:" << hex << imghdr.e2 << "blocksize" << dec << imghdr.blocksize();

    quint32 nBlocks     = 0;
    quint32 nFATBlocks  = 0;

    // calc. number of FAT Blocks
    QMap<QString,gms_subfile_desc_t>::iterator subfile = subfiles.begin();
    while(subfile != subfiles.end()){

        QMap<QString,gms_subfile_part_t>::iterator part = subfile->parts.begin();
        while(part != subfile->parts.end()){
            part->nBlocks       = part->size / largestBlocksize + (part->size % largestBlocksize ? 1 : 0);
            part->nFATBlocks    = part->nBlocks / 240 + (part->nBlocks % 240 ? 1 : 0);
            nBlocks            += part->nBlocks;
            nFATBlocks         += part->nFATBlocks;
            ++part;
        }
        ++subfile;
    }

    // MAPSOURC.MPS FAT section
    quint32 mpsBlocks       = mps.size() / largestBlocksize + (mps.size() % largestBlocksize ? 1 : 0);
    quint32 mpsFATBlocks    = mpsBlocks / 240 + (mpsBlocks % 240 ? 1 : 0);
    nBlocks                += mpsBlocks;
    nFATBlocks             += mpsFATBlocks;


    /**
        TODO: this calculation of the file size is not correct as it only reserves
        one FAT block to define the file header. If the header is larger than
        240 * largestBlocksize it needs additional FAT blocks. This will be quite a
        monster of a file, but who knows? SD memory cards get bigger and bigger. Or
        did Garmin plan to increase the block size in this case?
    */
    quint32 fsize = (sizeof(gms_imghdr_t) + sizeof(FATblock_t) + nFATBlocks * sizeof(FATblock_t) + nBlocks * largestBlocksize);
    //qDebug() << "filesize" << fsize;
    //qDebug() << nFATBlocks << nBlocks;

    *(quint16*)(p + 0x1CA) = fsize / largestBlocksize; ///< number of file blocks
    imghdr.nBlocks1 = (fsize / largestBlocksize); ///< number of file blocks

    // initialize file
    result.fill(0xFF,fsize);

    // write file header
    memcpy(result.data(),&imghdr, sizeof(gms_imghdr_t));

    /////// declare and initialize variables to copy data to result ///////
    // get pointer to start of FAT block
    FATblock_t * pFAT   = (FATblock_t*)(result.data() + sizeof(gms_imghdr_t));
    // this is the pointer to the block section of each FAT
    quint16 * pBlock    = 0;
    // source file pointer
    quint8 * pSrc       = 0;
    // target file pointer
    quint8 * pTar       = 0;

    // write file header FAT block
    initFATBlock(pFAT);
    pFAT->size = sizeof(gms_imghdr_t) + sizeof(FATblock_t) + nFATBlocks * sizeof(FATblock_t);
    pFAT->part = 3; //???

    pBlock = pFAT->blocks; i = 0;
    do{
        if(blockcnt > 240) throw QObject::tr("File is getting too big. This is an internal problem of QLandkarte and has to be fixed.");
        *pBlock++ = blockcnt++; i += largestBlocksize;
    } while(i < pFAT->size);

    // needed later to disable FAT blocks until start of data section
    quint32 dataoffset = blockcnt * largestBlocksize;

    // copy the parts of all subfiles to result
    subfile = subfiles.begin();
    while(subfile != subfiles.end()){

        QMap<QString,gms_subfile_part_t>::iterator part = subfile->parts.begin();
        while(part != subfile->parts.end()){

            // write 1st FAT block of this part
            initFATBlock(++pFAT);
            memcpy(pFAT->name,subfile.key().toLatin1(),sizeof(pFAT->name));
            memcpy(pFAT->type,part.key().toLatin1(),sizeof(pFAT->type));
            pFAT->size = part->size;
            pFAT->part = 0;

            // copy data from source to target
            pSrc = subfile->pRawData + part->offset;
            pTar = (quint8*)result.data() + blockcnt * largestBlocksize;
            memcpy(pTar,pSrc,part->size);

            // write block information to FAT
            quint16 partno      = 0;
            quint16 blockcnt_   = 0;
            pBlock              = pFAT->blocks;
            i                   = 0;
            do{
                *pBlock++ = blockcnt++; i += largestBlocksize; ++blockcnt_;
                // switch to next FAT block every 240 blocks
                if(blockcnt_ == 240){
                    initFATBlock(++pFAT);
                    memcpy(pFAT->name,subfile.key().toLatin1(),sizeof(pFAT->name));
                    memcpy(pFAT->type,part.key().toLatin1(),sizeof(pFAT->type));
                    pFAT->size  = 0; /*part->size;*/
                    pFAT->part  = (++partno << 8); //TODO: this shift smells fishy
                    pBlock      = pFAT->blocks;
                    blockcnt_   = 0;
                }
            } while(i < part->size);

            ++part;
        }
        ++subfile;
    }


    // write MAPSOURCE.MPS section
    initFATBlock(++pFAT);
    memcpy(pFAT->name,"MAPSOURC",sizeof(pFAT->name));
    memcpy(pFAT->type,"MPS",sizeof(pFAT->type));
    pFAT->size = mps.size();
    pFAT->part = 0;

    // copy data from source to target
    pSrc = (quint8*)mps.data();
    pTar = (quint8*)result.data() + blockcnt * largestBlocksize;
    memcpy(pTar,pSrc,mps.size());

    // write block information to FAT
    quint16 partno      = 0;
    quint16 blockcnt_   = 0;
    pBlock              = pFAT->blocks;
    i                   = 0;
    do{
        *pBlock++ = blockcnt++; i += largestBlocksize; ++blockcnt_;
        // switch to next FAT block every 240 blocks

        if(blockcnt_ == 240){
            initFATBlock(++pFAT);
            memcpy(pFAT->name,"MAPSOURC",sizeof(pFAT->name));
            memcpy(pFAT->type,"MPS",sizeof(pFAT->type));
            pFAT->size  = mps.size();
            pFAT->part  = (++partno << 8); //TODO: this shift smells fishy
            pBlock      = pFAT->blocks;
            blockcnt_   = 0;
        }

    } while(i < (unsigned)mps.size());

    // mark empty FAT blocks until start of data section as dummy
    while((char*)(++pFAT) < (result.data() + dataoffset)){
      pFAT->flag = 0;
    }

}
