/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/

#ifndef IDEVICE_H
#define IDEVICE_H

#include <stdint.h>
#include <list>
#include <string>
#include <vector>

#define INTERFACE_VERSION "01.09"

namespace Garmin
{


    /// common waypoint structure application side
    /**
        This structure has to be used to exchange a waypoint to QLandkarte.
        If an item is missing you can simply add it. This structure must
        never be copied assuming a certain alignment.

        Most values are the same like the ones used in the waypoint data
        structures used by Garmin.
    */
    struct Wpt_t
    {
        Wpt_t()
            : wpt_class(0)
            , dspl_color(0)
            , dspl_attr(0)
            , smbl(8287)
            , lat(1000.0)
            , lon(1000.0)
            , alt(1.0e25)
            , dpth(1.0e25)
            , dist(1.0e25)
            , ete(0xFFFFFFFF)
            , temp(1.0e25)
            , time(0xFFFFFFFF)
            , wpt_cat(0)
            {
                strcpy(state,"  ");
                strcpy(cc,"  ");

            }
        /// same as Garmin spec.
        uint8_t  wpt_class;
        /// bit 0..4 of dspl_color
        uint8_t  dspl_color;
        /// bit 5..6 of dspl_color
        uint8_t  dspl_attr;
        /// same as Garmin spec.
        uint16_t smbl;
        /// the latitude as degrees
        double   lat;
        /// the longitude as degrees
        double   lon;
        /// same as Garmin spec.
        float    alt;
        /// same as Garmin spec.
        float    dpth;
        /// same as Garmin spec.
        float    dist;
        /// same as Garmin spec.
        char     state[3];
        /// same as Garmin spec.
        char     cc[3];
        /// same as Garmin spec.
        uint32_t ete;
        /// same as Garmin spec.
        float    temp;
        /// same as Garmin spec.
        uint32_t time;
        /// same as Garmin spec.
        uint16_t wpt_cat;
        /// same as Garmin spec.
        std::string ident;
        /// same as Garmin spec.
        std::string comment;
        /// same as Garmin spec.
        std::string facility;
        /// same as Garmin spec.
        std::string city;
        /// same as Garmin spec.
        std::string addr;
        /// same as Garmin spec.
        std::string crossroad;
    };

    /// common route point structure application side
    /**
        This structure has to be used to exchange a waypoint to QLandkarte.
        If an item is missing you can simply add it. This structure must
        never be copied assuming a certain alignment.

        Most values are the same like the ones used in the waypoint and
        route point data structures used by Garmin.
    */

    struct RtePt_t : public Wpt_t
    {
        RtePt_t()
            : rte_class(3)
            , subclass_1(0x0000)
            , subclass_2(0x00000000)
            , subclass_3(0xFFFFFFFF)
            , subclass_4(0xFFFFFFFF)
            , subclass_5(0xFFFFFFFF)
            {}

        uint16_t rte_class;

        uint16_t subclass_1;
        uint32_t subclass_2;
        uint32_t subclass_3;
        uint32_t subclass_4;
        uint32_t subclass_5;

        std::string ident;
    };

    /// common route structure application side
    /**
        This structure has to be used to exchange a track to QLandkarte.
        If an item is missing you can simply add it. This structure must
        never be copied assuming a certain alignment.

        Most values are the same like the ones used in the point data
        structures used by Garmin.
    */

    struct Route_t {
        /// same as Garmin spec.
        std::string ident;
        /// route points
        std::vector<RtePt_t> route;
    };

    /// common track point structure application side
    /**
        This structure has to be used to exchange a track point to QLandkarte.
        If an item is missing you can simply add it. This structure must
        never be copied assuming a certain alignment.

        Most values are the same like the ones used in the track point data
        structures used by Garmin.
    */
    struct TrkPt_t{
        TrkPt_t()
            : lat(0.0)
            , lon(0.0)
            , time(0)
            , alt(1e25f)
            , dpth(1e25f)
        {

        }
        /// the latitude as degrees
        double   lat;
        /// the longitude as degrees
        double   lon;
        /// the time in sec. as specified by Garmin
        uint32_t time;
        /// same as Garmin spec.
        float    alt;
        /// same as Garmin spec.
        float    dpth;
    };

    /// common track structure application side
    /**
        This structure has to be used to exchange a track to QLandkarte.
        If an item is missing you can simply add it. This structure must
        never be copied assuming a certain alignment.

        Most values are the same like the ones used in the point data
        structures used by Garmin.
    */
    struct Track_t{
        Track_t()
            : dspl(true)
            , color(0xFF)
        {

        }
        /// same as Garmin spec.
        bool    dspl;
        /// same as Garmin spec.
        uint8_t color;
        /// same as Garmin spec.
        std::string ident;
        /// trackpoints
        std::vector<TrkPt_t> track;
    };

    struct Map_t{
        std::string mapName;
        std::string tileName;
    };

    struct Pvt_t {
        /// same as Garmin spec.
        float    alt;
        /// same as Garmin spec.
        float    epe;
        /// same as Garmin spec.
        float    eph;
        /// same as Garmin spec.
        float    epv;
        /// same as Garmin spec.
        uint16_t fix;
        /// same as Garmin spec.
        double   tow;
        /// the latitude as degrees
        double   lat;
        /// the longitude as degrees
        double   lon;
        /// same as Garmin spec.
        float    east;
        /// same as Garmin spec.
        float    north;
        /// same as Garmin spec.
        float    up;
        /// same as Garmin spec.
        float    msl_hght;
        /// same as Garmin spec.
        int16_t  leap_scnds;
        /// same as Garmin spec.
        uint32_t wn_days;
    };


    struct Icon_t {
        Icon_t(){ memset(data,0,sizeof(data));}
        /// custom icon index (0..511)
        uint16_t idx;
        /// the bitmap's color table
        char clrtbl[0x400];
        /// the bitmap's image data
        char data[0x100];
    };

    /// exception error code
    enum exce_e
    {
         errOpen        ///< error during opening the link
        ,errSync        ///< error during sync. up sequence
        ,errWrite       ///< error during write access
        ,errRead        ///< error during read access
        ,errNotImpl     ///< error because of missing implementation
        ,errRuntime     ///< error during operation
        ,errBlocked     ///< error because access is blocked by another process
    };

    /// exception type
    struct exce_t
    {

        exce_t(exce_e err, const std::string& msg) : err(err), msg(msg) {}
        exce_e err;
        std::string msg;
    };

    /// interface class for device driver plugins
    /**
        This is the common interface to all devices. The application has use
        this definition to gain access to the plugin. Thus if you are an
        application programmer simply load the plugin, resolve and call
        the init function. The object you will get will be of type IDevice.
        There is no need to link against libgarmin.a.

        If you are a plugin programmer you will rather use IDeviceDefault, than
        IDevice. The inheritance chain will look like:

        IDevice -> IDefaultDevice -> CDevice

        Thus if you miss a public method you have to add it here as pure virtual
        and as a default implementation to IDeviceDefault.

        NOTE: If you change this interface you _must_ increment the version
        number defined by INTERFACE_VERSION. This is important to prevent
        crashes due to different interface definitions.

        Most likely your device driver will implement the protected methods with
        leading '_' of IDeviceDefault.
    */
    class IDevice
    {
        public:
            IDevice(){};
            virtual ~IDevice(){};

            /// upload a single map file to device
            /**
                This will handle just a single file. Map tiles must be concatenated into
                one big file (gmapsupp.img). If the file containes tiles with locked data
                an array of 25 ASCII digits has to be passed as key. As map uploads naturally
                take some time a callback function has to be given to signal the progress. The
                function pointer must be of type:

                void callback(uint32_t bytesTransmitted, uint32_t bytesTotal, void * data);

                In most cases data will be a pointer to the application's object conducting the
                upload.

                @param mapdata pointer to the gmapsupp.img data array
                @param size the size of the gmapsupp.img data array
                @param key pointer to 25 digit key or 0 for no key
                @param callback progress callback function
                @param data pointer to multipurpose data. This will be passed to the progress callback.
            */
            virtual void uploadMap(const uint8_t * mapdata, uint32_t size, const char * key, void (*callback)(uint32_t,uint32_t,void*), void* data) = 0;

            /// query loaded map list
            /**
                This is not a real download of maps as just the information about the
                loaded maps is transfered.
            */
            virtual void queryMap(std::list<Map_t>& maps) = 0;

            /// download waypoints from device
            /**
                @param waypoints list object to receive waypoints
            */
            virtual void downloadWaypoints(std::list<Garmin::Wpt_t>& waypoints) = 0;

            /// upload waypoints to device
            /**
                @param waypoints list of waypoints to send
            */
            virtual void uploadWaypoints(std::list<Garmin::Wpt_t>& waypoints) = 0;

            /// download track from device
            /**
                @param tracks list object to receive tracks
            */
            virtual void downloadTracks(std::list<Garmin::Track_t>& tracks) = 0;

            /// upload route to device
            /**
                @param routes list of routes
            */
            virtual void uploadRoutes(std::list<Garmin::Route_t>& routes) = 0;

            /// upload custom icons to device
            /**
                @param routes list of icons
            */
            virtual void uploadCustomIcons(std::list<Garmin::Icon_t>& icons) = 0;

            /// download a screenshot from the device
            /**
                @param clrtbl a pointer reference to be set to the downloaded color table of size 0x100
                @param data a pointer reference to be set to the downloaded image data array of size width x height
                @param width a integer reference to store the image width at
                @param height a integer reference to store the image height at
            */
            virtual void screenshot(char *& clrtbl, char *& data, int& width, int& height) = 0;

            /// switch device into realtime position mode
            virtual void setRealTimeMode(bool on) = 0;

            /// request real time position
            virtual void getRealTimePos(Garmin::Pvt_t& pvt) = 0;

            /// get the copyright notice of this driver
            virtual const std::string& getCopyright() = 0;

            /// get reason string for last exception
            virtual const std::string& getLastError() = 0;

            /// set port string used for communication
            /**
                This should be called prior to an operation. As an operation will
                create a new ILink object, a changed port setting will apply imediately.
                If the ILink object does not need any port settings this value is ignored.
            */
            virtual void setPort(const char * port) = 0;
    };

}
#endif //IDEVICE_H

