/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/
#include "CUSB.h"
#include "IDevice.h"

#include <iostream>
#include <sstream>
#include <assert.h>
#include <errno.h>

using namespace Garmin;
using namespace std;

#ifndef ETIMEDOUT             // included for windows by dr, copied from pthread.h
#  define ETIMEDOUT 10060     /* This is the value in winsock.h. */
#endif

#define EA(x) (x & USB_ENDPOINT_ADDRESS_MASK)

#define GUSB_DATA_AVAILABLE     2
#define GUSB_SESSION_START      5
#define GUSB_SESSION_STARTED    6

#undef DBG

#define DBG_LINE_SIZE 16


CUSB::CUSB()
    : busses(0)
    , udev(0)
    , interface(-1)
    , max_tx_size(0)
    , epBulkIn(-1)
    , epBulkOut(-1)
    , epIntrIn(-1)
    , doBulkRead(false)
    , productId(0)
    , softwareVersion(0)
    , protocolArraySize(-1)
{
    usb_init();
    usb_find_busses();
    usb_find_devices();
    busses = usb_get_busses();
}

CUSB::~CUSB()
{
    close();
}

void CUSB::open()
{
    assert(busses);

    usb_bus *bus = 0;

    for(bus = busses; bus; bus = bus->next) {
        struct usb_device * dev = 0;
        for (dev = bus->devices; dev; dev = dev->next) {
#ifdef DBG
            cout << hex << dev->descriptor.idVendor << " " << dev->descriptor.idProduct << endl;
#endif
            if(dev->descriptor.idVendor == GARMIN_VID){

                if(dev->descriptor.idProduct == G60CSX_PID){
                    start(dev);
                    break;
                }
            }
        }
    }

    if(udev == 0){
        throw exce_t(errOpen,"Is the unit connected?");
    }
}

void CUSB::close()
{
    if(udev){
        usb_release_interface(udev, interface);
        usb_close(udev);
        udev = 0;
    }
}

void CUSB::close2()
{
    if(udev){
        usb_release_interface(udev, interface);
        usb_reset(udev);
        udev = 0;
    }
}

void CUSB::debug(const char * mark, const Packet_t& data)
{
#ifndef DBG
    return;
#endif
    unsigned i;
    uint32_t size;
    unsigned bytes = DBG_LINE_SIZE;
    char buf[DBG_LINE_SIZE + 1];
    memset(buf,0x20,sizeof(buf));buf[DBG_LINE_SIZE] = 0;

    cout << mark << endl << "     ";

    const uint8_t * pData = (const uint8_t*)&data;


    size = data.size;
    if(size > GUSB_MAX_BUFFER_SIZE){
        cerr << "WARNING! Data size " << data.size << " exceeds buffer size." << endl;
        cerr << "Truncate to " << GUSB_MAX_BUFFER_SIZE << "." << endl;
        size = GUSB_PAYLOAD_SIZE;
    }

    for(i = 0; i < (size + GUSB_HEADER_SIZE); ++i){
        if(i && !(i % DBG_LINE_SIZE)){
            cout << " " << buf << endl << "     ";
            memset(buf,0x20,sizeof(buf));buf[DBG_LINE_SIZE] = 0;
            bytes = DBG_LINE_SIZE;
        }

        cout.width(2);
        cout.fill('0');
        cout << hex << (unsigned)pData[i] << " ";

        if(isprint(pData[i])){
            buf[i%DBG_LINE_SIZE] = pData[i];
        }
        else{
            buf[i%DBG_LINE_SIZE] = '.';
        }

        --bytes;

    }
    for(i=0; i < bytes; i++) cout << "   ";
    cout << " " << buf << dec << endl;

}

int CUSB::read(Packet_t& data)
{
    int res;

    data.type = 0;
    data.id   = 0;
    data.size = 0;

    if(doBulkRead){
        res = ::usb_bulk_read(udev,epBulkIn,(char*)&data,sizeof(data),3000);
        if(res > 0){
            debug("b >>",data);
        }
    }
    else{
        res = ::usb_interrupt_read(udev,epIntrIn,(char*)&data,sizeof(data),3000);
        if(res > 0){
            debug("i >>",data);
        }
    }

    // Some devices sending data on the interrupt pipe seem
    // to timeout occasionally. It seems to be save to ignore this
    // timeout.
    if(res == -ETIMEDOUT && !doBulkRead){
         res = 0;
    }

    // switch to bulk pipe
    if((res > 0) && (data.id == GUSB_DATA_AVAILABLE)){
        doBulkRead = true;
    }

    // switch to interrupt pipe on errors or zero size packages
    if(res <= 0){
        doBulkRead = false;
    }

    if(res < 0){
        stringstream msg;
        msg << "USB read failed:" << usb_strerror();
        throw exce_t(errRead,msg.str());
    }

    return res;
}

void CUSB::write(const Packet_t& data)
{
    unsigned size = GUSB_HEADER_SIZE + data.size;
    int res = ::usb_bulk_write(udev,epBulkOut,(char*)&data,size,3000);

    debug("b <<",data);

    if(res < 0){
        stringstream msg;
        msg << "USB bulk write failed:" << usb_strerror();
        throw exce_t(errWrite,msg.str());
    }

    /*
	   The Garmin protocol requires that packets that are exactly
	   a multiple of the max tx size be followed by a zero length
	   packet.
    */
    if (size && !(size % max_tx_size)) {
        ::usb_bulk_write(udev,epBulkOut,(char*)&data,0,3000);
#ifdef DBG
        cout << "b << zero size packet to terminate" << endl;
#endif
    }
}

void CUSB::start(struct usb_device *dev)
{
    int i;
    if(udev) return;

    udev = usb_open(dev);
    if(udev == 0){
        stringstream msg;
        msg << "Failed to open USB device: " << usb_strerror();
        throw exce_t(errOpen,msg.str());
    }



    if (usb_set_configuration(udev, dev->config->bConfigurationValue) < 0) {
        stringstream msg;
#if __linux__
        char drvnm[128];
        drvnm[0] = 0;
        msg << "Failed to configure USB: " << usb_strerror();

        usb_get_driver_np(udev, 0, drvnm, sizeof(drvnm)-1);

        if(strlen(drvnm) != 0){
            msg << "\n\nThe kernel driver '" << drvnm << "' is blocking. "
                << "Please use 'rmmod " << drvnm << "' as root to remove it temporarily. "
                << "You might consider to add 'blacklist " << drvnm << "' to your "
                << "modeprobe.conf, to remove the module permanently.";
        }
        throw exce_t(errOpen,msg.str());
#endif

        msg << "Failed to configure USB: " << usb_strerror();
        throw exce_t(errOpen,msg.str());
    }

    interface = dev->config->interface->altsetting->bInterfaceNumber;
    if (usb_claim_interface(udev, interface) < 0) {
        stringstream msg;
        msg << "Failed to claim USB interface: " << usb_strerror();
        throw exce_t(errOpen,msg.str());
    }

    max_tx_size = dev->descriptor.bMaxPacketSize0;
#ifdef DBG
    cout << "  max. packet size:" << max_tx_size << endl;
#endif

    for (i = 0; i < dev->config->interface->altsetting->bNumEndpoints; i++) {
        struct usb_endpoint_descriptor * ep;
        ep = &dev->config->interface->altsetting->endpoint[i];

        switch (ep->bmAttributes & USB_ENDPOINT_TYPE_MASK) {

            case USB_ENDPOINT_TYPE_BULK:
                if (ep->bEndpointAddress & USB_ENDPOINT_DIR_MASK){
#ifdef DBG
                    cout << "  epBulkIn " << hex << EA(ep->bEndpointAddress) << endl;
#endif
                    epBulkIn = EA(ep->bEndpointAddress);
                }
                else{
#ifdef DBG
                    cout << "  epBulkOut " << hex << EA(ep->bEndpointAddress) << endl;
#endif
                    epBulkOut = EA(ep->bEndpointAddress);
                }
                break;

            case USB_ENDPOINT_TYPE_INTERRUPT:
                if (ep->bEndpointAddress & USB_ENDPOINT_DIR_MASK){
#ifdef DBG
                    cout << "  epIntrIn " << hex << EA(ep->bEndpointAddress) << endl;
#endif
                    epIntrIn = EA(ep->bEndpointAddress);
                }
                break;
        }
    }

    if ((epBulkIn > 0) && (epBulkOut > 0) && (epIntrIn > 0)) {
        return;
    }

    throw exce_t(errOpen,"Failed to identify USB endpoints for this device.");
}


void CUSB::syncup(void)
{
    static const Packet_t gpack_session_start(GUSB_PROTOCOL_LAYER,GUSB_SESSION_START);
    Packet_t response;

    write(gpack_session_start);
    read(response);
    if(response.id == GUSB_SESSION_STARTED){
        Packet_t command;
        Packet_t response;

        command.type = GUSB_APPLICATION_LAYER;
        command.id   = Pid_Product_Rqst;
        command.size = 0;

        write(command);

	protocolArraySize = 0;
        while(read(response)){
            if(response.id == Pid_Product_Data){
                //TODO read data
                Product_Data_t * pData = (Product_Data_t*)response.payload;
                productId       = pData->product_id;
                softwareVersion = pData->software_version;
                productString   = pData->str;
#ifdef DBG
                cout << hex << productId << " " << dec << softwareVersion << " " << productString << endl;
#endif
            }

            if(response.id == Pid_Ext_Product_Data){
                //TODO read data
            }

            if(response.id == Pid_Protocol_Array){
                //TODO read data
                Protocol_Data_t * pData = (Protocol_Data_t*)response.payload;
                for(uint32_t i = 0; i < response.size; i += sizeof(Protocol_Data_t)){
#ifdef DBG
                    cout << "Protocol: "<< (char)pData->tag <<  dec << pData->data << endl;
#endif
		    ++protocolArraySize;
		    protocolArray[protocolArraySize].tag = pData->tag;
		    protocolArray[protocolArraySize].data = pData->data;
                    ++pData;
                }
		++protocolArraySize;
#ifdef DBG
		cout << "protocolArraySize:" << protocolArraySize << endl;
#endif
                //
                if(!doBulkRead) return;
            }
        }
        return;
    }

    throw exce_t(errSync,"Failed to sync. up with device");
}

uint16_t CUSB::getDataType(int data_no, char tag, uint16_t protocol)
{
    // Find the right tag D<Data_no> for <tag><protocol>
    for (uint32_t i=0; i < protocolArraySize-1-data_no; i++) {
        if ((char)protocolArray[i].tag == tag) {
	    if (protocolArray[i].data == protocol) {
	        // accept data_no=-1 as a protocol verification only
	        if (data_no == -1) return (uint16_t) 1;
		if ((char)protocolArray[i+1+data_no].tag == 'D') {
		    return protocolArray[i+1+data_no].data;
		}
	    }
	}
    }
    return (uint16_t) 0;
}
