/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA
**********************************************************************************************/
#include "CToolViewMap.h"
#include "CGarminDBMap.h"
#include "CCentralResources.h"
#include "icons.h"

#include <QtGui>

CToolViewMap::CToolViewMap(QWidget * parent, CGarminDBMap& db)
    : QWidget(parent)
    , db(db)
{
    setupUi(this);
    setObjectName("Maps");

    comboSelectMap->addItem(tr("--- choose map ---"));
    comboSelectOverlay->addItem(tr("--- choose map ---"));

    comboSelectMap->setFocusPolicy(Qt::NoFocus);
    comboSelectOverlay->setFocusPolicy(Qt::NoFocus);

    connect(comboSelectMap,SIGNAL(currentIndexChanged(const QString&)),&db,SLOT(slotChangeMap(const QString&)));
    connect(comboSelectOverlay,SIGNAL(currentIndexChanged(const QString&)),&db,SLOT(slotChangeOverlay(const QString&)));
    connect(listSelectedMaps,SIGNAL(customContextMenuRequested(const QPoint&)),this,SLOT(slotContextMenu(const QPoint&)));

    connect(&db,SIGNAL(sigRegisterMap(const QString&, bool)),this,SLOT(slotRegisterMap(const QString&, bool)));

    connect(butSave,SIGNAL(clicked()),&db,SLOT(slotSaveMapSet()));

    contextMenu = new QMenu(this);
    contextMenu->addAction(QPixmap(iconTrash16x16),tr("Delete"),this,SLOT(slotDelete()));

}

CToolViewMap::~CToolViewMap()
{
    QSettings cfg;
    cfg.setValue("maps/current", comboSelectMap->itemText(comboSelectMap->currentIndex()));
    cfg.setValue("maps/overlay", comboSelectOverlay->itemText(comboSelectOverlay->currentIndex()));
}

void CToolViewMap::updateMapList()
{
    quint32 total = 0;
    QString size;
    listSelectedMaps->clear();


    QMap<QString,CGarminDBMap::map_t>::const_iterator map = db.begin();

    while(map != db.end()){

        const QList<CGarminDBMap::tile_t*>& tiles = map->selected;
        QList<CGarminDBMap::tile_t*>::const_iterator tile = tiles.begin();

        while(tile != tiles.end()){
            QTreeWidgetItem * entry = new QTreeWidgetItem(listSelectedMaps);

            total += (*tile)->memSize;
            size.sprintf("%1.2f MB",(*tile)->memSize / (1024.0 * 1024.0));

            entry->setText(0,size);
            entry->setText(1,(*tile)->name);
            entry->setData(0,Qt::UserRole,(*tile)->key);
            entry->setData(1,Qt::UserRole,map.key());

            ++tile;
        }
        ++map;
    }
    size.sprintf("Total: %1.2f MB",total / (1024.0 * 1024.0));
    labelTotalSize->setText(size);
}


void CToolViewMap::keyPressEvent(QKeyEvent * e)
{
    if(e->key() == Qt::Key_Delete){
        QTreeWidgetItem * item;
        const QList<QTreeWidgetItem*>& items = listSelectedMaps->selectedItems();
        foreach(item,items){
            db.deselectTile(item->data(1,Qt::UserRole).toString(),item->data(0,Qt::UserRole).toString());
            delete item;
        }

        updateMapList();
    }
    else{
        QWidget::keyPressEvent(e);
    }
}

void CToolViewMap::slotContextMenu(const QPoint& pos)
{
    if(!listSelectedMaps->selectedItems().isEmpty()){
        QPoint p = listSelectedMaps->mapToGlobal(pos);
        p.setY(p.y() + listSelectedMaps->header()->height());
        contextMenu->exec(p);
    }
}

void CToolViewMap::slotDelete()
{
    QTreeWidgetItem * item;
    const QList<QTreeWidgetItem*>& items = listSelectedMaps->selectedItems();
    foreach(item,items){
        db.deselectTile(item->data(1,Qt::UserRole).toString(),item->data(0,Qt::UserRole).toString());
        delete item;
    }
    updateMapList();
}

void CToolViewMap::slotRegisterMap(const QString& key, bool transparent)
{
    comboSelectMap->addItem(key);
    if(transparent) comboSelectOverlay->addItem(key);
}


