/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de,
    Michael Ritzert michael.ritzert@googlemail.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/

#include "CMainWindow.h"
#include "icons.h"
#include "CCentralResources.h"
#include "CCanvas.h"
#include "CFunctionStateMachine.h"
#include "CCopyrightMaps.h"
#include "CCopyright.h"
#include "CDlgConfig.h"
#include "CDlgCustomIcons.h"
#include "CDlgScreenshot.h"
#include "CToolViewGoogle.h"
#include "CGarminDBMap.h"
#include "CGarminDBWpt.h"
#include "CGarminDBTrack.h"
#include "CGarminDBRoute.h"
#include "CGpx.h"


#include <QtGui>

CMainWindow::CMainWindow()
{
    setObjectName("MainWidget");
    setWindowTitle("QLandkarte");
    setWindowIcon(QIcon(iconZoomWorld16x16));
    statusBar();

    createActions();
    createMenus();

    mainSplitter = new QSplitter(Qt::Horizontal,this);

    leftSplitter = new QSplitter(Qt::Vertical,this);
    mainSplitter->addWidget(leftSplitter);

    rightSplitter = new QSplitter(Qt::Vertical,this);
    mainSplitter->addWidget(rightSplitter);

    tabWidget = new QTabWidget(this);

    setCentralWidget(mainSplitter);

    resource = new CCentralResources(*this);

    CToolViewGoogle * google = new CToolViewGoogle(0);
    google->show();
    tabWidget->insertTab(0,google,tr("Search"));

    leftSplitter->addWidget(tabWidget);

    QSettings cfg;
    pathMaps = cfg.value("path/maps","./").toString();
    pathData = cfg.value("path/data","./").toString();

    QList<int> sizes = mainSplitter->sizes();
    sizes[0] = (int)(mainSplitter->width() * 0.1);
    sizes[1] = (int)(mainSplitter->width() * 0.9);
    mainSplitter->setSizes(sizes);
    sizes = leftSplitter->sizes();
    sizes[0] = (int)(mainSplitter->height() * 0.5);
    sizes[1] = (int)(mainSplitter->height() * 0.5);
    leftSplitter->setSizes(sizes);
    //mainSplitter->restoreState(cfg.value("mainWidget/mainSplitter",mainSplitter->saveState()).toByteArray());
    //leftSplitter->restoreState(cfg.value("mainWidget/leftSplitter",leftSplitter->saveState()).toByteArray());

    if( cfg.contains("mainWidget/mainSplitter") ) {
        mainSplitter->restoreState(cfg.value("mainWidget/mainSplitter",mainSplitter->saveState()).toByteArray());
    }
    if( cfg.contains("mainWidget/leftSplitter") ) {
        leftSplitter->restoreState(cfg.value("mainWidget/leftSplitter",leftSplitter->saveState()).toByteArray());
    }

    sizes.clear();
    sizes << 200 << 50 << 50;
    rightSplitter->setSizes(sizes);

    tabWidget->setCurrentIndex(0);
    showMaximized();

    connect(tabWidget,SIGNAL(currentChanged(int)),this,SLOT(slotTabChanged(int)));

}

CMainWindow::~CMainWindow()
{
    QSettings cfg;
    cfg.setValue("path/maps",pathMaps);
    cfg.setValue("path/data",pathData);
    cfg.setValue("mainWidget/mainSplitter",mainSplitter->saveState());
    cfg.setValue("mainWidget/leftSplitter",leftSplitter->saveState());
}

void CMainWindow::createActions()
{
    actLoadMap = new QAction(tr("Load Map ..."),this);
    actLoadMap->setStatusTip(tr("Load *.img files"));
    actLoadMap->setIcon(QIcon(iconFileOpen16x16));
    connect(actLoadMap, SIGNAL(triggered()), this, SLOT(slotLoadMap()));

    actImportMap = new QAction(tr("Import Map ..."),this);
    actImportMap->setStatusTip(tr("A map has to be imported, first, before it can be loaded."));
    actImportMap->setIcon(QIcon(iconImportMap16x16));
    connect(actImportMap, SIGNAL(triggered()), this, SLOT(slotImportMap()));

    actConfiguration = new QAction(tr("Configuration ..."),this);
    actConfiguration->setStatusTip(tr("Make your personal settings."));
    actConfiguration->setIcon(QIcon(iconConfiguration16x16));
    connect(actConfiguration, SIGNAL(triggered()), this, SLOT(slotConfiguration()));

    actCopyright = new QAction(tr("QLandkarte"),this);
    actCopyright->setIcon(QIcon(iconZoomWorld16x16));
    connect(actCopyright, SIGNAL(triggered()), this, SLOT(slotCopyright()));

    actCopyrightMaps = new QAction(tr("Maps"),this);
    actCopyrightMaps->setIcon(QIcon(iconZoomWorld16x16));
    connect(actCopyrightMaps, SIGNAL(triggered()), this, SLOT(slotCopyrightMaps()));

    actCopyrightDriver = new QAction(tr("Driver"),this);
    actCopyrightDriver->setIcon(QIcon(iconDriver16x16));
    connect(actCopyrightDriver, SIGNAL(triggered()), this, SLOT(slotCopyrightDriver()));

    actConfig = new QAction(tr("Config"),this);
    actConfig->setIcon(QIcon(":/icons/iconConfigure16x16.png"));
    connect(actConfig, SIGNAL(triggered()), this, SLOT(slotConfig()));

    actLoadGpx = new QAction(tr("&Load *.gpx"),this);
    actLoadGpx->setShortcut(Qt::CTRL + Qt::Key_L);
    actLoadGpx->setIcon(QIcon(iconFileOpen16x16));
    connect(actLoadGpx, SIGNAL(triggered()), this, SLOT(slotLoadGpx()));

    actSaveGpx = new QAction(tr("&Save *.gpx"),this);
    actSaveGpx->setShortcut(Qt::CTRL + Qt::Key_S);
    actSaveGpx->setIcon(QIcon(iconFileSave16x16));
    connect(actSaveGpx, SIGNAL(triggered()), this, SLOT(slotSaveGpx()));

    actPrint = new QAction(tr("&Print"),this);
    actPrint->setShortcut(Qt::CTRL + Qt::Key_P);
    actPrint->setIcon(QIcon(iconPrint16x16));
    connect(actPrint, SIGNAL(triggered()), this, SLOT(slotPrint()));

    actCustomIcons  = new QAction(tr("Custom Icons"),this);
    actCustomIcons->setIcon(QIcon(":/icons/iconCustomIcon16x16.png"));
    actCustomIcons->setStatusTip(tr("Define custom icons and mappings."));
    connect(actCustomIcons, SIGNAL(triggered()), this, SLOT(slotCustomIcons()));

    actScreenshot  = new QAction(tr("Screenshot"),this);
    actScreenshot->setIcon(QIcon(":/icons/iconScreenshot16x16.png"));
    actScreenshot->setStatusTip(tr("Take a screenshot from the device."));
    connect(actScreenshot, SIGNAL(triggered()), this, SLOT(slotScreenshot()));

    actExit  = new QAction(tr("Exit"),this);
    actExit->setIcon(QIcon(":/icons/iconExit16x16.png"));
    connect(actExit, SIGNAL(triggered()), this, SLOT(close()));

}

void CMainWindow::createMenus()
{
    QMenu * p;

    fileMenu = new QMenu(this);
    fileMenu->setTitle(tr("&File"));
    fileMenu->addAction(actLoadMap);
    fileMenu->addAction(actImportMap);
    fileMenu->addSeparator();
    fileMenu->addAction(actLoadGpx);
    fileMenu->addAction(actSaveGpx);
    fileMenu->addSeparator();
    fileMenu->addAction(actPrint);
    fileMenu->addAction(actScreenshot);
    fileMenu->addSeparator();
    fileMenu->addAction(actExit);
    menuBar()->addMenu(fileMenu);

    p = new QMenu(this);
    p->setTitle(tr("&Setup"));
    p->addAction(actConfig);
    p->addAction(actCustomIcons);
    menuBar()->addMenu(p);


    p = new QMenu(this);
    p->setTitle(tr("&Copyright"));
    p->addAction(actCopyrightMaps);
    p->addAction(actCopyrightDriver);
    p->addAction(actCopyright);
    menuBar()->addMenu(p);


}


void CMainWindow::slotLoadMap()
{

    QString filename = QFileDialog::getOpenFileName( 0, tr("Select *.img file")
                                                    ,pathMaps
                                                    ,"Map File (*.img)"
                                                );
    if(filename.isEmpty()) return;

    pathMaps = QFileInfo(filename).absolutePath();

    resource->mapdb().loadIMG(filename);
}


void CMainWindow::slotImportMap()
{
    QString path = QFileDialog::getExistingDirectory(this,tr("Map Source Path..."));
    if(path.isEmpty()) return;

    resource->mapdb().loadTDB(path);
}

void CMainWindow::slotConfiguration()
{

}

void CMainWindow::keyPressEvent(QKeyEvent * e)
{

    if((e->key() >= Qt::Key_F1) && (e->key() < Qt::Key_F11)){
        return resource->keyPressEvent(e);
    }
    else if(e->key() == Qt::Key_Escape) {
        return resource->keyPressEvent(e);
    }
    else if((e->key() == Qt::Key_Plus) || (e->key() == Qt::Key_Minus)) {
        return resource->keyPressEvent(e);
    }
    else if(e->modifiers() == Qt::AltModifier){
        if((e->key() == Qt::Key_Up) || (e->key() == Qt::Key_Down)
            || (e->key() == Qt::Key_Left) || (e->key() == Qt::Key_Right)) {
            return resource->keyPressEvent(e);
        }
    }
    return e->ignore();
}

void CMainWindow::slotCopyright()
{
    CCopyright dlg;
    dlg.exec();
}

void CMainWindow::slotCopyrightMaps()
{
    CCopyrightMaps dlg;
    dlg.exec();
}

void CMainWindow::slotCopyrightDriver()
{
	resource->resetDevice();	
    Garmin::IDevice * dev = gpResources->device();
    if(dev){
        QMessageBox::information(0,tr("Driver ..."),dev->getCopyright().c_str(),QMessageBox::Ok,QMessageBox::NoButton);
    }
}


void CMainWindow::slotConfig()
{
    CDlgConfig dlg;
    dlg.exec();
    gpResources->canvas().update();
}

void CMainWindow::slotTabChanged(int index)
{
    QString key = tabWidget->widget(index)->objectName();
    gpResources->fsm().switchByKeyWord(key);
}

void CMainWindow::slotLoadGpx()
{
    QString filename = QFileDialog::getOpenFileName( 0, tr("Select *.gpx file")
                                                    ,pathData
                                                    ,"GPS Exchange (*.gpx)"
                                                );
    if(filename.isEmpty()) return;

    pathData = QFileInfo(filename).absolutePath();

    CGpx gpx(0);
    gpx.load(filename);

    gpResources->mapdb().loadGPX(gpx);
    gpResources->wptdb().loadGPX(gpx);
    gpResources->trackdb().loadGPX(gpx);
    gpResources->routedb().loadGPX(gpx);

}

void CMainWindow::slotSaveGpx()
{
    QString filename = QFileDialog::getSaveFileName( 0, tr("Select *.gpx file")
                                                    ,pathData
                                                    ,"GPS Exchange (*.gpx)"
                                                );
    if(filename.isEmpty()) return;

    if(QFileInfo(filename).suffix() != "gpx"){
        filename += ".gpx";
    }

    pathData = QFileInfo(filename).absolutePath();

    CGpx gpx(0);

    gpResources->mapdb().saveGPX(gpx);
    gpResources->wptdb().saveGPX(gpx);
    gpResources->trackdb().saveGPX(gpx);
    gpResources->routedb().saveGPX(gpx);

    gpx.save(filename);
}

void CMainWindow::slotPrint()
{
    QPrinter printer;
    QPrintDialog *dialog = new QPrintDialog(&printer, this);
    dialog->setWindowTitle(tr("Print Map"));
    if (dialog->exec() != QDialog::Accepted)
        return;

    gpResources->canvas().print(printer);
}

void CMainWindow::slotCustomIcons()
{

    CDlgCustomIcons dlg;
    dlg.exec();
}


void CMainWindow::slotScreenshot()
{
    CDlgScreenshot dlg;
    dlg.exec();
}
