/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/

#include "CGarminLiveLog.h"
#include "CCentralResources.h"
#include "CToolViewLog.h"
#include "icons.h"

#include <limits>
#include <math.h>
#include <QtGui>

#define SAMPLETIME  500


float CGarminLiveLog::Pvt_t::velocity() const
{
   // may add vertical speed here (up)
   return sqrtf( north*north + east*east );
}


float CGarminLiveLog::Pvt_t::heading() const
{
    // multiply by 100 to avoid leaving the float range.
    float heading = fabsf((100.0*east) / (100.0*north));
    heading = atanf(heading) / (2.0 * M_PI) * 360.0;
    if( (north > 0.0) & (east > 0.0) ){
        // 1st quadrant
        heading = heading;
    }
    else if( (north > 0.0) & (east < 0.0) ){
        // 2nd quadrant
        heading = 360.0-heading;
    }
    else if( (north < 0.0) & (east < 0.0) ){
        // 3rd quadrant
        heading = 180.0+heading;
    }
    else if( (north < 0.0) & (east > 0.0) ){
        // 4th quadrant
       heading = 180.0-heading;
    }
    else{
       heading = std::numeric_limits<float>::quiet_NaN();
    }
    return heading;
}


QDateTime CGarminLiveLog::Pvt_t::localTime() const
{
   QDateTime t(QDate(1989,12,31), QTime(0,0), Qt::UTC);
   t = t.addDays(wn_days).addSecs(lrint(tow)).addSecs(-leap_scnds);
   return t.toLocalTime();
}


CGarminLiveLog::CGarminLiveLog(QTabWidget * parent)
    : QObject(parent)
    , tab(parent)
    , oldLon(-1000)
    , oldLat(-1000)
{
    timer = new QTimer(this);
    connect(timer,SIGNAL(timeout()),this,SLOT(slotGetPosition()));

    toolview = new CToolViewLog(0,this);
    toolview->show();
    toolview->setEnabled(false);
    parent->addTab(toolview,QPixmap(iconLiveLog16x16),"");
    parent->setTabToolTip(parent->indexOf(toolview), tr("Live Log View"));

    QSettings cfg;
    pvt.lockToCenter = cfg.value("livelog/lockToCenter",false).toBool();
}

CGarminLiveLog::~CGarminLiveLog()
{
    QSettings cfg;
    cfg.setValue("livelog/lockToCenter",pvt.lockToCenter);
}

void CGarminLiveLog::gainFocus()
{
    if(tab){
        if(tab->currentWidget() != toolview){
            tab->setCurrentWidget(toolview);
        }
    }
}

CGarminLiveLog::Pvt_t * CGarminLiveLog::getPosition()
{
    return timer->isActive() ? &pvt : 0;
}

void CGarminLiveLog::toggleLockToCenter()
{
    pvt.lockToCenter = !pvt.lockToCenter;

    if(pvt.lockToCenter){
        gpResources->canvas().move(pvt.lon,pvt.lat,QString::Null());
    }
    else{
        gpResources->canvas().update();
    }

    emit sigPosition(pvt);
}

void CGarminLiveLog::toggle()
{
    if(timer->isActive()){
        Garmin::IDevice * dev = 0;
        try{
            dev = gpResources->device();
            if(dev){
                dev->setRealTimeMode(false);
            }
        }
        catch(int e){
            if(dev == 0) return;
            QMessageBox::warning(0,tr("Device Link Error"),dev->getLastError().c_str(),QMessageBox::Ok,QMessageBox::NoButton);
            if(e == Garmin::errSync){
                gpResources->resetDevice();
            }
        }
        CGarminTrack * track = gpResources->trackdb().getTrack("log|Active");
        track->update();
        timer->stop();
        toolview->setEnabled(false);
    }
    else{
        Garmin::IDevice * dev = 0;
        try{
            dev = gpResources->device();
            if(dev){
                dev->setRealTimeMode(true);
            }
            timer->start(SAMPLETIME);
            toolview->setEnabled(true);
        }
        catch(int e){
            if(dev == 0) return;
            QMessageBox::warning(0,tr("Device Link Error"),dev->getLastError().c_str(),QMessageBox::Ok,QMessageBox::NoButton);
            if(e == Garmin::errSync){
                gpResources->resetDevice();
            }
        }
    }
}

void CGarminLiveLog::slotGetPosition()
{
    //qDebug() << "slotGetPosition()";
    Garmin::IDevice * dev = 0;
    try{
        dev = gpResources->device();
        if(dev){
            dev->getRealTimePos(pvt);
            if(oldLon == -1000){
                oldLon = pvt.lon;
                oldLat = pvt.lat;
            }

            if(oldLon != pvt.lon || oldLat != pvt.lat){
                if(pvt.lockToCenter){
                    gpResources->canvas().move(pvt.lon,pvt.lat,QString::Null());
                }
                else{
                    gpResources->canvas().update();
                }

                CGarminTrack * track = gpResources->trackdb().getTrack("log|Active");
                CGarminTrack::TrkPt_t pt;
                pt.lon = pvt.lon;
                pt.lat = pvt.lat;
                pt.alt = pvt.alt + pvt.msl_hght;

                track->addTrkPt(pt);
                emit sigPosition(pvt);
            }

            oldLon = pvt.lon;
            oldLat = pvt.lat;
        }
    }
    catch(int e){
        if(dev == 0) return;
        QMessageBox::warning(0,tr("Device Link Error"),dev->getLastError().c_str(),QMessageBox::Ok,QMessageBox::NoButton);
        if(e == Garmin::errSync){
            gpResources->resetDevice();
        }
        else{
            timer->stop();
            toolview->setEnabled(false);
        }
    }
}
