/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de, Michael Ritzert
    michael.ritzert@googlemail.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/
#ifndef CGARMINWPTDB_H
#define CGARMINWPTDB_H

#include <QObject>
#include <QWidget>
#include <QMap>
#include <QPointer>

#include <limits>

#include <IDevice.h>

#include "IProjection.h"

class QTabWidget;
class QString;
class CToolViewWpt;
class CGpx;
class QHttp;
class QProgressDialog;
class QHttpResponseHeader;
class CGarminWpt;

class CGarminDBWpt : public QObject
{
    Q_OBJECT
    public:
        CGarminDBWpt(QTabWidget * parent);
        virtual ~CGarminDBWpt();

        void gainFocus();

        /// receive waypoints from device
        void downloadWaypoints();
        /// send waypoints to device
        void uploadWaypoints();
        /// read access to database
        const QMap<QString,CGarminWpt*>& getWaypoints(){return waypoints;}
        /// create a new waypoint
        void newWaypoint(const double lon = std::numeric_limits<double>::quiet_NaN(), const double lat = std::numeric_limits<double>::quiet_NaN(),const float alt = std::numeric_limits<float>::quiet_NaN(),const QString& comment = QString::Null());
        /// edit existing waypoint defined by key
        void editWaypoint(const QString& key);
        /// remove a waypoint by name
        void removeWaypoint(const QString& key);
        /// remove several waypoints by name
        void removeWaypoint(const QStringList& keys);
        /// load waypoints from GPX file
        void loadGPX(CGpx& gpx);
        /// save waypoints to GPX file
        void saveGPX(CGpx& gpx);
        /// remove all waypoints from the list
        void clear();
        /// get a list of geo caching waypoints from www.opencaching.de
        void queryOpenChachingDe(double lon, double lat, double radius);

        /// get iterator access to track point list
        QMap<QString,CGarminWpt*>::iterator begin(){return waypoints.begin();}
        /// get iterator access to track point list
        QMap<QString,CGarminWpt*>::iterator end(){return waypoints.end();}

        void removeWaypointBy();

        void setProxyDistance(const QStringList& keys, double dist);

    signals:
        void sigWaypointListChanged();

    protected slots:
        void slotRequestStarted(int );
        void slotRequestFinished(int , bool error);
        void slotDataReadProgress(int done, int total);
        void slotSelectWpt(CGarminWpt * wpt);
    protected:
        friend class CDlgCustomIcons;
        QPointer<QTabWidget> tab;

        /// tool view widget to display database information
        CToolViewWpt * toolview;

        /// the waypoint database
        QMap<QString,CGarminWpt*> waypoints;

        /// request protocol handler for waypoint queries
        QPointer<QHttp> opencaching;
        /// the progress dialog to inform about the state of the waypoint query
        QPointer<QProgressDialog> progressDlg;
        /// internal flag used by slotRequestFinished()
        bool rcvData;
        /// radius used by geo cache query (used to fine tune results)
        double radius;
        /// center point of geo cache query (used to fine tune results)
        XY center;

        struct type_t{
            type_t() : type("Unknown"), icon("Geocache"){}
            type_t(const QString& type, const QString& icon) : type(type), icon(icon){}
            QString type;
            QString icon;
        };
        QMap<int,type_t> cacheIcons;
};

#endif //CGARMINWPTDB_H

