/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/
#ifndef CGARMINMAPDB_H
#define CGARMINMAPDB_H


#include "CGarminImg.h"

#include <QTabWidget>
#include <QMap>
#include <QList>
#include <QRectF>
#include <QPointer>
#include <QString>
#include <QPointer>



class CToolViewMap;
class CGarminImg;
class QProgressDialog;
class QTabWidget;
class IMap;
class CGpx;

class CGarminDBMap : public QObject
{
    Q_OBJECT
    public:
        CGarminDBMap(QTabWidget * parent);
        virtual ~CGarminDBMap();

        void gainFocus();

        enum maptype_e {eMap, eOvl};

        struct tile_t{
            quint32 id;
            QString key;
            QString name;
            std::string cname;
            QString file;
            double north;
            double east;
            double south;
            double west;
            QRectF area;
            QVector<XY> definitionArea;

            QPointer<CGarminImg> img;

            quint32 memSize;
        };

        struct map_level_t{
            quint8 bits;
            quint8 level;
            bool   useBaseMap;
        };

        struct map_t{
            map_t() : north(-90.0), east(-180.0), south(90.0), west(180.0), useBaseMap(true), isTransparent(false) {}
            /// the map name
            QString name;
            /// the base map filename
            QString basemap;
            /// north boundary of basemap
            double north;
            /// east boundary of basemap
            double east;
            /// south boundary of basemap
            double south;
            /// west boundary of basemap
            double west;
            /// the area in cartesian coordinates [m] covered by the basemap
            QRectF area;

            /// the basemap img database
            QPointer<CGarminImg> img;
            /// the basemap as list to be compatible to the selcted tiles list
            QList<CGarminImg*> base;
            /// a list of visible map tiles
            QList<CGarminImg*> visible;
            /// a list of selected tiles
            QList<tile_t*> selected;

            /// set true to use base for rendering. Else use visible list
            bool useBaseMap;

            /// all tiles by name
            QMap<QString,tile_t> tiles;

            /// combined maplevels of basemap & submap tiles
            QVector<map_level_t> maplevels;

            /// unlock key for locked maps
            QString key;

            ///global transparent flag. Set true if test tile during registration is transparent.
            bool isTransparent;
        };


        /// load a single map tile
        void loadIMG(const QString& path);
        /// load / register a map
        void loadTDB(const QString& path);
        /// load *.img files by resolution bits and visible area
        quint8 selectVisibleTiles(maptype_e type, quint8 bits, const QRectF& area);
        /// get a list of currently visible map tiles
        const QList<CGarminImg*>& getVisibleTiles(maptype_e type);
        /// release all loaded map tiles and switch back to basemap
        void releaseVisibleTiles();
        /// toggle selection state of tiles under area
        void selectTiles(const QVector<XY>& area);
        /// clear selection state of tile by key
        void deselectTile(const QString& mapKey, const QString& tileKey);

        /// get a list of selected map tile definitions
        const QList<tile_t*>& getSelectedTiles(){return activeMap->selected;}
        /// send selected map tiles to device
        void uploadSelectedTiles(bool save);
        /// query info about loaded maps
        void downloadMapInfo();
        /// hm, cheesy. Return area covered by current map
        const QRectF& getWorldRect(){return activeMap->area;}
        /// get copyright strings from all maps
        const QString& getCopyrights(){return copyright;}
        /// get info about elements close to point
        /**
            The parameter pt will be moved to the current captured line.

            @param pt the point in units of meters
            @param dict a multimap object, keys will be the found element type strings.
            @param theMap pointer to the current map render object
        */
        void getInfo(QPointF& pt, QMultiMap<QString,QString>& dict, IMap * theMap, bool streetOnly=false);
        /// load selected maps from GPX file
        void loadGPX(CGpx& );
        /// save selected maps to GPX file
        void saveGPX(CGpx& );
        /// remove all selected maps
        void clear();

        /// get iterator access to track point list
        QMap<QString,map_t>::iterator begin(){return maps.begin();}
        /// get iterator access to track point list
        QMap<QString,map_t>::iterator end(){return maps.end();}

    signals:
        void sigMapChanged(QRectF& area);
        void sigMapChanged();
        void sigRegisterMap(const QString& key, bool transparent);

    protected slots:
        void slotChangeMap(const QString& key);
        void slotChangeOverlay(const QString& key);
        void slotSaveMapSet();

    protected:
        friend void progress(quint32 size, quint32 total, void * self);

        void decodeTDB(const QString& filepath);

        void getInfoPoints(const subdiv_desc_t& subdiv, QPointF& pt, QMultiMap<QString,QString>& dict, IMap * theMap);
        void getInfoPOIs(const subdiv_desc_t& subdiv, QPointF& pt, QMultiMap<QString,QString>& dict, IMap * theMap);
        void getInfoPolylines(const subdiv_desc_t& subdiv, QPointF& pt, QMultiMap<QString,QString>& dict, IMap * theMap);
        void getInfoPolygons(const subdiv_desc_t& subdiv, QPointF& pt, QMultiMap<QString,QString>& dict, IMap * theMap);

        /// used by loadTDB() and loadIMG()
        bool registerMap(map_t& map);

        /// copy the definition area polygons from the base map file to the tile definitions
        void copyDefinitionAreaInfo(map_t& baseMap);

        quint8 selectVisibleTiles(map_t& m, quint8 bits, const QRectF& area);

        QPointer<QTabWidget> tab;

        /// all registered maps by name
        QMap<QString,map_t> maps;
        /// the current active map
        map_t * activeMap;
        /// the current active overlay (transparent) map
        map_t * overlayMap;

        /// tool view widget to display database information
        CToolViewMap * toolview;

        /// shown during upload
        QProgressDialog * progressDlg;
        /// to estimate uplaod time
        QTime uploadTime;

        /// list of all copyright informations
        QString copyright;

        quint8 activeMapLevel;

        quint8 activeOvlLevel;
};



#endif //CGARMINMAPDB_H

