/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/

#include "CDlgSetupWpt.h"
#include "CDlgWptIcon.h"
#include "GarminIcons.h"
#include "GeoMath.h"
#include "CCentralResources.h"

#include <QtGui>

CDlgSetupWpt::CDlgSetupWpt(CGarminWpt& wpt, QWidget * parent)
    : QDialog(parent)
    , wpt(wpt)
{
    setupUi(this);
    connect(butIcon,SIGNAL(pressed()),this,SLOT(slotSelectIcon()));
    connect(butLink,SIGNAL(pressed()),this,SLOT(slotEditLink()));
    connect(lblLink,SIGNAL(linkActivated(const QString&)),this,SLOT(slotOpenLink(const QString&)));
}

CDlgSetupWpt::~CDlgSetupWpt()
{
}

void CDlgSetupWpt::slotOpenLink(const QString& link)
{
    gpResources->openLink(link);
}

int CDlgSetupWpt::exec()
{
    QPixmap pix;
    QString name;
    getWptIconById(wpt.smbl,pix,name);
    butIcon->setIcon(pix);
    butIcon->setToolTip(name);

    lineIdent->setText(wpt.ident);
    lineComment->setText(wpt.comment);

    if(wpt.lat == 1000 || wpt.lon == 1000){
        gpResources->canvas().getCursorPos(wpt.lon, wpt.lat);
    }
    qint32 degN,degE;
    double minN,minE;
    GPS_Math_Deg_To_DegMin(wpt.lat, &degN, &minN);
    GPS_Math_Deg_To_DegMin(wpt.lon, &degE, &minE);

    QString str,lat,lng;
    lat = degN < 0 ? "S" : "N";
    lng = degE < 0 ? "W" : "E";
    str.sprintf("%s%02d\260 %06.3f %s%03d\260 %06.3f",lat.toUtf8().data(),abs(degN),minN,lng.toUtf8().data(),abs(degE),minE);

    linePosition->setText(str);

    if(gpResources->doMetric()){
        if(wpt.alt != 1e25f){
            lineAltitude->setText(QString::number(wpt.alt,'f',0));
        }
        else if(wpt.dpth != 1e25f){
            lineAltitude->setText(QString::number(-wpt.dpth,'f',0));
        }

        if(wpt.dist != 1e25f){
            lineDistance->setText(QString::number(wpt.dist,'f',0));
        }
    }
    else{
        lblUnitAltitude->setText("ft");
        lblUnitDistance->setText("ft");
        if(wpt.alt != 1e25f){
            lineAltitude->setText(QString::number(wpt.alt * 3.28084,'f',0));
        }
        else if(wpt.dpth != 1e25f){
            lineAltitude->setText(QString::number(-wpt.dpth  * 3.28084,'f',0));
        }

        if(wpt.dist != 1e25f){
            lineDistance->setText(QString::number(wpt.dist * 3.28084,'f',0));
        }
    }

    link = wpt.link;

    if(!link.isEmpty()){
        QString str;
        str = "<a href='" + link + "'>" + link + "</a>";
        lblLink->setText(str);
    }

    checkSticky->setChecked(wpt.sticky);
    return QDialog::exec();
}

void CDlgSetupWpt::slotSelectIcon()
{
    quint16 smbl = wpt.smbl;
    QPixmap pix;
    QString name;

    CDlgWptIcon dlg(smbl,0);
    dlg.exec();

    getWptIconById(smbl,pix,name);
    butIcon->setIcon(pix);
    butIcon->setToolTip(name);

}

void CDlgSetupWpt::slotEditLink()
{
    bool ok = false;
    QString _link = QInputDialog::getText(0,tr("Edit link ..."),tr("Link: 'http://...'"),QLineEdit::Normal,link,&ok);
    if(ok){
        link = _link;
    }

    if(!link.isEmpty()){
        QString str;
        str = "<a href='" + link + "'>" + link + "</a>";
        lblLink->setText(str);
    }

}

void CDlgSetupWpt::accept()
{
    if(lineIdent->text().isEmpty()){
        QMessageBox::warning(0,tr("Error"),tr("You must provide a waypoint indentifier."),QMessageBox::Ok,QMessageBox::NoButton);
        return;
    }
    if(linePosition->text().isEmpty()){
        QMessageBox::warning(0,tr("Error"),tr("You must provide a waypoint position."),QMessageBox::Ok,QMessageBox::NoButton);
        return;
    }

    wpt.ident   = lineIdent->text();
    wpt.comment = lineComment->text();

    QRegExp re("^\\s*([N|S]){1}\\W*([0-9]+)\\W*([0-9]+\\.[0-9]+)\\W([E|W]){1}\\W*([0-9]+)\\W*([0-9]+\\.[0-9]+)\\s*$");
    if(!re.exactMatch(linePosition->text())){
        QMessageBox::warning(0,tr("Error"),tr("Bad position format. Must be: [N|S] ddd mm.sss [W|E] ddd mm.sss"),QMessageBox::Ok,QMessageBox::NoButton);
        return;
    }

    bool signLat    = re.cap(1) == "S";
    int degLat      = re.cap(2).toInt();
    double minLat   = re.cap(3).toDouble();

    GPS_Math_DegMin_To_Deg((signLat ? -degLat : degLat), minLat, wpt.lat);

    bool signLon    = re.cap(4) == "W";
    int degLon      = re.cap(5).toInt();
    double minLon   = re.cap(6).toDouble();

    GPS_Math_DegMin_To_Deg((signLon ? -degLon : degLon), minLon, wpt.lon);

    getWptIconByName(butIcon->toolTip(),wpt.smbl,wpt.icon);

    if(gpResources->doMetric()){
        float x;
        if(!lineAltitude->text().isEmpty()){
            x = lineAltitude->text().toFloat();
            if(x < 0){
                wpt.alt = 1e25f;
                wpt.dpth = -x;
            }
            else{
                wpt.alt = x;
                wpt.dpth = 1e25f;
            }
        }
        else{
            wpt.alt = 1e25f;
            wpt.dpth = 1e25f;
        }

        if(!lineDistance->text().isEmpty()){
            wpt.dist = lineDistance->text().toFloat();
            wpt.dist = wpt.dist == 0 ? 1e25f : wpt.dist;
        }
        else{
            wpt.dist = 1e25f;
        }

    }
    else{
        float x;
        if(!lineAltitude->text().isEmpty()){
            x = lineAltitude->text().toFloat();
            if(x < 0){
                wpt.alt = 1e25f;
                wpt.dpth = -x * 0.3048;
            }
            else{
                wpt.alt = x * 0.3048;
                wpt.dpth = 1e25f;
            }
        }
        else{
            wpt.alt = 1e25f;
            wpt.dpth = 1e25f;
        }

        if(!lineDistance->text().isEmpty()){
            wpt.dist = lineDistance->text().toFloat() * 0.3048;
        }
        else{
            wpt.dist = 1e25f;
        }

    }

    wpt.link = link;

    wpt.sticky = checkSticky->isChecked();
    QDialog::accept();
}
