import canvas
import font
import pychart_util
import chart_object
import line_style
import types
import math
import theme

class Base(chart_object.T):
   keys = {
      "tic_interval" : (pychart_util.IntervalType, 1, None, 
			pychart_util.interval_desc("tick marks")),
      "tic_len" : (pychart_util.NumType, 0, 6, "The length of tick lines"),
      "first_tic_value": (pychart_util.NumType, 1, None, "The location of the first tick mark. Defaults to the x_range[0] (or y_range[0]) of the area."),
      "grid_interval" : (pychart_util.IntervalType, 1, None,
                         pychart_util.interval_desc("grid lines")),
      "grid_style" : (line_style.T, 1, None, "The style of grid lines."),
      "minor_tic_interval" : (pychart_util.IntervalType, 1, None,
                              pychart_util.interval_desc("minor tick marks")),
      "minor_tic_len" : (pychart_util.NumType, 0, 3, "The length of minor tick marks."),
      "line_style": (line_style.T, 1, line_style.default, 
                     "The style of tick lines."),
      "label": (types.StringType, 1, "axis label",
		"The description of the axis. <<font>>."),
      "format": (pychart_util.FormatType, 0, "%s", 
                 """The format string for tick labels.
                 It can be a `printf' style format string, or 
                 a single-parameter function that returns a string.
                 <<font>>.
                 """),
      "label_offset": (pychart_util.CoordOrNoneType, 0, (None,None),
                       "The location where the axis label is drawn. "
                       "Relative to the left-bottom corner of the axis."),
      "tic_label_offset": (pychart_util.CoordType, 0, (0,0),
                           "The location where the tick labels is drawn. "
                           "Relative to the tip of the tick mark.")
      }

def log_tics(min, max, tic_interval):
   "Generate the list of places for drawing tick marks."
   v = []
   x = min
   while x <= max:
      v.append(x)
      x = x * tic_interval
   return v

def linear_tics(min, max, tic_interval):
   "Generate the list of places for drawing tick marks."   
   v = []
   x = min
   while x <= max:
      v.append(x)
      x = x + tic_interval
   return v

def pick_tick_interval(axis_len, p):
   range = p[1]-p[0]
   interval = 10 ** int(math.log(range)/math.log(10))
   while axis_len * interval / range >= 30:
      interval = float(interval / 10.0)
   if axis_len * interval / range < 10:
      interval = interval * 5
   return interval

class X(Base):
   def tics(self, ar, interval):
      "Return the list of X values for which tick marks and grid lines are drawn."
      minx = ar.x_range[0]
      if self.first_tic_value != None:
         minx = self.first_tic_value
         
      if type(interval) == types.FunctionType:
	 return apply(interval, (minx, ar.x_range[1]))
      elif ar.x_coord_system == 'linear':
         interval = interval or pick_tick_interval(ar.size[0], ar.x_range)
         return linear_tics(minx, ar.x_range[1], interval)
      elif ar.x_coord_system == 'log':
         interval = interval or 10
         return log_tics(minx, ar.x_range[1], interval)
      else:
         return map(lambda pair, col=ar.x_category_col: pair[col],
                    ar.x_category_data)
                    
   def draw(self, ar):
      self.type_check()
      canvas.line(self.line_style, ar.loc[0], ar.loc[1],
              ar.loc[0]+ ar.size[0], ar.loc[1])

      tic_dic = {}
      max_tic_height = 0
      grid_interval = self.grid_interval or self.tic_interval
      
      for i in self.tics(ar, self.tic_interval):
         tic_dic[i] = 1
         ticx = ar.x_pos(i)

	 str = "/hC" + pychart_util.apply_format(self.format, (i,), 0)

         (total_height, base_height) = font.text_height(str)
         max_tic_height = max(max_tic_height, total_height)
         
         canvas.line(self.line_style, ticx, ar.loc[1], ticx, ar.loc[1]-self.tic_len)
         canvas.show(ticx+self.tic_label_offset[0], 
                     ar.loc[1]-self.tic_len-base_height+self.tic_label_offset[1],
                     str)
         
      if self.minor_tic_interval:
         for i in self.tics(ar, self.minor_tic_interval):
            if tic_dic.has_key(i):
               # a major tic was drawn already.
               pass
            else:
               ticx = ar.x_pos(i)
               canvas.line(self.line_style, ticx, ar.loc[1], ticx,
                           ar.loc[1]-self.minor_tic_len)
      if self.grid_style:
         for i in self.tics(ar, grid_interval):
            x = ar.x_pos(i)
            if x > ar.loc[0]:
               canvas.line(self.grid_style,
                           x, ar.loc[1], x, ar.loc[1]+ar.size[1])

      if self.label != None:
         str = "/hC/vM" + self.label
         (label_height, base_height) = font.text_height(str)
         if self.label_offset[0] != None:
            xlabel = ar.loc[0] + self.label_offset[0]
         else:
            xlabel = ar.loc[0] + ar.size[0]/2.0
         if self.label_offset[1] != None:
            ylabel = ar.loc[1] + self.label_offset[1]
         else:
            ylabel = ar.loc[1] - self.tic_len - max_tic_height - 10
         canvas.show(xlabel, ylabel, str)



class Y(Base):
   keys = pychart_util.union_dict(Base.keys,
                                  { "grid_style" : (line_style.T, 1, line_style.gray7dash3)
                          })
   def tics(self, ar, interval):
      miny = ar.y_range[0]
      
      if self.first_tic_value != None:
         miny = self.first_tic_value
      if type(interval) == types.FunctionType:
	 return apply(interval, (miny, ar.y_range[1]))
      elif ar.y_coord_system == 'linear':
         interval = interval or pick_tick_interval(ar.size[1], ar.y_range)
         return linear_tics(miny, ar.y_range[1], interval)
      elif ar.y_coord_system == 'log':
         interval = interval or 10
         return log_tics(miny, ar.y_range[1], interval)
      else:
         return map(lambda pair, col=ar.y_category_col: pair[col],
                    ar.y_category_data)
                    
   def draw(self, ar):
      self.type_check()
      grid_interval = self.grid_interval or self.tic_interval      

      if ar.y_zap != None and not ar.y_zap_entire_area:
	 canvas.beginZigZagClip(ar.loc[0], ar.y_pos((ar.y_zap[0]+ar.y_zap[1])/2.0),10)

      canvas.line(self.line_style, ar.loc[0], ar.loc[1],
                  ar.loc[0], ar.loc[1]+ar.size[1])
      
      xmin = ar.loc[0] + ar.size[0] # somebigvalue
      tic_dic = {}
      for i in self.tics(ar, self.tic_interval):
	 if ar.y_zap and i >= ar.y_zap[0] and i < ar.y_zap[1]:
		continue
         y_tic = ar.y_pos(i)
         tic_dic[i] = 1
         canvas.line(self.line_style, ar.loc[0], y_tic,
                     ar.loc[0] - self.tic_len, y_tic)
         x = ar.loc[0] - self.tic_len + self.tic_label_offset[0]
         tic_label = "/hR" + pychart_util.apply_format(self.format, (i,), 0)
         tic_height, base_height = font.text_height(tic_label)
         canvas.show(x, y_tic - tic_height/2.0 + self.tic_label_offset[1],
		     tic_label)
         xmin = min(xmin, x - font.text_width(tic_label))
      if self.minor_tic_interval:
         for i in self.tics(ar, self.minor_tic_interval):
            if tic_dic.has_key(i):
               # a major tic was drawn already.
               pass
            else:
               y_tic = ar.y_pos(i)
               canvas.line(self.line_style, ar.loc[0], y_tic,
                       ar.loc[0] - self.minor_tic_len, y_tic)
      if self.grid_style:
         for i in self.tics(ar, grid_interval):
            y = ar.y_pos(i)
            if y > ar.loc[1]:
               canvas.line(self.grid_style,
                       ar.loc[0], y,
                       ar.loc[0]+ar.size[0], y)
         
      if self.label != None:
         if self.label_offset[0] != None:
            xlabel = ar.loc[0] + self.label_offset[0]
         else:
            xlabel = xmin - theme.default_font_size/2.0
            
         if self.label_offset[1] != None:
            ylabel = ar.loc[1] + ar.size[1]/2 + self.label_offset[1]
         else:
            ylabel = ar.loc[1] + ar.size[1] / 2
         canvas.show(xlabel, ylabel, "/a90/hC" + self.label)
