/*****************************************************************************/
/*                                                                           */
/*  Copyright (c) 2001, Peter Shannon                                        */
/*  All rights reserved.                                                     */
/*                                                                           */
/*  Redistribution and use in source and binary forms, with or without       */
/*  modification, are permitted provided that the following conditions       */
/*  are met:                                                                 */
/*                                                                           */
/*      * Redistributions of source code must retain the above               */
/*        copyright notice, this list of conditions and the following        */
/*        disclaimer.                                                        */
/*                                                                           */
/*      * Redistributions in binary form must reproduce the above            */
/*        copyright notice, this list of conditions and the following        */
/*        disclaimer in the documentation and/or other materials             */
/*        provided with the distribution.                                    */
/*                                                                           */
/*      * The name of the contributors may be used to endorse or promote     */
/*        products derived from this software without specific prior         */
/*        written permission.                                                */
/*                                                                           */
/*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS      */
/*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT        */
/*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS        */
/*  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS   */
/*  OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,          */
/*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT         */
/*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,    */
/*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY    */
/*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT      */
/*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE    */
/*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.     */
/*                                                                           */
/*****************************************************************************/

#include <python2.1/Python.h>

#include <openssl/crypto.h>
#include <openssl/rand.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/evp.h>
#include <openssl/err.h>
#include <openssl/md5.h>
#include <openssl/md2.h>
#include <openssl/sha.h>
#include <openssl/hmac.h>

#include <time.h>
#include <pthread.h>

// semetric ciphers
#define DES_ECB               1
#define DES_EDE               2
#define DES_EDE3              3
#define DES_CFB               4
#define DES_EDE_CFB           5
#define DES_EDE3_CFB          6
#define DES_OFB               7
#define DES_EDE_OFB           8
#define DES_EDE3_OFB          9
#define DES_CBC               10
#define DES_EDE_CBC           11
#define DES_EDE3_CBC          12
#define DESX_CBC              13
#define RC4                   14
#define RC4_40                15
#define IDEA_ECB              16
#define IDEA_CFB              17
#define IDEA_OFB              18
#define IDEA_CBC              19
#define RC2_ECB               20
#define RC2_CBC               21
#define RC2_40_CBC            22
#define RC2_CFB               23
#define RC2_OFB               24
#define BF_ECB                25
#define BF_CBC                26
#define BF_CFB                27
#define BF_OFB                28
#define CAST5_ECB             29
#define CAST5_CBC             30
#define CAST5_CFB             31
#define CAST5_OFB             32
#define RC5_32_12_16_CBC      33
#define RC5_32_12_16_CFB      34
#define RC5_32_12_16_ECB      35
#define RC5_32_12_16_OFB      36

// SSL connection methods
#define SSLV2_SERVER_METHOD   1
#define SSLV2_CLIENT_METHOD   2
#define SSLV2_METHOD          3
#define SSLV3_SERVER_METHOD   4
#define SSLV3_CLIENT_METHOD   5
#define SSLV3_METHOD          6
#define TLSV1_SERVER_METHOD   7
#define TLSV1_CLIENT_METHOD   8
#define TLSV1_METHOD          9
#define SSLV23_SERVER_METHOD  10
#define SSLV23_CLIENT_METHOD  11
#define SSLV23_METHOD         12

// SSL connection states

// PEM encoded data types
#define RSA_PUBLIC_KEY        1 
#define RSA_PRIVATE_KEY       2 
#define RSA_PUBLIC_PRIVATE_KEY   3 
#define DSA_PUBLIC_KEY        4 
#define DSA_PRIVATE_KEY       5 
#define DH_PUBLIC_KEY         6 
#define DH_PRIVATE_KEY        7 
#define X509_CERTIFICATE      8
#define X_X509_CRL             9     //X509_CRL already used by OpenSSL library

// asymmetric ciphers
#define RSA_CIPHER            1
#define DSA_CIPHER            2
#define DH_CIPHER             3
#define NO_DSA
#define NO_DH

// digests
#define MD2_DIGEST            1
#define MD5_DIGEST            2
#define SHA_DIGEST            3
#define SHA1_DIGEST           4
#define RIPEMD160_DIGEST      5

//object format
#define SHORTNAME_FORMAT      1
#define LONGNAME_FORMAT       2

//object check functions
#define X_X509_Check(op) ((op)->ob_type == &x509type)
#define X_X509_store_Check(op) ((op)->ob_type == &x509_storetype)
#define X_X509_crl_Check(op) ((op)->ob_type == &x509_crltype)
#define X_X509_revoked_Check(op) ((op)->ob_type == &x509_revokedtype)
#define X_asymmetric_Check(op) ((op)->ob_type == &asymmetrictype)
#define X_symmetric_Check(op) ((op)->ob_type == &symmetrictype)
#define X_Digest_Check(op) ((op)->ob_type == &digesttype)
#define X_Ssl_Check(op) ((op)->ob_type == &ssltype)

static char pow_module__doc__ [] = "
<moduleDescription>
   <header>
      <name>Python OpenSSL Wrappers v0.6.1</name>
      <author>Peter Shannon</author>
   </header>
   <body>
      <para>
         This is the second release of POW and many of the missing gaps in
         functionality have been plugged.  The <classname>Ssl</classname> class has received
         several new features relating to security.  Other areas have been
         improved: PRNG support, certificate and CRL signing, certificate chain
         and client verification.  Many bugs have been fixed, and certain
         parts of code re-written where necessary.  I hope you enjoy using POW 
         and please feel free to send me feature requests and bug reports.
      </para>
   </body>
</moduleDescription>";

/*========== Pre-definitions ==========*/
static PyObject *SSLErrorObject;
static PyTypeObject x509type;
static PyTypeObject x509_storetype;
static PyTypeObject x509_crltype;
static PyTypeObject x509_revokedtype;
static PyTypeObject asymmetrictype;
static PyTypeObject symmetrictype;
static PyTypeObject digesttype;
static PyTypeObject hmactype;
static PyTypeObject ssltype;
/*========== Pre-definitions ==========*/

/*========== C stucts ==========*/
typedef struct {
	PyObject_HEAD
   X509 *x509;
} x509_object;

typedef struct {
	PyObject_HEAD
   X509_STORE *store;
} x509_store_object;

typedef struct {
	PyObject_HEAD
   X509_CRL *crl;
} x509_crl_object;

typedef struct {
	PyObject_HEAD
   X509_REVOKED *revoked;
} x509_revoked_object;

typedef struct {
	PyObject_HEAD
   void *cipher;
   int key_type;
   int cipher_type;
} asymmetric_object;

typedef struct {
	PyObject_HEAD
   EVP_CIPHER_CTX cipher_ctx;
   int cipher_type;
} symmetric_object;

typedef struct {
	PyObject_HEAD
   EVP_MD_CTX digest_ctx;
   int digest_type;
} digest_object;

typedef struct {
	PyObject_HEAD
   HMAC_CTX hmac_ctx;
} hmac_object;

typedef struct {
	PyObject_HEAD
   int ctxset;
   SSL *ssl;
   SSL_CTX *ctx;
} ssl_object;
int ssl_object_index;
/*========== C stucts ==========*/

/*========== helper funcitons ==========*/

/* 
   Simple function to install a constant in the module name space.
*/
static void
install_int_const( PyObject *d, char *name, int value )
{
   PyObject *v = PyInt_FromLong( (long)value );
   if (!v || PyDict_SetItemString(d, name, v) )
      PyErr_Clear();

   Py_XDECREF(v);
}

int
docset_helper_add(PyObject *set, char *v)
{
   PyObject *value=NULL;

   if ( !(value = PyString_FromString(v) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   if ( PyList_Append( set, value ) != 0)
      goto error;

   return 1;

error:

   Py_XDECREF(value);
   return 0;
}

/*
   Generate an encrypion envelope.  Saves a lot of space having thie case
   statement in one place.
*/
static EVP_CIPHER *
evp_cipher_factory(int cipher_type)
{
   switch(cipher_type)
   {
#ifndef NO_DES
      case DES_ECB:           return EVP_des_ecb();
      case DES_EDE:           return EVP_des_ede();
      case DES_EDE3:          return EVP_des_ede3();
      case DES_CFB:           return EVP_des_cfb();
      case DES_EDE_CFB:       return EVP_des_ede_cfb();
      case DES_EDE3_CFB:      return EVP_des_ede3_cfb();
      case DES_OFB:           return EVP_des_ofb();
      case DES_EDE_OFB:       return EVP_des_ede_ofb();
      case DES_EDE3_OFB:      return EVP_des_ede3_ofb();
      case DES_CBC:           return EVP_des_cbc();
      case DES_EDE_CBC:       return EVP_des_ede_cbc();
      case DES_EDE3_CBC:      return EVP_des_ede3_cbc();
      case DESX_CBC:          return EVP_desx_cbc();
#endif
#ifndef NO_RC4
      case RC4:               return EVP_rc4();
      case RC4_40:            return EVP_rc4_40();
#endif
#ifndef NO_IDEA
      case IDEA_ECB:          return EVP_idea_ecb();
      case IDEA_CFB:          return EVP_idea_cfb();
      case IDEA_OFB:          return EVP_idea_ofb();
      case IDEA_CBC:          return EVP_idea_cbc();
#endif
#ifndef NO_RC2
      case RC2_ECB:           return EVP_rc2_ecb();
      case RC2_CBC:           return EVP_rc2_cbc();
      case RC2_40_CBC:        return EVP_rc2_40_cbc();
      case RC2_CFB:           return EVP_rc2_cfb();
      case RC2_OFB:           return EVP_rc2_ofb();
#endif
#ifndef NO_BF
      case BF_ECB:            return EVP_bf_ecb();
      case BF_CBC:            return EVP_bf_cbc();
      case BF_CFB:            return EVP_bf_cfb();
      case BF_OFB:            return EVP_bf_ofb();
#endif
#ifndef NO_CAST5
      case CAST5_ECB:         return EVP_cast5_ecb();
      case CAST5_CBC:         return EVP_cast5_cbc();
      case CAST5_CFB:         return EVP_cast5_cfb();
      case CAST5_OFB:         return EVP_cast5_ofb();
#endif
#ifndef NO_RC5_32_12_16
      case RC5_32_12_16_CBC:  return EVP_rc5_32_12_16_cbc();
      case RC5_32_12_16_CFB:  return EVP_rc5_32_12_16_cfb();
      case RC5_32_12_16_ECB:  return EVP_rc5_32_12_16_ecb();
      case RC5_32_12_16_OFB:  return EVP_rc5_32_12_16_ofb();
#endif
      default:                return NULL;
   }
}

/*
   Builds a tuple of time and local offset from a ASN1 time.
   This function does not check for overflows of time_t.
*/
static PyObject *
helper_get_date (ASN1_UTCTIME *time)
{
   int local_hour=0, local_minute=0, local_seconds=0;
   time_t seconds=0;
   char buf[16];
   struct tm cert_time;

   switch( time->type )
   {
      case V_ASN1_UTCTIME:
      {
         memcpy( buf, time->data, 12 );
         if (!strptime( buf, "%y%m%d%H%M%S", &cert_time ) )
            { PyErr_SetString( SSLErrorObject, "problem converting UTCTIME" ); goto error; }
         seconds = mktime( &cert_time );
         if (time->data[12] != 'Z')
         {
            memcpy( buf, &time->data[13], 2 ); // get hour value
            buf[2] = 0; local_hour = atoi(buf); 
            memcpy( buf, &time->data[16], 2 ); // get minute value
            buf[2] = 0; local_minute = atoi(buf); 

            local_seconds = (local_hour * 60 * 60) + (local_minute * 60);
            if (time->data[12] == '-')
               local_seconds *= -1;
         }
         break;
      }
      case V_ASN1_GENERALIZEDTIME:
      {
         memcpy( buf, time->data, 12 );
         if (!strptime( buf, "%Y%m%d%H%M%S", &cert_time ) )
            { PyErr_SetString( SSLErrorObject, "problem converting GENERALIZEDTIME" ); goto error; }
         seconds = mktime( &cert_time );
         if (time->data[14] != 'Z')
         {
            memcpy( buf, &time->data[15], 2 ); // get hour value
            buf[2] = 0; local_hour = atoi(buf); 
            memcpy( buf, &time->data[18], 2 ); // get minute value
            buf[2] = 0; local_minute = atoi(buf); 

            local_seconds = (local_hour * 60 * 60) + (local_minute * 60);
            if (time->data[14] == '-')
               local_seconds *= -1;
         }
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "problem with time" ); goto error; }
   }

   return Py_BuildValue("(ii)", seconds, local_seconds);

error:

   return NULL;
}

static PyObject *
ssl_err_factory(int err)
{
   switch(err)
   {
      case SSL_ERROR_NONE: 
         return Py_BuildValue( "(is)", SSL_ERROR_NONE, "SSL_ERROR_NONE" );
      case SSL_ERROR_ZERO_RETURN: 
         return Py_BuildValue( "(is)", SSL_ERROR_ZERO_RETURN, "SSL_ERROR_ZERO_RETURN" ); 
      case SSL_ERROR_WANT_READ:  
         return Py_BuildValue( "(is)", SSL_ERROR_WANT_READ, "SSL_ERROR_WANT_READ" );
      case SSL_ERROR_WANT_WRITE: 
         return Py_BuildValue( "(is)", SSL_ERROR_WANT_WRITE, "SSL_ERROR_WANT_WRITE" ); 
      case SSL_ERROR_WANT_X509_LOOKUP: 
         return Py_BuildValue( "(is)", SSL_ERROR_WANT_X509_LOOKUP, "SSL_ERROR_WANT_X509_LOOKUP" ); 
      case SSL_ERROR_SYSCALL: 
         return Py_BuildValue( "(is)", SSL_ERROR_SYSCALL, "SSL_ERROR_SYSCALL" ); 
      case SSL_ERROR_SSL: 
         return Py_BuildValue( "(is)", SSL_ERROR_SSL, "SSL_ERROR_SSL" ); 

      default:
         return Py_BuildValue( "(is)", err, "UNKOWN_SSL_ERROR" ); 
   }
}
/*========== helper funcitons ==========*/

/*========== X509 Code ==========*/
static x509_object *
X509_object_new(void)
{
   x509_object *self;

   self = PyObject_New( x509_object, &x509type );
   if (self == NULL)
      goto error;

   self->x509 = X509_new();
   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

/*
   This function is pretty dumb.  Most of the work is done by the module
   function pow_module_pem_read().
*/
static x509_object *
X509_object_pem_read(BIO *in)
{
   x509_object *self;

   if ( !(self = PyObject_New( x509_object, &x509type ) ) )
      goto error;

   if( !(self->x509 = PEM_read_bio_X509( in, NULL, NULL, NULL ) ) )
      { PyErr_SetString( SSLErrorObject, "could not load certificate" ); goto error; }

   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

/*
   Unlike the previous function this creates the BIO itself.  The BIO_s_mem
   is used as a buffer which the certificate is read into, from this buffer
   it is read into a char[] and returned as a string.
*/
static char X509_object_pem_write__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>pemWrite</name>
   </header>
   <body>
      <para>
         This method returns a PEM encoded certificate as a
         string.
      </para>
   </body>
</method>";

static PyObject *
X509_object_pem_write(x509_object *self, PyObject *args)
{
   int len=0, ret=0;
   char *buf=NULL;
   BIO *out_bio=NULL;
   PyObject *cert=NULL;
   
	if (!PyArg_ParseTuple(args, ""))
		return NULL;

   out_bio = BIO_new(BIO_s_mem());

   if (!PEM_write_bio_X509(out_bio, self->x509) )
      { PyErr_SetString( SSLErrorObject, "unable to write certificate" ); goto error; }

   if ( !(len = BIO_ctrl_pending(out_bio) ) )
      { PyErr_SetString( SSLErrorObject, "unable to get bytes stored in bio" ); goto error; }

   if ( !(buf = malloc(len) ) )
      { PyErr_SetString( SSLErrorObject, "unable to allocate memory" ); goto error; }

   if ( (ret = BIO_read( out_bio, buf, len ) ) != len )
      { PyErr_SetString( SSLErrorObject, "unable to write out cert" ); goto error; }

   cert = Py_BuildValue("s#", buf, len);

   BIO_free(out_bio);
   free(buf);
   return cert;
   
error:   

   if (out_bio)
      BIO_free(out_bio);

   if (buf)
      free(buf);

   Py_XDECREF(cert);
   return NULL;
}


/*
   Currently this function only supports RSA keys.
*/
static char X509_object_set_public_key__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>setPublicKey</name>
      <parameter>key</parameter>
   </header>
   <body>
      <para>
         This method sets the public key for this certificate object.  The
         parameter <parameter>key</parameter> should be an instance of
         <classname>Asymmetric</classname> containing a public key.
      </para>
   </body>
</method>";


static PyObject *
X509_object_set_public_key(x509_object *self, PyObject *args)
{
	EVP_PKEY *pkey=NULL;
   asymmetric_object *asym;

	if (!PyArg_ParseTuple(args, "O!", &asymmetrictype, &asym))
		return NULL;

   if ( !(pkey = EVP_PKEY_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   switch( asym->key_type )
   {
      case RSA_PUBLIC_KEY:
      {
         if ( !(EVP_PKEY_assign_RSA(pkey, asym->cipher) ) )
            { PyErr_SetString( SSLErrorObject, "EVP_PKEY assignment error" ); goto error; }
         break;
      }
      case RSA_PUBLIC_PRIVATE_KEY:
      {
         if ( !(EVP_PKEY_assign_RSA(pkey, asym->cipher) ) )
            { PyErr_SetString( SSLErrorObject, "EVP_PKEY assignment error" ); goto error; }
         break;
      }

      default:
         { PyErr_SetString( SSLErrorObject, "cannot use this type of key" ); goto error; }
   }

	if ( !(X509_set_pubkey(self->x509,pkey) ) )
      { PyErr_SetString( SSLErrorObject, "could not set certificate's public key" ); goto error; }

   return Py_BuildValue("");

error:

   if (pkey)
      EVP_PKEY_free(pkey);

   return NULL;

}

static char X509_object_sign__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>sign</name>
      <parameter>key</parameter>
      <parameter>digest=MD5_DIGEST</parameter>
   </header>
   <body>
      <para>
         This method signs a certificate with a private key.  See the
         example for the methods which should be invoked before signing a
         certificate.  <parameter>key</parameter> should be an instance of
         <classname>Asymmetric</classname> containing a private key.
         The optional parameter <parameter>digest</parameter> indicates 
         which digest function should be used to compute the hash to be 
         signed, it should be one of the following:
      </para>
      <simplelist>
         <member><constant>MD2_DIGEST</constant></member>
         <member><constant>MD5_DIGEST</constant></member>
         <member><constant>SHA_DIGEST</constant></member>
         <member><constant>SHA1_DIGEST</constant></member>
         <member><constant>RIPEMD160_DIGEST</constant></member>
     </simplelist>
   </body>
</method>";


static PyObject *
X509_object_sign(x509_object *self, PyObject *args)
{
	EVP_PKEY *pkey=NULL;
   asymmetric_object *asym;
   int digest=MD5_DIGEST;

	if (!PyArg_ParseTuple(args, "O!|i", &asymmetrictype, &asym, &digest))
		return NULL;

   if ( !(pkey = EVP_PKEY_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   switch( asym->key_type )
   {
      case RSA_PRIVATE_KEY:
      {
         if ( !(EVP_PKEY_assign_RSA(pkey, asym->cipher) ) )
            { PyErr_SetString( SSLErrorObject, "EVP_PKEY assignment error" ); goto error; }
         break;
      }
      case RSA_PUBLIC_PRIVATE_KEY:
      {
         if ( !(EVP_PKEY_assign_RSA(pkey, asym->cipher) ) )
            { PyErr_SetString( SSLErrorObject, "EVP_PKEY assignment error" ); goto error; }
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "cannot use this type of key" ); goto error; }
   }

   switch (digest)
   {
      case MD5_DIGEST:
      { 
         if (!X509_sign(self->x509, pkey, EVP_md5() ) ) 
            { PyErr_SetString( SSLErrorObject, "could not sign certificate" ); goto error; }
         break;
      }
      case MD2_DIGEST:
      { 
         if (!X509_sign(self->x509, pkey, EVP_md2() ) ) 
            { PyErr_SetString( SSLErrorObject, "could not sign certificate" ); goto error; }
         break;
      }
      case SHA_DIGEST:
      { 
         if (!X509_sign(self->x509, pkey, EVP_sha() ) ) 
            { PyErr_SetString( SSLErrorObject, "could not sign certificate" ); goto error; }
         break;
      }
      case SHA1_DIGEST:
      { 
         if (!X509_sign(self->x509, pkey, EVP_sha1() ) ) 
            { PyErr_SetString( SSLErrorObject, "could not sign certificate" ); goto error; }
         break;
      }
      case RIPEMD160_DIGEST:
      { 
         if (!X509_sign(self->x509, pkey, EVP_ripemd160() ) ) 
            { PyErr_SetString( SSLErrorObject, "could not sign certificate" ); goto error; }
         break;
      }
   }

   return Py_BuildValue("");

error:

   if (pkey)
      EVP_PKEY_free(pkey);

   return NULL;

}

static PyObject *
X509_object_helper_get_name(X509_NAME *name, int format)
{
   int no_entries=0, no_pairs=0, i=0, j=0, value_len=0, nid=0;
   X509_NAME_ENTRY *entry=NULL;
   char *value=NULL, long_name[512];
   const char *short_name;

   PyObject *result_list = NULL;
   PyObject *pair = NULL;
   PyObject *py_type = NULL;
   PyObject *py_value = NULL;

   no_entries = X509_NAME_entry_count( name );

   if ( !(result_list = PyTuple_New( no_entries ) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   for(i=0; i<no_entries; i++)
   {
      if ( !(entry = X509_NAME_get_entry( name, i ) ) )
         { PyErr_SetString( SSLErrorObject, "could not get certificate name" ); goto error; }

      if (entry->value->length + 1 > value_len)
      {
         if (value)
            free(value);

         if ( !(value = malloc( entry->value->length + 1 ) ) )
            { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

         value_len = entry->value->length + 1; 
      }
      memcpy( value, entry->value->data, entry->value->length );
      value[ entry->value->length ] = 0;

      if ( !(i2t_ASN1_OBJECT(long_name, sizeof(long_name), entry->object) ) )
         { PyErr_SetString( SSLErrorObject, "could not object name" ); goto error; }

      if ( format == SHORTNAME_FORMAT )
      {
         nid = OBJ_ln2nid( long_name );
         short_name = OBJ_nid2sn( nid );
         py_type = PyString_FromString(short_name);
      }
      else if ( format == LONGNAME_FORMAT )
         py_type = PyString_FromString(long_name);
      else
         { PyErr_SetString( SSLErrorObject, "unkown name format" ); goto error; }

      py_value = PyString_FromString(value);

      if ( !(pair = PyTuple_New( 2 ) ) )
         { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

      PyTuple_SetItem( pair, 0, py_type );
      PyTuple_SetItem( pair, 1, py_value );
      PyTuple_SetItem( result_list, i, pair );
   }

   if (value)
      free(value);

   return result_list;

error:

   if (value)
      free(value);

   if (result_list)
   {
      no_pairs = PyTuple_Size( result_list );
      for (i=0; i < no_pairs; i++)
      {
         pair = PyTuple_GetItem( result_list, i );
         no_entries = PyTuple_Size( result_list );
         for (j=0; j < no_entries; j++)
         {
            py_value = PyTuple_GetItem( pair, i );
            Py_DECREF( py_value );
         }
      }
   }

   Py_XDECREF(py_type);
   Py_XDECREF(py_value);
   Py_XDECREF(result_list);
   return NULL;
}

static char X509_object_get_version__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>getVersion</name>
   </header>
   <body>
      <para>
         This method returns the version number from the version field of
         this certificate. 
      </para>
   </body>
</method>";


static PyObject *
X509_object_get_version(x509_object *self, PyObject *args)
{
   long version=0;

	if (!PyArg_ParseTuple(args, ""))
		goto error;

   if ( !(version = X509_get_version( self->x509 ) ) )
      { PyErr_SetString( SSLErrorObject, "could not get certificate version" ); goto error; }

   return Py_BuildValue("l", version);

error:

   return NULL;
}

static char X509_object_set_version__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>setVersion</name>
      <parameter>version</parameter>
   </header>
   <body>
      <para>
         This method sets the version number in the version field of
         this certificate.  <parameter>version</parameter> should be an
         integer.
      </para>
   </body>
</method>";


static PyObject *
X509_object_set_version(x509_object *self, PyObject *args)
{
   long version=0;

	if (!PyArg_ParseTuple(args, "l", &version))
		goto error;

   if ( !X509_set_version( self->x509, version ) )
      { PyErr_SetString( SSLErrorObject, "could not set certificate version" ); goto error; }

   return Py_BuildValue("");

error:

   return NULL;
}

static char X509_object_get_serial__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>getSerial</name>
   </header>
   <body>
      <para>
         This method get the serial number in the serial field of
         this certificate.
      </para>
   </body>
</method>";


static PyObject *
X509_object_get_serial(x509_object *self, PyObject *args)
{
   long serial=0;
   ASN1_INTEGER *asn1i=NULL;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;

   if ( !(asn1i = X509_get_serialNumber( self->x509 ) ) )
      { PyErr_SetString( SSLErrorObject, "could not get serial number" ); goto error; }

   if ( (serial = ASN1_INTEGER_get(asn1i) ) == -1 )
      { PyErr_SetString( SSLErrorObject, "could not convert ASN1 Integer to long" ); goto error; }

   return Py_BuildValue("l", serial);

error:

   return NULL;
}

static char X509_object_set_serial__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>setSerial</name>
      <parameter>serial</parameter>
   </header>
   <body>
      <para>
         This method sets the serial number in the serial field of
         this certificate.  <parameter>serial</parameter> should ba an
         integer.
      </para>
   </body>
</method>";


static PyObject *
X509_object_set_serial(x509_object *self, PyObject *args)
{
   long serial=0;
   ASN1_INTEGER *asn1i=NULL;

	if (!PyArg_ParseTuple(args, "l", &serial))
		goto error;

   if ( !(asn1i = ASN1_INTEGER_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   if ( !ASN1_INTEGER_set( asn1i, serial ) )
      { PyErr_SetString( SSLErrorObject, "could not set ASN1 integer" ); goto error; }

   if ( !X509_set_serialNumber( self->x509, asn1i ) )
      { PyErr_SetString( SSLErrorObject, "could not set certificate serial" ); goto error; }

   ASN1_INTEGER_free(asn1i);

   return Py_BuildValue("");

error:

   if (asn1i)
      ASN1_INTEGER_free(asn1i);

   return NULL;
}

static char X509_object_get_issuer__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>getIssuer</name>
      <parameter>format=SHORTNAME_FORMAT</parameter>
   </header>
   <body>
      <para>
         This method returns a tuple containing the issuers name.  Each
         element of the tuple is a tuple with 2 elements.  The first tuple
         is an object name and the second is it's value.  Both issuer and
         subject are names distinguished normally composed of a small
         number of objects:
      </para>
      <simplelist>
         <member><constant>c</constant> or <constant>countryName</constant></member>
         <member><constant>st</constant> or <constant>stateOrProvinceName</constant></member>
         <member><constant>o</constant> or <constant>organizationName</constant></member>
         <member><constant>l</constant> or <constant>localityName</constant></member>
         <member><constant>ou</constant> or <constant>organizationalUnitName</constant></member>
         <member><constant>cn</constant> or <constant>commonName</constant></member>
      </simplelist>
      <para>
         The data type varies from one object to another, however, all the
         common objects are strings.  It would be possible to specify any
         kind of object but that would certainly adversely effect
         portability and is not recommended.
      </para>
   </body>
</method>";


static PyObject *
X509_object_get_issuer(x509_object *self, PyObject *args)
{
   PyObject *result_list = NULL;
   X509_NAME *name = NULL;
   int format=SHORTNAME_FORMAT;

	if (!PyArg_ParseTuple(args, "|i", &format))
		goto error;

   if ( !(name = X509_get_issuer_name( self->x509 ) ) )
      { PyErr_SetString( SSLErrorObject, "could not get issuers name" ); goto error; }

   if ( !(result_list = X509_object_helper_get_name(name, format) ) )
      { PyErr_SetString( SSLErrorObject, "failed to produce name list" ); goto error; }

   return result_list;

error:

   return NULL;
}

static char X509_object_get_subject__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>getSubject</name>
      <parameter>format=SHORTNAME_FORMAT</parameter>
   </header>
   <body>
      <para>
         This method returns a tuple containg the subjects name.  See
         <function>getIssuer</function> for a description of the returned
         object's format.
      </para>
   </body>
</method>";


static PyObject *
X509_object_get_subject(x509_object *self, PyObject *args)
{
   PyObject *result_list = NULL;
   X509_NAME *name = NULL;
   int format=SHORTNAME_FORMAT;

	if (!PyArg_ParseTuple(args, "|i", &format))
		goto error;

   if ( !(name = X509_get_subject_name( self->x509 ) ) )
      { PyErr_SetString( SSLErrorObject, "could not get issuers name" ); goto error; }

   if ( !(result_list = X509_object_helper_get_name(name, format) ) )
      { PyErr_SetString( SSLErrorObject, "failed to produce name list" ); goto error; }

   return result_list;

error:

   return NULL;
}

static PyObject *
X509_object_helper_set_name(X509_NAME *name, PyObject *name_sequence)
{
   PyObject *pair = NULL;
   PyObject *type = NULL;
   PyObject *value = NULL;
   int no_pairs = 0, i = 0, str_type = 0, nid;
   char *valueptr = NULL, *typeptr = NULL;

   no_pairs = PySequence_Size( name_sequence );
   for (i=0; i < no_pairs; i++)
   {
      if ( ( pair = PySequence_GetItem( name_sequence, i ) ) == NULL  )
         return NULL;

      if ( PySequence_Size(pair) != 2 )
         { PyErr_SetString( SSLErrorObject, "each name entry must have 2 elements" ); goto error; }

      if ( !( PyTuple_Check(pair) || PyList_Check(pair) ) )
         { PyErr_SetString( PyExc_TypeError, "inapropriate type" ); goto error; }

      if ( !(type = PySequence_GetItem( pair, 0 ) ) )
         { PyErr_SetString( PyExc_TypeError, "could not get type string" ); goto error; }

      if ( !PyString_Check(type) )
         { PyErr_SetString( PyExc_TypeError, "inapropriate type" ); goto error; }

      if ( !( value = PySequence_GetItem( pair, 1 ) ) )
         { PyErr_SetString( PyExc_TypeError, "could not get value string" ); goto error; }

      if ( !PyString_Check(value) )
         { PyErr_SetString( PyExc_TypeError, "inapropriate type" ); goto error; }

      typeptr = PyString_AsString(type);
      valueptr = PyString_AsString(value);

      str_type = ASN1_PRINTABLE_type( valueptr, -1 );
      if ( !(nid = OBJ_ln2nid(typeptr)) )
         if ( !(nid = OBJ_sn2nid(typeptr)) )
            { PyErr_SetString( SSLErrorObject, "unknown ASN1 object" ); goto error; }

      if ( !X509_NAME_add_entry_by_NID( name, nid, str_type, valueptr, strlen(valueptr), -1, 0 ) )
         { PyErr_SetString( SSLErrorObject, "unable to add name entry" ); goto error; }
   }
   return name_sequence;

error:

   return NULL;
}

static char X509_object_set_subject__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>setSubject</name>
      <parameter>name</parameter>
   </header>
   <body>
      <para>
         This method is used to set the subjects name.
         <parameter>name</parameter> can be comprised of lists or tuples in
         the format described in the <function>getIssuer</function> method.
      </para>
   </body>
</method>";


static PyObject *
X509_object_set_subject(x509_object *self, PyObject *args)
{
   PyObject *name_sequence = NULL;
   X509_NAME *name = NULL;

	if (!PyArg_ParseTuple(args, "O", &name_sequence))
		return NULL;

   if ( !( PyTuple_Check( name_sequence ) || PyList_Check(name_sequence) ) )
      { PyErr_SetString( PyExc_TypeError, "Inapropriate type" ); goto error; }

   if ( !(name = X509_NAME_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   if ( !X509_object_helper_set_name(name, name_sequence) )
      { PyErr_SetString( SSLErrorObject, "unable to set new name" ); goto error; }

	if ( !X509_set_subject_name(self->x509,name) )
      { PyErr_SetString( SSLErrorObject, "unable to set name" ); goto error; }
   
   X509_NAME_free(name);

   return Py_BuildValue("");

error:

   return NULL;
}

static char X509_object_set_issuer__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>setIssuer</name>
      <parameter>name</parameter>
   </header>
   <body>
      <para>
         This method is used to set the issuers name.
         <parameter>name</parameter> can be comprised of lists or tuples in
         the format described in the <function>getissuer</function> method.
      </para>
   </body>
</method>";


static PyObject *
X509_object_set_issuer(x509_object *self, PyObject *args)
{
   PyObject *name_sequence = NULL;
   X509_NAME *name = NULL;

	if (!PyArg_ParseTuple(args, "O", &name_sequence))
		return NULL;

   if ( !( PyTuple_Check( name_sequence ) || PyList_Check(name_sequence) ) )
      { PyErr_SetString( PyExc_TypeError, "Inapropriate type" ); goto error; }

   if ( !(name = X509_NAME_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   if ( !X509_object_helper_set_name(name, name_sequence) )
      { PyErr_SetString( SSLErrorObject, "unable to set new name" ); goto error; }

	if ( !X509_set_issuer_name(self->x509,name) )
      { PyErr_SetString( SSLErrorObject, "unable to set name" ); goto error; }

   X509_NAME_free(name);

   return Py_BuildValue("");

error:

   return  NULL;
}

static char X509_object_get_not_before__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>getNotBefore</name>
   </header>
   <body>
      <para>
         This method returns a tuple containing two integers. The first
         number represents the time in seconds and is the same as the C 
         <constant>time_t</constant> typedef and the second represents the time zone offset in
         seconds.
      </para>
   </body>
</method>";


static PyObject *
X509_object_get_not_before (x509_object *self, PyObject *args)
{
   ASN1_UTCTIME *time;

	if (!PyArg_ParseTuple(args, ""))
		goto error;

   if ( !(time = X509_get_notBefore(self->x509) ) )
      { PyErr_SetString( SSLErrorObject, "could not get time" ); goto error; }

   return helper_get_date( time );

error:

   return NULL;
}

static char X509_object_get_not_after__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>getNotAfter</name>
   </header>
   <body>
      <para>
         This method returns a tuple containing two integers. The first
         number represents the time in seconds and is the same as the C 
         <constant>time_t</constant> typedef and the second represents the time zone offset in
         seconds.
      </para>
   </body>
</method>";


static PyObject *
X509_object_get_not_after (x509_object *self, PyObject *args)
{
   ASN1_UTCTIME *time=NULL;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;

   if ( !(time = X509_get_notAfter(self->x509) ) )
      { PyErr_SetString( SSLErrorObject, "could not get time" ); goto error; }

   return helper_get_date( time );

error:

   return NULL;
}

static char X509_object_set_not_after__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>setNotAfter</name>
      <parameter>time</parameter>
   </header>
   <body>
      <para>
         This method sets part of the <constant>Validity</constant>
         sequence of the certificate, the <constant>notAfter</constant>
         time.  <parameter>time</parameter> should be a time in seconds,
         as generated by the <function>time</function> function in the Python
         Standard Library.
      </para>
   </body>
</method>";


static PyObject *
X509_object_set_not_after (x509_object *self, PyObject *args)
{
   int new_time = 0;

	if (!PyArg_ParseTuple(args, "i", &new_time))
		goto error;

	if ( !ASN1_UTCTIME_set(X509_get_notAfter(self->x509),new_time) )
      { PyErr_SetString( SSLErrorObject, "could not set time" ); goto error; }

   return Py_BuildValue("");

error:

   return NULL;
}

static char X509_object_set_not_before__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>setNotBefore</name>
      <parameter>time</parameter>
   </header>
   <body>
      <para>
         This method sets part of the <constant>Validity</constant>
         sequence of the certificate, the <constant>notBefore</constant>
         time.  <parameter>time</parameter> should be a time in seconds,
         as generated by the <function>time</function> function in the Python
         Standard Library.
      </para>
   </body>
</method>";


static PyObject *
X509_object_set_not_before (x509_object *self, PyObject *args)
{
   int new_time = 0;

	if (!PyArg_ParseTuple(args, "i", &new_time))
		goto error;

	if ( !ASN1_UTCTIME_set(X509_get_notBefore(self->x509),new_time) )
      { PyErr_SetString( SSLErrorObject, "could not set time" ); goto error; }

   return Py_BuildValue("");

error:

   return NULL;
}

static char x509_object_pprint__doc__[] = "
<method>
   <header>
      <memberof>X509</memberof>
      <name>pprint</name>
   </header>
   <body>
      <para>
         This method returns a formatted string showing the information
         held in the certificate.
      </para>
   </body>
</method>";


static PyObject *
x509_object_pprint(x509_object *self, PyObject *args)
{
   int len=0, ret=0;
   char *buf=NULL;
   BIO *out_bio=NULL;
   PyObject *cert=NULL;
   
	if (!PyArg_ParseTuple(args, ""))
		return NULL;

   out_bio = BIO_new(BIO_s_mem());

   if (!X509_print(out_bio, self->x509) )
      { PyErr_SetString( SSLErrorObject, "unable to write crl" ); goto error; }

   if ( !(len = BIO_ctrl_pending(out_bio) ) )
      { PyErr_SetString( SSLErrorObject, "unable to get bytes stored in bio" ); goto error; }

   if ( !(buf = malloc(len) ) )
      { PyErr_SetString( SSLErrorObject, "unable to allocate memory" ); goto error; }

   if ( (ret = BIO_read( out_bio, buf, len ) ) != len )
      { PyErr_SetString( SSLErrorObject, "unable to write out cert" ); goto error; }

   cert = Py_BuildValue("s#", buf, len);

   BIO_free(out_bio);
   free(buf);
   return cert;
   
error:   

   if (out_bio)
      BIO_free(out_bio);

   if (buf)
      free(buf);

   return NULL;

}

static struct PyMethodDef X509_object_methods[] = {
   {"pemWrite",      (PyCFunction)X509_object_pem_write,  
                           METH_VARARGS,  X509_object_pem_write__doc__}, 
   {"sign",          (PyCFunction)X509_object_sign,  
                           METH_VARARGS,  X509_object_sign__doc__}, 
   {"setPublicKey",        (PyCFunction)X509_object_set_public_key,  
                           METH_VARARGS,  X509_object_set_public_key__doc__}, 
   {"getVersion",    (PyCFunction)X509_object_get_version,  
                           METH_VARARGS,  X509_object_get_version__doc__}, 
   {"setVersion",    (PyCFunction)X509_object_set_version,    
                           METH_VARARGS,  X509_object_set_version__doc__}, 
   {"getSerial",     (PyCFunction)X509_object_get_serial,   
                           METH_VARARGS,  X509_object_get_serial__doc__}, 
   {"setSerial",     (PyCFunction)X509_object_set_serial,   
                           METH_VARARGS,  X509_object_set_serial__doc__}, 
   {"getIssuer",     (PyCFunction)X509_object_get_issuer,  
                           METH_VARARGS,  X509_object_get_issuer__doc__}, 
   {"setIssuer",     (PyCFunction)X509_object_set_issuer,   
                           METH_VARARGS,  X509_object_set_issuer__doc__}, 
   {"getSubject",    (PyCFunction)X509_object_get_subject,  
                           METH_VARARGS,  X509_object_get_subject__doc__}, 
   {"setSubject",    (PyCFunction)X509_object_set_subject,  
                           METH_VARARGS,  X509_object_set_subject__doc__}, 
   {"getNotBefore",  (PyCFunction)X509_object_get_not_before,  
                           METH_VARARGS,  X509_object_get_not_before__doc__}, 
   {"getNotAfter",   (PyCFunction)X509_object_get_not_after,   
                           METH_VARARGS,  X509_object_get_not_after__doc__}, 
   {"setNotAfter",   (PyCFunction)X509_object_set_not_after,  
                           METH_VARARGS,  X509_object_set_not_after__doc__}, 
   {"setNotBefore",  (PyCFunction)X509_object_set_not_before,  
                           METH_VARARGS,  X509_object_set_not_before__doc__}, 
   {"pprint",        (PyCFunction)x509_object_pprint,     
                           METH_VARARGS,  x509_object_pprint__doc__}, 
 
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
X509_object_getattr(x509_object *self, char *name)
{
	return Py_FindMethod(X509_object_methods, (PyObject *)self, name);
}

static void
X509_object_dealloc(x509_object *self, char *name)
{
   X509_free( self->x509 );
   PyObject_Del(self);
}

static char x509type__doc__[] = "
<class>
   <header>
      <name>X509</name>
   </header>
   <body>
      <para>
         This class provides access to a significant proportion of X509 
         functionality of OpenSSL.
      </para>

      <example>
         <title><classname>x509</classname> class usage</title>
         <programlisting>
      privateFile = open('test/private.key', 'r')
      publicFile = open('test/public.key', 'r')
      certFile = open('test/cacert.pem', 'w')

      publicKey = POW.pemRead(POW.RSA_PUBLIC_KEY, publicFile.read())
      privateKey = POW.pemRead(POW.RSA_PRIVATE_KEY, privateFile.read(), 'pass')

      c = POW.X509()

      name = [  ['C', 'GB'], ['ST', 'Hertfordshire'], 
                ['O','The House'], ['CN', 'Peter Shannon'] ]

      c.setIssuer( name )
      c.setSubject( name )
      c.setSerial(0)
      c.setNotBefore( time.time() )
      c.setNotAfter( time.time() + 60*60*24*365)
      c.setPublicKey(publicKey)
      c.sign(privateKey)

      certFile.write( c.pemWrite() )

      privateFile.close()
      publicFile.close()
      certFile.close()
         </programlisting>
      </example>

   </body>
</class>";


static PyTypeObject x509type = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				                        /*ob_size*/
	"X509",			                     /*tp_name*/
	sizeof(x509_object),		            /*tp_basicsize*/
	0,				                        /*tp_itemsize*/
	(destructor)X509_object_dealloc,	   /*tp_dealloc*/
	(printfunc)0,		                  /*tp_print*/
	(getattrfunc)X509_object_getattr,   /*tp_getattr*/
	(setattrfunc)0,	                  /*tp_setattr*/
	(cmpfunc)0,		                     /*tp_compare*/
	(reprfunc)0,      		            /*tp_repr*/
	0,			                           /*tp_as_number*/
	0,		                              /*tp_as_sequence*/
	0,		                              /*tp_as_mapping*/
	(hashfunc)0,		                  /*tp_hash*/
	(ternaryfunc)0,		               /*tp_call*/
	(reprfunc)0,		                  /*tp_str*/
	0,
   0,
   0,
   0,
	x509type__doc__                     /* Documentation string */
};
/*========== X509 Code ==========*/

/*========== x509 store Code ==========*/
static x509_store_object *
x509_store_object_new(void)
{
   x509_store_object *self=NULL;

   self = PyObject_New( x509_store_object, &x509_storetype );
   if (self == NULL)
      goto error;

   self->store = X509_STORE_new();

   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

static char x509_store_object_verify__doc__[] = "
<method>
   <header>
      <memberof>X509Store</memberof>
      <name>verify</name>
      <parameter>certificate</parameter>
   </header>
   <body>
      <para>
         The <classname>X509Store</classname> method
         <function>verify</function> is based on the
         <function>X509_verify_cert</function>.  It handles certain aspects
         of verification but not others.  The certificate will be verified
         against <constant>notBefore</constant>, 
         <constant>notAfter</constant> and trusted certificates.
         It crucially will not handle checking the certificate against
         CRLs.  This functionality will probably make it into OpenSSL
         0.9.7.
      </para>
   </body>
</method>";

static PyObject *
x509_store_object_verify(x509_store_object *self, PyObject *args)
{
   X509_STORE_CTX csc;
   x509_object *x509=NULL;
   int result=0;

	if (!PyArg_ParseTuple(args, "O!", &x509type, &x509))
		goto error;

   X509_STORE_CTX_init( &csc, self->store, x509->x509, NULL );
   result = X509_verify_cert( &csc );

   X509_STORE_CTX_cleanup( &csc );

   return Py_BuildValue("i", result);

error:

   return NULL;
}

static char x509_store_object_verify_chain__doc__[] = "
<method>
   <header>
      <memberof>X509Store</memberof>
      <name>verifyChain</name>
      <parameter>certificate</parameter>
      <parameter>chain</parameter>
   </header>
   <body>
      <para>
         The <classname>X509Store</classname> method <function>verifyChain</function> 
         is based on the <function>X509_verify_cert</function> but is initialised 
         with a <classname>X509</classname> object to verify and list of 
         <classname>X509</classname> objects which form a chain to a trusted 
         certificate.  Certain aspects of the verification are handled but not others.  
         The certificates will be verified against <constant>notBefore</constant>, 
         <constant>notAfter</constant> and trusted certificates.  It crucially will 
         not handle checking the certificate against CRLs.  This functionality will 
         probably make it into OpenSSL 0.9.7.
      </para>
      <para>
         This may all sound quite straight forward but determining the 
         certificate associated with the signature on another certificate
         can be very time consuming.  The management aspects of
         certificates are addressed by various V3 extensions which are not
         currently supported.
      </para>
   </body>
</method>";

static PyObject *
x509_store_object_verify_chain(x509_store_object *self, PyObject *args)
{
   PyObject *x509_sequence=NULL;
   X509_STORE_CTX csc;
   x509_object *x509=NULL, *tmpX509=NULL;
   STACK_OF(X509) *x509_stack=NULL;
   int result=0, size=0, i=0;

	if (!PyArg_ParseTuple(args, "O!O", &x509type, &x509, &x509_sequence))
		return NULL;

   if ( !( PyTuple_Check( x509_sequence ) || PyList_Check(x509_sequence) ) )
      { PyErr_SetString( PyExc_TypeError, "inapropriate type" ); goto error; }

   size = PySequence_Size( x509_sequence );

   if (!(x509_stack = sk_X509_new_null() ) )
      { PyErr_SetString( SSLErrorObject, "could not create new x509 stack" ); goto error; }

   for (i=0; i < size; i++)
   {
      if ( !( tmpX509 = (x509_object*)PySequence_GetItem( x509_sequence, i ) ) )
         goto error;

      if ( !X_X509_Check( tmpX509 ) )
         { PyErr_SetString( PyExc_TypeError, "inapropriate type" ); goto error; }

      if (!sk_X509_push( x509_stack, tmpX509->x509 ) )
         { PyErr_SetString( SSLErrorObject, "could not add x509 to stack" ); goto error; }
   }

   X509_STORE_CTX_init( &csc, self->store, x509->x509, x509_stack );
   result = X509_verify_cert( &csc );

   X509_STORE_CTX_cleanup( &csc );
   sk_X509_free(x509_stack);
   return Py_BuildValue("i", result);

error:

   if(x509_stack)
      sk_X509_free(x509_stack);

   return NULL;
}

static char x509_store_object_add_trust__doc__[] = "
<method>
   <header>
      <memberof>X509Store</memberof>
      <name>addTrust</name>
      <parameter>cert</parameter>
   </header>
   <body>
      <para>
         This method adds a new certificate to the store to be used in the
         verification process.  <parameter>cert</parameter> should be an
         instance of <classname>X509</classname>.  Using trusted certificates to manage
         verification is relatively primitive, more sophisticated systems
         can be constructed at an application level by by constructing
         certifcate chains to verify. 
      </para>
   </body>
</method>";

static PyObject *
x509_store_object_add_trust(x509_store_object *self, PyObject *args)
{
   x509_object *x509=NULL;

	if (!PyArg_ParseTuple(args, "O!", &x509type, &x509))
		goto error;

   X509_STORE_add_cert( self->store, x509->x509 );

   return Py_BuildValue("");

error:

   return NULL;
}

static char x509_store_object_add_crl__doc__[] = "
<method>
   <header>
      <memberof>X509Store</memberof>
      <name>addCrl</name>
      <parameter>crl</parameter>
   </header>
   <body>
      <para>
         This method adds a CRL to a store to be used for verification.
         <parameter>crl</parameter> should be an instance of
         <classname>X509Crl</classname>.
         Unfortunately, the current stable release of OpenSSL does not
         support CRL checking for certificate verification.
         This functionality will probably make it into OpenSSL 0.9.7, until
         it does this function is useless and CRL verification must be
         implemented by the application.
      </para>
   </body>
</method>";

static PyObject *
x509_store_object_add_crl(x509_store_object *self, PyObject *args)
{
   x509_crl_object *crl=NULL;

	if (!PyArg_ParseTuple(args, "O!", &x509_crltype, &crl))
		goto error;

   X509_STORE_add_crl( self->store, crl->crl );

   return Py_BuildValue("");

error:

   return NULL;
}

static struct PyMethodDef x509_store_object_methods[] = {
   {"verify",           (PyCFunction)x509_store_object_verify,  
                           METH_VARARGS,  x509_store_object_verify__doc__}, 
   {"verifyChain",      (PyCFunction)x509_store_object_verify_chain,  
                           METH_VARARGS,  x509_store_object_verify_chain__doc__}, 
   {"addTrust",         (PyCFunction)x509_store_object_add_trust,  
                           METH_VARARGS,  x509_store_object_add_trust__doc__}, 
   {"addCrl",           (PyCFunction)x509_store_object_add_crl, //   
                           METH_VARARGS,  x509_store_object_add_crl__doc__}, 
 
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
x509_store_object_getattr(x509_store_object *self, char *name)
{
	return Py_FindMethod(x509_store_object_methods, (PyObject *)self, name);
}

static void
x509_store_object_dealloc(x509_store_object *self, char *name)
{
   X509_STORE_free( self->store );
   PyObject_Del(self);
}

static char x509_storetype__doc__[] = "
<class>
   <header>
      <name>X509Store</name>
   </header>
   <body>
      <para>
         This class provides preliminary access to OpenSSL X509 verification
         facilities.
      </para>

      <example>
         <title><classname>x509_store</classname> class usage</title>
         <programlisting>
      store = POW.X509Store()

      caFile = open( 'test/cacert.pem', 'r' )
      ca = POW.pemRead( POW.X509_CERTIFICATE, caFile.read() )
      caFile.close()

      store.addTrust( ca )

      certFile = open( 'test/foocom.cert', 'r' )
      x509 = POW.pemRead( POW.X509_CERTIFICATE, certFile.read() )
      certFile.close()

      print x509.pprint()
      
      if store.verify( x509 ):
         print 'Verified certificate!.'
      else:
         print 'Failed to verify certificate!.'
         </programlisting>
      </example>
   </body>
</class>";


static PyTypeObject x509_storetype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				                              /*ob_size*/
	"X509Store",		                        /*tp_name*/
	sizeof(x509_store_object),	               /*tp_basicsize*/
	0,				                              /*tp_itemsize*/
	(destructor)x509_store_object_dealloc,	   /*tp_dealloc*/
	(printfunc)0,		                        /*tp_print*/
	(getattrfunc)x509_store_object_getattr,	/*tp_getattr*/
	(setattrfunc)0,	                        /*tp_setattr*/
	(cmpfunc)0,		                           /*tp_compare*/
	(reprfunc)0,      		                  /*tp_repr*/
	0,			                                 /*tp_as_number*/
	0,		                                    /*tp_as_sequence*/
	0,		                                    /*tp_as_mapping*/
	(hashfunc)0,		                        /*tp_hash*/
	(ternaryfunc)0,		                     /*tp_call*/
	(reprfunc)0,		                        /*tp_str*/
	0,
   0,
   0,
   0,
	x509_storetype__doc__                    /* Documentation string */
};
/*========== x509 store Code ==========*/

/*========== x509 crl Code ==========*/
static x509_crl_object *
x509_crl_object_new(void)
{
   x509_crl_object *self=NULL;

   self = PyObject_New( x509_crl_object, &x509_crltype );
   if (self == NULL)
      goto error;

   self->crl = X509_CRL_new();

   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

static x509_crl_object *
x509_crl_object_pem_read(BIO *in)
{
   x509_crl_object *self;

   self = PyObject_New( x509_crl_object, &x509_crltype );
   if (self == NULL)
      goto error;

   if( !(self->crl = PEM_read_bio_X509_CRL( in, NULL, NULL, NULL ) ) )
      { PyErr_SetString( SSLErrorObject, "could not load certificate" ); goto error; }

   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

static char x509_crl_object_get_version__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>getVersion</name>
   </header>
   <body>
      <para>
         This method returns the version number from the version field of
         this CRL. 
      </para>
   </body>
</method>";


static PyObject *
x509_crl_object_get_version(x509_crl_object *self, PyObject *args)
{
   long version=0;

	if (!PyArg_ParseTuple(args, ""))
		goto error;

   if ( (version = ASN1_INTEGER_get( self->crl->crl->version ) ) == -1 )
      { PyErr_SetString( SSLErrorObject, "could not get crl version" ); goto error; }

   return Py_BuildValue("l", version);

error:

   return NULL;
}

static char x509_crl_object_set_version__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>setVersion</name>
      <parameter>version</parameter>
   </header>
   <body>
      <para>
         This method sets the version number in the version field of
         this CRL.  <parameter>version</parameter> should be an
         integer.
      </para>
   </body>
</method>";


static PyObject *
x509_crl_object_set_version(x509_crl_object *self, PyObject *args)
{
   long version=0;
   ASN1_INTEGER *asn1_version=NULL;

	if (!PyArg_ParseTuple(args, "i", &version))
		goto error;

   if ( !(asn1_version = ASN1_INTEGER_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   if ( !ASN1_INTEGER_set( asn1_version, version ) )
      { PyErr_SetString( SSLErrorObject, "could not get set version" ); goto error; }

   self->crl->crl->version = asn1_version;

   return Py_BuildValue("");

error:

   if (asn1_version)
      ASN1_INTEGER_free(asn1_version);

   return NULL;
}

static char x509_crl_object_get_issuer__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>getIssuer</name>
      <parameter>format=SHORTNAME_FORMAT</parameter>
   </header>
   <body>
      <para>
         This method returns a tuple containg the issuers name.  See the
         <function>getIssuer</function> method of
         <classname>X509</classname> for more details.
      </para>
   </body>
</method>";


static PyObject *
x509_crl_object_get_issuer(x509_crl_object *self, PyObject *args)
{
   PyObject *result_list = NULL;
   int format=SHORTNAME_FORMAT;

	if (!PyArg_ParseTuple(args, "|i", &format))
		goto error;

   if ( !(result_list = X509_object_helper_get_name(self->crl->crl->issuer, format) ) )
      { PyErr_SetString( SSLErrorObject, "failed to produce name list" ); goto error; }

   return result_list;

error:

   return NULL;
}

static char x509_crl_object_set_issuer__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>setIssuer</name>
      <parameter>name</parameter>
   </header>
   <body>
      <para>
         This method is used to set the issuers name.
         <parameter>name</parameter> can be comprised of lists or tuples in
         the format described in the <function>getIssuer</function> method
         of <classname>X509</classname>.
      </para>
   </body>
</method>";


static PyObject *
x509_crl_object_set_issuer(x509_crl_object *self, PyObject *args)
{
   PyObject *name_sequence = NULL;
   X509_NAME *name = NULL;

	if (!PyArg_ParseTuple(args, "O", &name_sequence))
		return NULL;

   if ( !( PyTuple_Check( name_sequence ) || PyList_Check(name_sequence) ) )
      { PyErr_SetString( PyExc_TypeError, "Inapropriate type" ); goto error; }

   if ( !(name = X509_NAME_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   if ( !X509_object_helper_set_name(name, name_sequence) )
      { PyErr_SetString( SSLErrorObject, "unable to set new name" ); goto error; }

	if ( !X509_NAME_set(&self->crl->crl->issuer,name ) )
      { PyErr_SetString( SSLErrorObject, "unable to set name" ); goto error; }

   X509_NAME_free(name);

   return Py_BuildValue("");

error:

   if (name)
      X509_NAME_free(name);

   return  NULL;
}

static char x509_crl_object_set_this_update__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>setThisUpdate</name>
      <parameter>time</parameter>
   </header>
   <body>
      <para>
         This method sets the <constant>thisUpdate</constant>
         field of this CRL.  <parameter>time</parameter> should be a time in seconds,
         as generated by the <function>time</function> function in the Python
         Standard Library.
      </para>
   </body>
</method>";

static PyObject *
x509_crl_object_set_this_update (x509_crl_object *self, PyObject *args)
{
   int new_time = 0;

	if (!PyArg_ParseTuple(args, "i", &new_time))
		goto error;

	if ( !ASN1_UTCTIME_set(self->crl->crl->lastUpdate,new_time) )
      { PyErr_SetString( SSLErrorObject, "could not set time" ); goto error; }

   return Py_BuildValue("");

error:

   return NULL;
}

static char x509_crl_object_get_this_update__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>getThisUpdate</name>
   </header>
   <body>
      <para>
         This method returns a tuple containing two integers. The first
         number represents the time in seconds and is the same as the C 
         <constant>time_t</constant> typedef and the second represents the time zone offset in
         seconds.
      </para>
   </body>
</method>";

static PyObject *
x509_crl_object_get_this_update (x509_crl_object *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ""))
		goto error;

   return helper_get_date( self->crl->crl->lastUpdate );

error:

   return NULL;
}

static char x509_crl_object_set_next_update__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>setNextUpdate</name>
      <parameter>time</parameter>
   </header>
   <body>
      <para>
         This method sets the <constant>thisUpdate</constant>
         field of this CRL.  <parameter>time</parameter> should be a time in seconds,
         as generated by the <function>time</function> function in the Python
         Standard Library.
      </para>
   </body>
</method>";

static PyObject *
x509_crl_object_set_next_update (x509_crl_object *self, PyObject *args)
{
   int new_time = 0;
   ASN1_UTCTIME *time=NULL;

	if (!PyArg_ParseTuple(args, "i", &new_time))
		goto error;

   if ( !(time = ASN1_UTCTIME_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

	if (!ASN1_UTCTIME_set(time, new_time) )
      { PyErr_SetString( SSLErrorObject, "could not set next update" ); goto error; }

	self->crl->crl->nextUpdate = time;

   return Py_BuildValue("");

error:

   return NULL;
}

static char x509_crl_object_get_next_update__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>getNextUpdate</name>
   </header>
   <body>
      <para>
         This method returns a tuple containing two integers. The first
         number represents the time in seconds and is the same as the C 
         <constant>time_t</constant> typedef and the second represents the time zone offset in
         seconds.
      </para>
   </body>
</method>";

static PyObject *
x509_crl_object_get_next_update (x509_crl_object *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ""))
		goto error;

   return helper_get_date( self->crl->crl->nextUpdate );

error:

   return NULL;
}

static char x509_crl_object_set_revoked__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>setRevoked</name>
      <parameter>revoked</parameter>
   </header>
   <body>
      <para>
         This method sets the sequence of revoked certificates in this CRL.
         <parameter>revoked</parameter> should be a list or tuple of 
         <classname>X509Revoked</classname>.
      </para>
      <example>
         <title><function>setRevoked</function> function usage</title>
         <programlisting>
      privateFile = open('test/private.key', 'r')
      publicFile = open('test/public.key', 'r')
      crlFile = open('test/crl.pem', 'w')

      publicKey = POW.pemRead(POW.RSA_PUBLIC_KEY, publicFile.read())
      privateKey = POW.pemRead(POW.RSA_PRIVATE_KEY, privateFile.read(), 'pass')

      crl = POW.X509Crl()

      name = [  ['C', 'GB'], ['ST', 'Hertfordshire'], 
                ['O','The House'], ['CN', 'Peter Shannon'] ]

      crl.setIssuer( name )
      rev = [  POW.X509Revoked(3, int( time.time() ) - 24*60*60 ),
               POW.X509Revoked(4, int( time.time() ) - 24*60*60 ),
               POW.X509Revoked(5, int( time.time() ) - 24*60*60 )    ]

      crl.setRevoked( rev )
      crl.setThisUpdate( time.time() )
      crl.setNextUpdate( time.time() + 2*60*60*24*365)
      crl.sign(privateKey)

      crlFile.write( crl.pemWrite() )

      privateFile.close()
      publicFile.close()
      crlFile.close()
         </programlisting>
      </example>

   </body>
</method>";

static PyObject *
x509_crl_object_set_revoked(x509_crl_object *self, PyObject *args)
{
   PyObject *revoked_sequence = NULL;
   x509_revoked_object *revoked = NULL;
   STACK_OF(X509_REVOKED) *revoked_stack = NULL;
   int i=0,size=0;

	if (!PyArg_ParseTuple(args, "O", &revoked_sequence))
		return NULL;

   if ( !( PyTuple_Check( revoked_sequence ) || PyList_Check(revoked_sequence) ) )
      { PyErr_SetString( PyExc_TypeError, "inapropriate type" ); goto error; }

   revoked_stack = self->crl->crl->revoked;

   size = PySequence_Size( revoked_sequence );
   for (i=0; i < size; i++)
   {
      if ( !( revoked = (x509_revoked_object*)PySequence_GetItem( revoked_sequence, i ) ) )
         goto error;

      if ( !X_X509_revoked_Check( revoked ) )
         { PyErr_SetString( PyExc_TypeError, "inapropriate type" ); goto error; }

      if (!sk_X509_REVOKED_push( revoked_stack, revoked->revoked ) )
         { PyErr_SetString( SSLErrorObject, "could not add revokation to stack" ); goto error; }
   }

   return Py_BuildValue("");

error:

   return  NULL;
}

static PyObject *
x509_crl_object_helper_get_revoked(STACK_OF(X509_REVOKED) *revoked)
{
   int no_entries=0, inlist=0, i=0;
   X509_REVOKED *revoke_tmp=NULL;
   x509_revoked_object *revoke_obj=NULL;
   PyObject *item=NULL, *result_list=NULL, *result_tuple=NULL;

   no_entries = sk_X509_REVOKED_num( revoked );

   if ( !(result_list = PyList_New(0) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   for(i=0; i<no_entries; i++)
   {
      if ( !(revoke_obj = PyObject_New( x509_revoked_object, &x509_revokedtype ) ) )
         { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

      if ( !(revoke_tmp = sk_X509_REVOKED_value( revoked, i ) ) )
         { PyErr_SetString( SSLErrorObject, "could not get revocation" ); goto error; }

      revoke_obj->revoked = revoke_tmp;

      if ( PyList_Append( result_list, (PyObject*)revoke_obj ) != 0)
         goto error;
   }

	result_tuple = PyList_AsTuple( result_list );
   Py_DECREF(result_list);
 
   return Py_BuildValue("O", result_tuple);

error:

   if (result_list)
   {
      inlist = PyList_Size( result_list );
      for (i=0; i < inlist; i++)
      {
         item = PyList_GetItem( result_list, i );
         Py_DECREF(item);
      }
      Py_DECREF(result_list);
   }

   return NULL;
}

static char x509_crl_object_get_revoked__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>getRevoked</name>
   </header>
   <body>
      <para>
         This method returns a tuple of <classname>X509Revoked</classname>
         objects described in the CRL.
      </para>
      <example>
         <title><function>getRevoked</function> function usage</title>
         <programlisting>
      publicFile = open('test/public.key', 'r')
      crlFile = open('test/crl.pem', 'r')

      publicKey = POW.pemRead(POW.RSA_PUBLIC_KEY, publicFile.read())

      crl = POW.pemRead( POW.X509_CRL, crlFile.read() )

      print crl.pprint()
      if crl.verify( publicKey ):
         print 'signature ok!'
      else:
         print 'signature not ok!'

      revocations = crl.getRevoked()
      for revoked in revocations:
         print 'serial number:', revoked.getSerial()
         print 'date:', time.ctime( revoked.getDate()[0] )

      publicFile.close()
      crlFile.close()
         </programlisting>
      </example>

   </body>
</method>";

static PyObject *
x509_crl_object_get_revoked(x509_crl_object *self, PyObject *args)
{
   PyObject *revoked = NULL;

	if (!PyArg_ParseTuple(args, ""))
		goto error;

   revoked = x509_crl_object_helper_get_revoked( X509_CRL_get_REVOKED(self->crl) );

   return revoked;

error:

   return  NULL;
}

static char x509_crl_object_sign__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>sign</name>
      <parameter>key</parameter>
      <parameter>digest=MD5_DIGEST</parameter>
   </header>
   <body>
      <para>
         <parameter>key</parameter> should be an instance of
         <classname>Asymmetric</classname> and contain a private key.
         <parameter>digest</parameter> indicates 
         which digest function should be used to compute the hash to be 
         signed, it should be one of the following:
      </para>
      <simplelist>
         <member><constant>MD2_DIGEST</constant></member>
         <member><constant>MD5_DIGEST</constant></member>
         <member><constant>SHA_DIGEST</constant></member>
         <member><constant>SHA1_DIGEST</constant></member>
         <member><constant>RIPEMD160_DIGEST</constant></member>
     </simplelist>
   </body>

</method>";

static PyObject *
x509_crl_object_sign(x509_crl_object *self, PyObject *args)
{
	EVP_PKEY *pkey=NULL;
   asymmetric_object *asym;
   int digest=MD5_DIGEST;

	if (!PyArg_ParseTuple(args, "O!|i", &asymmetrictype, &asym, &digest))
		return NULL;

   if ( !(pkey = EVP_PKEY_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   switch( asym->key_type )
   {
      case RSA_PRIVATE_KEY:
      {
         if ( !(EVP_PKEY_assign_RSA(pkey, asym->cipher) ) )
            { PyErr_SetString( SSLErrorObject, "EVP_PKEY assignment error" ); goto error; }
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "cannot use this type of key" ); goto error; }
   }

   switch (digest)
   {
      case MD5_DIGEST:
      { 
         if (!X509_CRL_sign(self->crl, pkey, EVP_md5() ) ) 
            { PyErr_SetString( SSLErrorObject, "could not sign certificate" ); goto error; }
         break;
      }
      case MD2_DIGEST:
      { 
         if (!X509_CRL_sign(self->crl, pkey, EVP_md2() ) ) 
            { PyErr_SetString( SSLErrorObject, "could not sign certificate" ); goto error; }
         break;
      }
      case SHA_DIGEST:
      { 
         if (!X509_CRL_sign(self->crl, pkey, EVP_sha() ) ) 
            { PyErr_SetString( SSLErrorObject, "could not sign certificate" ); goto error; }
         break;
      }
      case SHA1_DIGEST:
      { 
         if (!X509_CRL_sign(self->crl, pkey, EVP_sha1() ) ) 
            { PyErr_SetString( SSLErrorObject, "could not sign certificate" ); goto error; }
         break;
      }
      case RIPEMD160_DIGEST:
      { 
         if (!X509_CRL_sign(self->crl, pkey, EVP_ripemd160() ) ) 
            { PyErr_SetString( SSLErrorObject, "could not sign certificate" ); goto error; }
         break;
      }
   }

   return Py_BuildValue("");

error:

   if (pkey)
      EVP_PKEY_free(pkey);

   return NULL;

}

static char x509_crl_object_verify__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>verify</name>
      <parameter>key</parameter>
   </header>
   <body>
      <para>
         The <classname>X509Crl</classname> method
         <function>verify</function> is based on the
         <function>X509_CRL_verify</function> function.  Unlike the
         <classname>X509</classname> function of the same name, this
         function simply checks the CRL was signed with the private key
         which corresponds the parameter <parameter>key</parameter>.
         <parameter>key</parameter> should be an instance of
         <classname>Asymmetric</classname> and contain a public key.
      </para>
   </body>
</method>";

static PyObject *
x509_crl_object_verify(x509_crl_object *self, PyObject *args)
{
   int result=0;
	EVP_PKEY *pkey=NULL;
   asymmetric_object *asym;

	if (!PyArg_ParseTuple(args, "O!", &asymmetrictype, &asym))
		return NULL;

   if ( !(pkey = EVP_PKEY_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   switch( asym->key_type )
   {
      case RSA_PUBLIC_KEY:
      {
         if ( !(EVP_PKEY_assign_RSA(pkey, asym->cipher) ) )
            { PyErr_SetString( SSLErrorObject, "EVP_PKEY assignment error" ); goto error; }
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "cannot use this type of key" ); goto error; }
   }

	result = X509_CRL_verify(self->crl,pkey);

   return Py_BuildValue("i", result);

error:

   if (pkey)
      EVP_PKEY_free(pkey);

   return NULL;

}

static char x509_crl_object_pem_write__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>pemWrite</name>
   </header>
   <body>
      <para>
         This method returns a PEM encoded CRL as a
         string.
      </para>
   </body>
</method>";


static PyObject *
x509_crl_object_pem_write(x509_crl_object *self, PyObject *args)
{
   int len=0, ret=0;
   char *buf=NULL;
   BIO *out_bio=NULL;
   PyObject *cert=NULL;
   
	if (!PyArg_ParseTuple(args, ""))
		return NULL;

   out_bio = BIO_new(BIO_s_mem());

   if (!PEM_write_bio_X509_CRL(out_bio, self->crl) )
      { PyErr_SetString( SSLErrorObject, "unable to write certificate" ); goto error; }

   if ( !(len = BIO_ctrl_pending(out_bio) ) )
      { PyErr_SetString( SSLErrorObject, "unable to get bytes stored in bio" ); goto error; }

   if ( !(buf = malloc(len) ) )
      { PyErr_SetString( SSLErrorObject, "unable to allocate memory" ); goto error; }

   if ( (ret = BIO_read( out_bio, buf, len ) ) != len )
      { PyErr_SetString( SSLErrorObject, "unable to write out cert" ); goto error; }

   cert = Py_BuildValue("s#", buf, len);

   BIO_free(out_bio);
   free(buf);
   return cert;
   
error:   

   if (out_bio)
      BIO_free(out_bio);

   if (buf)
      free(buf);

   return NULL;
}

static char x509_crl_object_pprint__doc__[] = "
<method>
   <header>
      <memberof>X509Crl</memberof>
      <name>pprint</name>
   </header>
   <body>
      <para>
         This method returns a formatted string showing the information
         held in the CRL.
      </para>
   </body>
</method>";


static PyObject *
x509_crl_object_pprint(x509_crl_object *self, PyObject *args)
{
   int len=0, ret=0;
   char *buf=NULL;
   BIO *out_bio=NULL;
   PyObject *crl=NULL;
   
	if (!PyArg_ParseTuple(args, ""))
		return NULL;

   out_bio = BIO_new(BIO_s_mem());

   if (!X509_CRL_print(out_bio, self->crl) )
      { PyErr_SetString( SSLErrorObject, "unable to write crl" ); goto error; }

   if ( !(len = BIO_ctrl_pending(out_bio) ) )
      { PyErr_SetString( SSLErrorObject, "unable to get bytes stored in bio" ); goto error; }

   if ( !(buf = malloc(len) ) )
      { PyErr_SetString( SSLErrorObject, "unable to allocate memory" ); goto error; }

   if ( (ret = BIO_read( out_bio, buf, len ) ) != len )
      { PyErr_SetString( SSLErrorObject, "unable to write out cert" ); goto error; }

   crl = Py_BuildValue("s#", buf, len);

   BIO_free(out_bio);
   free(buf);
   return crl;
   
error:   

   if (out_bio)
      BIO_free(out_bio);

   if (buf)
      free(buf);

   return NULL;

}

static struct PyMethodDef x509_crl_object_methods[] = {
   {"sign",          (PyCFunction)x509_crl_object_sign,  
                           METH_VARARGS,  x509_crl_object_sign__doc__}, 
   {"verify",          (PyCFunction)x509_crl_object_verify,  
                           METH_VARARGS,  x509_crl_object_verify__doc__}, 
   {"getVersion",    (PyCFunction)x509_crl_object_get_version,  
                           METH_VARARGS,  x509_crl_object_get_version__doc__}, 
   {"setVersion",    (PyCFunction)x509_crl_object_set_version,    
                           METH_VARARGS,  x509_crl_object_set_version__doc__}, 
   {"getIssuer",     (PyCFunction)x509_crl_object_get_issuer,  
                           METH_VARARGS,  x509_crl_object_get_issuer__doc__}, 
   {"setIssuer",     (PyCFunction)x509_crl_object_set_issuer,   
                           METH_VARARGS,  x509_crl_object_set_issuer__doc__}, 
   {"getThisUpdate",  (PyCFunction)x509_crl_object_get_this_update,  
                           METH_VARARGS,  x509_crl_object_get_this_update__doc__}, 
   {"setThisUpdate",   (PyCFunction)x509_crl_object_set_this_update,  
                           METH_VARARGS,  x509_crl_object_set_this_update__doc__}, 
   {"getNextUpdate",   (PyCFunction)x509_crl_object_get_next_update,   
                           METH_VARARGS,  x509_crl_object_get_next_update__doc__}, 
   {"setNextUpdate",  (PyCFunction)x509_crl_object_set_next_update,  
                           METH_VARARGS,  x509_crl_object_set_next_update__doc__}, 
   {"setRevoked",  (PyCFunction)x509_crl_object_set_revoked,  
                           METH_VARARGS,  x509_crl_object_set_revoked__doc__}, 
   {"getRevoked",  (PyCFunction)x509_crl_object_get_revoked,  
                           METH_VARARGS,  x509_crl_object_get_revoked__doc__}, 
   {"pemWrite",  (PyCFunction)x509_crl_object_pem_write,  
                           METH_VARARGS,  x509_crl_object_pem_write__doc__}, 
   {"pprint",        (PyCFunction)x509_crl_object_pprint,     
                           METH_VARARGS,  x509_crl_object_pprint__doc__}, 
 
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
x509_crl_object_getattr(x509_crl_object *self, char *name)
{
	return Py_FindMethod(x509_crl_object_methods, (PyObject *)self, name);
}

static void
x509_crl_object_dealloc(x509_crl_object *self, char *name)
{
   X509_CRL_free( self->crl );
   PyObject_Del(self);
}

static char x509_crltype__doc__[] = "
<class>
   <header>
      <name>X509Crl</name>
   </header>
   <body>
      <para>
         This class provides access to OpenSSL X509 CRL management
         facilities.
      </para>
   </body>
</class>";


static PyTypeObject x509_crltype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				                           /*ob_size*/
	"X509Crl",		                        /*tp_name*/
	sizeof(x509_crl_object),	            /*tp_basicsize*/
	0,				                           /*tp_itemsize*/
	(destructor)x509_crl_object_dealloc,	/*tp_dealloc*/
	(printfunc)0,		                     /*tp_print*/
	(getattrfunc)x509_crl_object_getattr,	/*tp_getattr*/
	(setattrfunc)0,	                     /*tp_setattr*/
	(cmpfunc)0,		                        /*tp_compare*/
	(reprfunc)0,      		               /*tp_repr*/
	0,			                              /*tp_as_number*/
	0,		                                 /*tp_as_sequence*/
	0,		                                 /*tp_as_mapping*/
	(hashfunc)0,		                     /*tp_hash*/
	(ternaryfunc)0,		                  /*tp_call*/
	(reprfunc)0,		                     /*tp_str*/
	0,
   0,
   0,
   0,
	x509_crltype__doc__                   /* Documentation string */
};
/*========== x509 crl Code ==========*/

/*========== revoked Code ==========*/
x509_revoked_object* x509_revoked_object_new(void)
{
   x509_revoked_object *self=NULL;

   if ( !(self = PyObject_New( x509_revoked_object, &x509_revokedtype ) ) )
      goto error;

   self->revoked = X509_REVOKED_new();

   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

static char x509_revoked_object_set_serial__doc__[] = "
<method>
   <header>
      <memberof>X509Revoked</memberof>
      <name>setSerial</name>
      <parameter>serial</parameter>
   </header>
   <body>
      <para>
         This method sets the serial number in the serial field of
         this object.  <parameter>serial</parameter> should be an
         integer.
      </para>
   </body>
</method>";

static PyObject *
x509_revoked_object_set_serial(x509_revoked_object *self, PyObject *args)
{
   int serial=0;

	if (!PyArg_ParseTuple(args, "i", &serial))
		goto error;

   if (!ASN1_INTEGER_set( self->revoked->serialNumber, serial ) )
      { PyErr_SetString( SSLErrorObject, "unable to set serial number" ); goto error; }

   return Py_BuildValue("");

error:

   return NULL;
}

static char x509_revoked_object_get_serial__doc__[] = " 
<method>
   <header>
      <memberof>X509Revoked</memberof>
      <name>getSerial</name>
   </header>
   <body>
      <para>
         This method gets the serial number in the serial field of
         this object.
      </para>
   </body>
</method>";

static PyObject *
x509_revoked_object_get_serial(x509_revoked_object *self, PyObject *args)
{
   int serial=0;

	if (!PyArg_ParseTuple(args, ""))
		goto error;

   if ( (serial = ASN1_INTEGER_get( self->revoked->serialNumber ) ) == -1 )
      { PyErr_SetString( SSLErrorObject, "unable to get serial number" ); goto error; }

   return Py_BuildValue("i", serial);

error:

   return NULL;
}

static char x509_revoked_object_get_date__doc__[] = "
<method>
   <header>
      <memberof>X509Revoked</memberof>
      <name>getDate</name>
   </header>
   <body>
      <para>
         This method returns a tuple containing two integers representing
         <constant>revocationDate</constant>. The first
         number represents the time in seconds and is the same as the C 
         <constant>time_t</constant> typedef and the second represents the time zone offset in
         seconds.
      </para>
   </body>
</method>";


static PyObject *
x509_revoked_object_get_date(x509_revoked_object *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ""))
		goto error;

   return helper_get_date( self->revoked->revocationDate );

error:

   return NULL;
}

static char x509_revoked_object_set_date__doc__[] = "
<method>
   <header>
      <memberof>X509Revoked</memberof>
      <name>setDate</name>
      <parameter>time</parameter>
   </header>
   <body>
      <para>
         This method sets the <constant>revocationDate</constant>
         field of this object.  <parameter>time</parameter> should be a time in seconds,
         as generated by the <function>time</function> function in the Python
         Standard Library.
      </para>
   </body>
</method>";


static PyObject *
x509_revoked_object_set_date(x509_revoked_object *self, PyObject *args)
{
   int time=0;

	if (!PyArg_ParseTuple(args, "i", &time))
		goto error;

   if (!ASN1_UTCTIME_set( self->revoked->revocationDate, time ))
      { PyErr_SetString( PyExc_TypeError, "could not set revocationDate" ); goto error; }

   return Py_BuildValue("");

error:

   return NULL;
}

static struct PyMethodDef x509_revoked_object_methods[] = {
   {"getSerial",        (PyCFunction)x509_revoked_object_get_serial,  
                           METH_VARARGS,  x509_revoked_object_get_serial__doc__}, 
   {"setSerial",        (PyCFunction)x509_revoked_object_set_serial,  
                           METH_VARARGS,  x509_revoked_object_set_serial__doc__}, 
   {"getDate",        (PyCFunction)x509_revoked_object_get_date,  
                           METH_VARARGS,  x509_revoked_object_get_date__doc__}, 
   {"setDate",        (PyCFunction)x509_revoked_object_set_date,  
                           METH_VARARGS,  x509_revoked_object_set_date__doc__}, 
 
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
x509_revoked_object_getattr(x509_revoked_object *self, char *name)
{
	return Py_FindMethod(x509_revoked_object_methods, (PyObject *)self, name);
}

static void
x509_revoked_object_dealloc(x509_revoked_object *self, char *name)
{
   X509_REVOKED_free( self->revoked );
   PyObject_Del(self);
}

static char x509_revokedtype__doc__[] = "
<class>
   <header>
      <name>X509Revoked</name>
   </header>
   <body>
      <para>
         This class provides a container for details of a revoked
         certificate.  It normally would only be used in association with
         a CRL, its not much use by itself.  Indeed the only reason this
         class exists is because in the future POW is likely to be extended
         to support extensions for certificates, CRLs and revocations.
         <classname>X509Revoked</classname> existing as an object in its
         own right will make adding this support easier, while avoiding
         backwards compatibility issues.
      </para>
   </body>
</class>";


static PyTypeObject x509_revokedtype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				                              /*ob_size*/
	"X509Revoked",		                        /*tp_name*/
	sizeof(x509_revoked_object),	            /*tp_basicsize*/
	0,				                              /*tp_itemsize*/
	(destructor)x509_revoked_object_dealloc,	/*tp_dealloc*/
	(printfunc)0,		                        /*tp_print*/
	(getattrfunc)x509_revoked_object_getattr,	/*tp_getattr*/
	(setattrfunc)0,	                        /*tp_setattr*/
	(cmpfunc)0,		                           /*tp_compare*/
	(reprfunc)0,      		                  /*tp_repr*/
	0,			                                 /*tp_as_number*/
	0,		                                    /*tp_as_sequence*/
	0,		                                    /*tp_as_mapping*/
	(hashfunc)0,		                        /*tp_hash*/
	(ternaryfunc)0,		                     /*tp_call*/
	(reprfunc)0,		                        /*tp_str*/
	0,
   0,
   0,
   0,
	x509_revokedtype__doc__                  /* Documentation string */
};
/*========== x509 revoked Code ==========*/

/*========== ssl Code ==========*/
static char ssl_object_use_certificate__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>useCertificate</name>
      <parameter>cert</parameter>
   </header>
   <body>
      <para>
         The parameter <parameter>cert</parameter> must be an
         instance of the <classname>X590</classname> class and must be
         called before <function>setFd</function>.  
      </para>
   </body>
</method>";

static PyObject *
ssl_object_use_certificate(ssl_object *self, PyObject *args)
{
   x509_object *x509=NULL;

	if (!PyArg_ParseTuple(args, "O!", &x509type, &x509))
		goto error;

   if (self->ctxset)
      { PyErr_SetString( SSLErrorObject, "cannont be called after setFd()" ); goto error; }

   if ( !SSL_CTX_use_certificate(self->ctx, x509->x509) )
      { PyErr_SetString( SSLErrorObject, "could not use certificate" ); goto error; }

   return Py_BuildValue("");

error:

   return NULL;
}

static char ssl_object_use_key__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>useKey</name>
      <parameter>key</parameter>
   </header>
   <body>
      <para>
         The parameter <parameter>key</parameter> must be an
         instance of the <classname>Asymmetric</classname> class and
         must contain the private key.  This function cannot be called 
         after <function>useKey</function>.
      </para>
   </body>
</method>";

static PyObject *
ssl_object_use_key(ssl_object *self, PyObject *args)
{
   asymmetric_object *asym=NULL;
	EVP_PKEY *pkey=NULL;

	if (!PyArg_ParseTuple(args, "O!", &asymmetrictype, &asym))
		goto error;

   if (self->ctxset)
      { PyErr_SetString( SSLErrorObject, "cannont be called after setFd()" ); goto error; }

   if ( !(pkey = EVP_PKEY_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   switch( asym->key_type )
   {
      case RSA_PRIVATE_KEY:
      {
         if ( !EVP_PKEY_assign_RSA(pkey, asym->cipher) )
            { PyErr_SetString( SSLErrorObject, "EVP_PKEY assignment error" ); goto error; }
         break;
      }
      case RSA_PUBLIC_PRIVATE_KEY:
      {
         if ( !EVP_PKEY_assign_RSA(pkey, asym->cipher) )
            { PyErr_SetString( SSLErrorObject, "EVP_PKEY assignment error" ); goto error; }
         break;
      }

      default:
         { PyErr_SetString( SSLErrorObject, "cannot use this type of key" ); goto error; }
   }

	if ( !SSL_CTX_use_PrivateKey(self->ctx, pkey) )
      { PyErr_SetString( SSLErrorObject, "ctx key assignment error" ); goto error; }

   return Py_BuildValue("");

error:

   if(pkey)
      EVP_PKEY_free(pkey);

   return NULL;
}

static char ssl_object_check_key__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>checkKey</name>
   </header>
   <body>
      <para>
         This simple method will return 1 if the public key, contained in
         the X509 certificate this <classname>Ssl</classname> instance is using,
         matches the private key this <classname>Ssl</classname> instance is using.
         Otherwise it will return 0.
      </para>
   </body>
</method>";

static PyObject *
ssl_object_check_key(ssl_object *self, PyObject *args)
{
   if ( SSL_CTX_check_private_key(self->ctx) )
      return Py_BuildValue("i", 1);
   else
      return Py_BuildValue("i", 0);
}

static char ssl_object_set_fd__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>setFd</name>
      <parameter>descriptor</parameter>
   </header>
   <body>
      <para>
         This function is used to associate a file descriptor with a
         <classname>Ssl</classname> object.  The file descriptor should
         belong to an open TCP connection.  Once this function has
         been called, calling <function>useKey</function> or
         <function>useCertificate</function> will, fail rasing exceptions.
      </para>
   </body>
</method>";

static PyObject *
ssl_object_set_fd(ssl_object *self, PyObject *args)
{
   int fd=0, self_index=0;
   
	if (!PyArg_ParseTuple(args, "i", &fd))
		goto error;

   if ( !(self->ssl = SSL_new( self->ctx ) ) )
      { PyErr_SetString( SSLErrorObject, "unable to create ssl sturcture" ); goto error; }

   if ( !SSL_set_fd( self->ssl, fd ) )
      { PyErr_SetString( SSLErrorObject, "unable to set file descriptor" ); goto error; }

   if ( (self_index = SSL_get_ex_new_index(0, "self_index", NULL, NULL, NULL) ) != -1 )
      SSL_set_ex_data(self->ssl, self_index, self);
   else
      { PyErr_SetString( SSLErrorObject, "unable to create ex data index" ); goto error; }

   self->ctxset = 1;

   return Py_BuildValue("");

error:

   return NULL;
}

static char ssl_object_accept__doc__[] = " 
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>accept</name>
   </header>
   <body>
      <para>
         This function will attempt the SSL level accept with a
         client.  The <classname>Ssl</classname> object must have been
         created using a <constant>XXXXX_SERVER_METHOD</constant> or
         a <constant>XXXXX_METHOD</constant> and this function should only be
         called after <function>useKey</function>,
         <function>useCertificate</function> and
         <function>setFd</function> functions have been called.
      </para>

      <example>
         <title><function>accept</function> function usage</title>
         <programlisting>
      keyFile = open( 'test/private.key', 'r' )
      certFile = open( 'test/cacert.pem', 'r' )

      rsa = POW.pemRead( POW.RSA_PRIVATE_KEY, keyFile.read(), 'pass' )
      x509 = POW.pemRead( POW.X509_CERTIFICATE, certFile.read() )

      keyFile.close()
      certFile.close()

      sl = POW.Ssl( POW.SSLV23_SERVER_METHOD )
      sl.useCertificate( x509 )
      sl.useKey( rsa )

      s = socket.socket( socket.AF_INET, socket.SOCK_STREAM )
      s.bind( ('localhost', 1111) )
      s.listen(5)
      s2, addr = s.accept()

      s.close()

      sl.setFd( s2.fileno() )
      sl.accept()
      print sl.read(1024)
      sl.write('Message from server to client...')

      s2.close()     
         </programlisting>
      </example>
   </body>
</method>";

static PyObject *
ssl_object_accept(ssl_object *self, PyObject *args)
{
   int ret=0, err=0;
   
	if (!PyArg_ParseTuple(args, ""))
		goto error;
   
   ret = SSL_accept( self->ssl );
   if (ret <= 0) 
   {
      err = SSL_get_error( self->ssl, ret );
      PyErr_SetObject(SSLErrorObject, ssl_err_factory( err ) );
      goto error;
   }
   return Py_BuildValue("");

error:

   return NULL;
}

static char ssl_object_connect__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>connect</name>
   </header>
   <body>
      <para>
         This function will attempt the SSL level connection with a
         server.  The <classname>Ssl</classname> object must have been
         created using a <constant>XXXXX_CLIENT_METHOD</constant> or
         a <constant>XXXXX_METHOD</constant> and this function should only be
         called after <function>setFd</function> has already been
         called.
      </para>

      <example>
         <title><function>connect</function> function usage</title>
         <programlisting>
      s = socket.socket( socket.AF_INET, socket.SOCK_STREAM )
      s.connect(('localhost', 1111))

      sl = POW.Ssl( POW.SSLV23_CLIENT_METHOD )
      sl.setFd( s.fileno() )
      sl.connect()
      sl.write('Message from client to server...')
      print sl.read(1024)
         </programlisting>
      </example>
   </body>
</method>";

static PyObject *
ssl_object_connect(ssl_object *self, PyObject *args)
{
   int ret, err=0;
   
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
   
   ret = SSL_connect( self->ssl );
   if (ret <= 0) 
   {
      err = SSL_get_error( self->ssl, ret );
      PyErr_SetObject(SSLErrorObject, ssl_err_factory( err ) );
      return NULL;
   }
   return Py_BuildValue("");
}

static char ssl_object_write__doc__[] = " 
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>write</name>
      <parameter>string</parameter>
   </header>
   <body>
      <para>
         This method writes the <parameter>string</parameter> to the
         <classname>Ssl</classname> object, to be read by it's peer.  This
         function is analogous to the <classname>socket</classname>
         classes <function>write</function> function.
      </para>
   </body>
</method>";

static PyObject *
ssl_object_write(ssl_object *self, PyObject *args)
{
   char *msg;
   int length=0, ret=0, err=0;
   
	if (!PyArg_ParseTuple(args, "s#", &msg, &length))
		goto error;
   
   ret = SSL_write( self->ssl, msg, length );
   if (ret <= 0) 
   {
      err = SSL_get_error( self->ssl, ret );
      PyErr_SetObject(SSLErrorObject, ssl_err_factory( err ) );
      goto error;
   }
   return Py_BuildValue("i", ret);

error:

   return NULL;
}

static char ssl_object_read__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>read</name>
      <parameter>amount=1024</parameter>
   </header>
   <body>
      <para>
         This method reads up to <parameter>amount</parameter> characters from the
         <classname>Ssl</classname> object.  This
         function is analogous to the <classname>socket</classname>
         classes <function>read</function> function.
      </para>
   </body>
</method>";

static PyObject *
ssl_object_read(ssl_object *self, PyObject *args)
{
   PyObject *data;
   char *msg;
   int len = 1024;
   int ret;
   int err;
   
	if (!PyArg_ParseTuple(args, "|i", &len))
		return NULL;

   if ( !(msg = malloc(len) ) )
      { PyErr_SetString( SSLErrorObject, "unable to allocate memory" ); goto error; }

   ret = SSL_read( self->ssl, msg, len );

   if (ret <= 0) 
   {
      free(msg);
      err = SSL_get_error( self->ssl, ret );
      PyErr_SetObject(SSLErrorObject, ssl_err_factory( err ) );
      return NULL;
   }
   else
      data = Py_BuildValue("s#", msg, ret);

   free(msg);
   return data;

error:

   if (msg)
      free(msg);

   return NULL;
}

static char ssl_object_peer_certificate__doc__[] = " 
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>peerCertificate</name>
   </header>
   <body>
      <para>
         This method returns any peer certificate presented in the initial
         SSL negotiation or <constant>None</constant>.  If a certificate is
         returned, it will be an instance of <classname>X509</classname>.
      </para>
   </body>
</method>";


static PyObject *
ssl_object_peer_certificate(ssl_object *self, PyObject *args)
{
   X509 *x509=NULL;
   x509_object *x509_obj=NULL;

	if (!PyArg_ParseTuple(args, ""))
		goto error;

   if ( !(x509_obj = X509_object_new() ) )
      { PyErr_SetString( SSLErrorObject, "could not create x509 object" ); goto error; }

   x509 = SSL_get_peer_certificate( self->ssl );

   if (x509)
   {
      X509_free( x509_obj->x509 ); 

      if ( !(x509_obj->x509 = x509 ) )
         { PyErr_SetString( SSLErrorObject, "could not create x509 object" ); goto error; }
      return Py_BuildValue("O", x509_obj);
   }
   else
   {
      Py_XDECREF( x509_obj );
      return Py_BuildValue("");
   }

error:

   if (x509)
      X509_free(x509);

   Py_XDECREF( x509_obj );
   return NULL;
}

static char ssl_object_clear__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>clear</name>
   </header>
   <body>
      <para>
         This method will clear the SSL session ready for
         a new SSL connection.  It will not effect the underlying socket.
      </para>
   </body>
</method>";

static PyObject *
ssl_object_clear(ssl_object *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ""))
		goto error;
   
   if (!SSL_clear( self->ssl ) )
      { PyErr_SetString( SSLErrorObject, "failed to clear ssl connection" ); goto error; }

   return Py_BuildValue("");

error:

   return NULL;
}

static char ssl_object_shutdown__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>shutdown</name>
   </header>
   <body>
      <para>
         This method will issue a <constant>shutdown</constant> signal to it's peer. 
         If this connection's peer has already initiated a shutdown this call
         will succeed, otherwise it will raise and exception.  In order to
         check the shutdown handshake was successful,
         <function>shutdown</function> must be called again.  If no
         exception is raised, the handshake is complete.  
      </para>
      <para>
         The odd
         implementation of this function reflects the underlying OpenSSL
         function, which reflects the SSL protocol.  Although rasing an
         exception is a bit annoying, the alternative, returning true all
         false will not tell you why the call failed and the exception
         will, at least that is the theory.  Look up the exact meaning
         of the exceptions in the OpenSSL man page SSL_get_error.
      </para>
   </body>
</method>";

static PyObject *
ssl_object_shutdown(ssl_object *self, PyObject *args)
{
   int ret=0, err=0;

	if (!PyArg_ParseTuple(args, ""))
		goto error;
   
   ret = SSL_shutdown(self->ssl);

   if (ret <= 0) 
   {
      err = SSL_get_error( self->ssl, ret );
      PyErr_SetObject(SSLErrorObject, ssl_err_factory( err ) );
      return NULL;
   }

   return Py_BuildValue("");

error:

   return NULL;
}

static char ssl_object_get_shutdown__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>getShutdown</name>
   </header>
   <body>
      <para>
         This function returns an integer indicating the state of the
         SSL connection. <constant>SSL_RECIEVED_SHUTDOWN</constant>
         will be set the if it's peer sends a <constant>shutdown</constant>
         signal or the underlying socket
         receives a close notify .  The possible values are:
      </para>
      <simplelist>
         <member><constant>SSL_NO_SHUTDOWN</constant></member>
         <member><constant>SSL_SENT_SHUTDOWN</constant></member>
         <member><constant>SSL_RECIEVED_SHUTDOWN</constant></member>
         <member><constant>SSL_SENT_SHUTDOWN</constant> | <constant>SSL_RECIEVED_SHUTDOWN</constant></member>
      </simplelist>
   </body>
</method>";

static PyObject *
ssl_object_get_shutdown(ssl_object *self, PyObject *args)
{
   int state=0;

	if (!PyArg_ParseTuple(args, ""))
		goto error;
   
   state = SSL_get_shutdown(self->ssl);

   return Py_BuildValue("i", state);

error:

   return NULL;
}

static char ssl_object_get_ciphers__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>getCiphers</name>
   </header>
   <body>
      <para>
         This function returns a list of available ciphers ordered from
         most favoured to least.  This function must be called after
         <function>setFd</function>.  
      </para>
   </body>
</method>";

static PyObject *
ssl_object_get_ciphers(ssl_object *self, PyObject *args)
{
   int inlist=0, i=0;
   const char *cipher=NULL;
   PyObject *list=NULL, *name=NULL;

	if (!PyArg_ParseTuple(args, ""))
		goto error;

   if (!self->ctxset)
      { PyErr_SetString( SSLErrorObject, "cannont be called before setFd()" ); goto error; }

   list = PyList_New(0);
   
   cipher = SSL_get_cipher_list(self->ssl, 0);
   while (cipher)
   {
      if ( !(name = PyString_FromString(cipher) ) )
         goto error;
      if ( PyList_Append( list, name ) != 0)
         goto error;
      cipher = SSL_get_cipher_list(self->ssl, ++i);
   }
   return Py_BuildValue("O", list);

error:

   if (list)
   {
      inlist = PyList_Size( list );
      for (i=0; i < inlist; i++)
      {
         name = PyList_GetItem( list, i );
         Py_DECREF(name);
      }
      Py_DECREF(list);
   }

   return NULL;
}

static char ssl_object_set_ciphers__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>setCiphers</name>
      <parameter>ciphers</parameter>
   </header>
   <body>
      <para>
         <function>setCiphers</function>
         can help protect against certain types of attacks which try to
         coerce the server, client or both to negotiate a weak cipher.          
         <parameter>ciphers</parameter> should be a list of strings, as
         produced by <function>getCiphers</function> and described in the
         OpenSSL man page ciphers.   <function>setCiphers</function> should
         only be called after <function>setFd</function>.
      </para>
   </body>
</method>";

static PyObject *
ssl_object_set_ciphers(ssl_object *self, PyObject *args)
{
   PyObject *ciphers=NULL;
   PyObject *cipher=NULL;
   int size=0, cipherstrlen=0, nextstrlen=0, i=0;
   char *cipherstr=NULL;

	if (!PyArg_ParseTuple(args, "O", &ciphers))
		goto error;

   if ( !(PyList_Check(ciphers) || PyTuple_Check(ciphers)) )
      { PyErr_SetString( PyExc_TypeError, "inapropriate type" ); goto error; }

   if (!self->ctxset)
      { PyErr_SetString( SSLErrorObject, "cannont be called before setFd()" ); goto error; }

   cipherstr = malloc(8);  //very bogus, realloc dosn't work with out some
                           //previously allocated memory! Really should.
   memset(cipherstr, 0, 8);
   size = PySequence_Size(ciphers);
   for (i=0; i < size; i++)
   {
      if ( !( cipher = PySequence_GetItem( ciphers, i ) ) )
         goto error;

      if ( !PyString_Check(cipher) )
         { PyErr_SetString( PyExc_TypeError, "inapropriate type" ); goto error; }

      cipherstrlen = strlen(cipherstr);
      nextstrlen = strlen( PyString_AsString(cipher) );

      if ( !(cipherstr = realloc( cipherstr, cipherstrlen + nextstrlen + 2)) )
         { PyErr_SetString( PyExc_TypeError, "could allocate memory" ); goto error; }

      if (cipherstrlen)
         strcat( cipherstr, ":\0" );

      strcat( cipherstr, PyString_AsString(cipher) );
   }
   SSL_set_cipher_list( self->ssl, cipherstr );
   free(cipherstr);
   Py_DECREF(cipher);
   return Py_BuildValue("");

error:

   if (cipherstr)
      free(cipherstr);

   Py_XDECREF(cipher);

   return NULL;
}

static char ssl_object_get_cipher__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>getCipher</name>
   </header>
   <body>
      <para>
         This function returns the current cipher in use.
      </para>
   </body>
</method>";

static PyObject *
ssl_object_get_cipher(ssl_object *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ""))
		goto error;

   if (!self->ctxset)
      { PyErr_SetString( SSLErrorObject, "cannont be called before setFd()" ); goto error; }
   
   return Py_BuildValue("s", SSL_get_cipher( self->ssl ));

error:

   return NULL;
}

static int stub_callback(int preverify_ok, X509_STORE_CTX *ctx)
{
   return 1;
}

static char ssl_object_set_verify_mode__doc__[] = "
<method>
   <header>
      <memberof>Ssl</memberof>
      <name>setVerifyMode</name>
      <parameter>mode</parameter>
   </header>
   <body>
      <para>
         This function sets the behaviour of the SSL handshake.  The
         parameter <parameter>mode</parameter> should be one of the
         following:
      </para>
      <simplelist>
         <member><constant>SSL_VERIFY_NONE</constant></member>
         <member><constant>SSL_VERIFY_PEER</constant></member>
      </simplelist>
      <para>
         See the OpenSSL man page <function>SSL_CTX_set_verify</function> 
         for details.  This function must be called after <function>setfd</function> 
         has been called.
      </para>
   </body>
</method>";

static PyObject *
ssl_object_set_verify_mode(ssl_object *self, PyObject *args)
{
   int mode=0;

	if (!PyArg_ParseTuple(args, "i", &mode))
		goto error;

   if (self->ctxset)
      { PyErr_SetString( SSLErrorObject, "cannont be called after setfd()" ); goto error; }

   SSL_CTX_set_verify( self->ctx, mode, stub_callback );

   return Py_BuildValue("");

error:

   return NULL;
}

static struct PyMethodDef ssl_object_methods[] = {
	{"useCertificate",   (PyCFunction)ssl_object_use_certificate,	   
                           METH_VARARGS,	ssl_object_use_certificate__doc__},
	{"useKey",           (PyCFunction)ssl_object_use_key,	   
                           METH_VARARGS,	ssl_object_use_key__doc__},
	{"checkKey",         (PyCFunction)ssl_object_check_key,	   
                           METH_VARARGS,	ssl_object_check_key__doc__},
	{"setFd",	         (PyCFunction)ssl_object_set_fd,	   
                           METH_VARARGS,	ssl_object_set_fd__doc__},
	{"connect",	         (PyCFunction)ssl_object_connect,	   
                           METH_VARARGS,	ssl_object_connect__doc__},
	{"accept",	         (PyCFunction)ssl_object_accept,	   
                           METH_VARARGS,	ssl_object_accept__doc__},
	{"write",	         (PyCFunction)ssl_object_write,	   
                           METH_VARARGS,	ssl_object_write__doc__},
	{"read",	            (PyCFunction)ssl_object_read,	      
                           METH_VARARGS,	ssl_object_read__doc__},
	{"peerCertificate",  (PyCFunction)ssl_object_peer_certificate,	   
                           METH_VARARGS,	ssl_object_peer_certificate__doc__},
	{"clear",	         (PyCFunction)ssl_object_clear,	      
                           METH_VARARGS,	ssl_object_clear__doc__},
	{"shutdown",	      (PyCFunction)ssl_object_shutdown,	      
                           METH_VARARGS,	ssl_object_shutdown__doc__},
	{"getShutdown",	   (PyCFunction)ssl_object_get_shutdown,	      
                           METH_VARARGS,	ssl_object_get_shutdown__doc__},
	{"getCiphers",	      (PyCFunction)ssl_object_get_ciphers,	      
                           METH_VARARGS,	ssl_object_get_ciphers__doc__},
	{"setCiphers",	      (PyCFunction)ssl_object_set_ciphers,	      
                           METH_VARARGS,	ssl_object_set_ciphers__doc__},
	{"getCipher",	      (PyCFunction)ssl_object_get_cipher,	      
                           METH_VARARGS,	ssl_object_get_cipher__doc__},
	{"setVerifyMode",	   (PyCFunction)ssl_object_set_verify_mode,	      
                           METH_VARARGS,	ssl_object_set_verify_mode__doc__},
 
	{NULL,		NULL}		/* sentinel */
};

static ssl_object *
newssl_object(int type)
{
	ssl_object *self;
   SSL_METHOD *method;

	
	if ( !(self = PyObject_NEW(ssl_object, &ssltype) ) )
		goto error;

   self->ctxset = 0;
   self->ssl = NULL;

   switch(type)
   {
      case SSLV2_SERVER_METHOD:  method = SSLv2_server_method();   break;
      case SSLV2_CLIENT_METHOD:  method = SSLv2_client_method();   break;
      case SSLV2_METHOD:         method = SSLv2_method();          break;
      case SSLV3_SERVER_METHOD:  method = SSLv3_server_method();   break;
      case SSLV3_CLIENT_METHOD:  method = SSLv3_client_method();   break;
      case SSLV3_METHOD:         method = SSLv3_method();          break;
      case TLSV1_SERVER_METHOD:  method = TLSv1_server_method();   break;
      case TLSV1_CLIENT_METHOD:  method = TLSv1_client_method();   break;
      case TLSV1_METHOD:         method = TLSv1_method();          break;
      case SSLV23_SERVER_METHOD: method = SSLv23_server_method();  break;
      case SSLV23_CLIENT_METHOD: method = SSLv23_client_method();  break;
      case SSLV23_METHOD:        method = SSLv23_method();         break;
                                                                     
      default:    
         { PyErr_SetString( SSLErrorObject, "unkown ctx method" ); goto error; }
                   
   } 

   if ( !(self->ctx = SSL_CTX_new( method ) ) )
      { PyErr_SetString( SSLErrorObject, "unable to create new ctx" ); goto error; }

	return self;

error:

   Py_XDECREF( self );
   return NULL;
}

static PyObject *
ssl_object_getattr(ssl_object *self, char *name)
{
	return Py_FindMethod(ssl_object_methods, (PyObject *)self, name);
}

static void
ssl_object_dealloc(ssl_object *self)
{
   SSL_free( self->ssl );
   SSL_CTX_free( self->ctx );
   PyObject_Del(self);
}

static char ssltype__doc__[] = "
<class>
   <header>
      <name>Ssl</name>
   </header>
   <body>
      <para>
         This class provides access to the Secure Socket Layer
         functionality of OpenSSL.  It is designed to be a simple as
         possible to use and is not designed for high performance
         applications which handle many simultaneous connections.  The
         original motivation for writing this library was to provide a
         security layer for network agents written in Python, for this
         application, good performance with multiple concurrent connections
         is not an issue. 
      </para>
   </body>
</class>";

static PyTypeObject ssltype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				                     /*ob_size*/
	"Ssl",			                  /*tp_name*/
	sizeof(ssl_object),               /*tp_basicsize*/
	0,				                     /*tp_itemsize*/
	(destructor)ssl_object_dealloc,	/*tp_dealloc*/
	(printfunc)0,		               /*tp_print*/
	(getattrfunc)ssl_object_getattr,	/*tp_getattr*/
	(setattrfunc)0,	               /*tp_setattr*/
	(cmpfunc)0,		                  /*tp_compare*/
	(reprfunc)0,	      	         /*tp_repr*/
	0,			                        /*tp_as_number*/
	0,		                           /*tp_as_sequence*/
	0,		                           /*tp_as_mapping*/
	(hashfunc)0,		               /*tp_hash*/
	(ternaryfunc)0,		            /*tp_call*/
	(reprfunc)0,		               /*tp_str*/
	0,
   0,
   0,
   0,
	ssltype__doc__                   /* Documentation string */
};
/*========== ssl Object ==========*/

/*========== asymmetric Object ==========*/
static asymmetric_object *
asymmetric_object_new(int cipher_type, int key_size)
{
   asymmetric_object *self=NULL;

   self = PyObject_New( asymmetric_object, &asymmetrictype );
   if (self == NULL)
      goto error;

   switch(cipher_type)
   {
      case RSA_CIPHER: 
      {
         if ( !(self->cipher = RSA_generate_key(key_size,RSA_F4,NULL,NULL) ) )
            {  PyErr_SetString( SSLErrorObject, "could not generate key" ); goto error; }

         self->key_type = RSA_PUBLIC_PRIVATE_KEY;
         self->cipher_type = RSA_CIPHER;
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "unsupported cipher" ); goto error; }
   }

   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

static asymmetric_object *
asymmetric_object_pem_read(int key_type, BIO *in, char *pass)
{
   asymmetric_object *self=NULL;

   self = PyObject_New( asymmetric_object, &asymmetrictype );
   if (self == NULL)
      goto error;

   switch (key_type)
   {
      case RSA_PUBLIC_KEY:
      {
         if( !(self->cipher = PEM_read_bio_RSAPublicKey( in, NULL, NULL, NULL ) ) )
            {  PyErr_SetString( SSLErrorObject, "could not load public key" ); goto error; }
         self->key_type = RSA_PUBLIC_KEY;
         self->cipher_type = RSA_CIPHER;
         break;
      }
      case RSA_PRIVATE_KEY:
      {
         if( !(self->cipher = PEM_read_bio_RSAPrivateKey( in, NULL, NULL, pass) ) )
            {  PyErr_SetString( SSLErrorObject, "could not load private key" ); goto error; }
         self->key_type = RSA_PRIVATE_KEY;
         self->cipher_type = RSA_CIPHER;
         break;
      }
      default:
         {  PyErr_SetString( SSLErrorObject, "unkown key type" ); goto error; }
   }

   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

static char asymmetric_object_pem_write__doc__[] = "
<method>
   <header>
      <memberof>Asymmetric</memberof>
      <name>pemWrite</name>
      <parameter>keytype</parameter>
      <parameter>ciphertype=None</parameter>
      <parameter>passphrase=None</parameter>
   </header>
   <body>
      <para>
         This method is used to write <classname>Asymmetric</classname>
         objects out to strings.  The first argument should be either
         <constant>RSA_PUBLIC_KEY</constant> or
         <constant>RSA_PRIVATE_KEY</constant>.  Private keys are often
         saved in encrypted files to offer extra security above access
         control mechanisms.  If the <parameter>keytype</parameter> is
         <constant>RSA_PRIVATE_KEY</constant> a
         <parameter>ciphertype</parameter> and
         <parameter>passphrase</parameter> can also be specified.  The
         <parameter>ciphertype</parameter> should be one of those listed in
         the <classname>Symmetric</classname> class section.
      </para>
   </body>
</method>";


static PyObject *
asymmetric_object_pem_write(asymmetric_object *self, PyObject *args)
{
   int key_type=0, cipher=0, len=0, ret=0;
   char *kstr=NULL, *buf=NULL;
   BIO *out_bio=NULL;
   PyObject *asymmetric=NULL;

	if (!PyArg_ParseTuple(args, "i|is", &key_type, &cipher, &kstr))
		return NULL;

   if ( !(out_bio = BIO_new(BIO_s_mem()) ) )
      { PyErr_SetString( SSLErrorObject, "unable to create new BIO" ); goto error; }

   if ( (kstr && !cipher) || (cipher && !kstr) )
      {PyErr_SetString(SSLErrorObject,"cipher type and key string must both be supplied");goto error;}


   switch( self->cipher_type )
   {
      case RSA_CIPHER:
      {
         switch( key_type )
         {
            case RSA_PRIVATE_KEY:
            {
               if (kstr && cipher)
               {
                  if (!PEM_write_bio_RSAPrivateKey(out_bio, self->cipher, evp_cipher_factory(cipher), NULL, 0, NULL, kstr) )
                     { PyErr_SetString( SSLErrorObject, "unable to write key" ); goto error; }
               }
               else
               {
                  if (!PEM_write_bio_RSAPrivateKey(out_bio, self->cipher, NULL, NULL, 0, NULL, NULL) )
                     { PyErr_SetString( SSLErrorObject, "unable to write key" ); goto error; }
               }
               break;
            }
            case RSA_PUBLIC_KEY:
            {
               if (kstr && cipher)
                  { PyErr_SetString( SSLErrorObject, "public keys should not encrypted" ); goto error; }
               else
               {
                  if (!PEM_write_bio_RSAPublicKey(out_bio, self->cipher) )
                     { PyErr_SetString( SSLErrorObject, "unable to write key" ); goto error; }
               }
               break;
            }
            default:
               { PyErr_SetString( SSLErrorObject, "unsupported key type" ); goto error; }
         }
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "unsupported cipher type" ); goto error; }
   }

   if ( !(len = BIO_ctrl_pending(out_bio) ) )
      { PyErr_SetString( SSLErrorObject, "unable to get number of bytes in bio" ); goto error; }

   if ( !(buf = malloc(len) ) )
      { PyErr_SetString( SSLErrorObject, "unable to allocate memory" ); goto error; }

   if ( (ret = BIO_read( out_bio, buf, len ) ) != len )
      { PyErr_SetString( SSLErrorObject, "unable to write out key" ); goto error; }

   asymmetric = Py_BuildValue("s#", buf, len);

   BIO_free(out_bio);
   free(buf);
   return asymmetric;

error:

   if (out_bio);
      BIO_free(out_bio);

   if (buf)
      free(buf);

   return NULL;
}

static char asymmetric_object_public_encrypt__doc__[] = "
<method>
   <header>
      <memberof>Asymmetric</memberof>
      <name>publicEncrypt</name>
      <parameter>plaintext</parameter>
   </header>
   <body>
      <para>
         This method is used to encrypt the <parameter>plaintext</parameter>
         using a public key. It should be noted; in practice this
         function would be used almost exclusively to encrypt symmetric cipher
         keys and not data since asymmetric cipher operations are very slow.
      </para>
   </body>
</method>";

static PyObject *
asymmetric_object_public_encrypt(asymmetric_object *self, PyObject *args)
{
   char *plain_text=NULL, *cipher_text=NULL;
   int len=0, size=0;
   PyObject *obj=NULL;

   switch( self->cipher_type )
   {
      case RSA_CIPHER:
      {
         if ( !(self->key_type == RSA_PUBLIC_KEY || self->key_type == RSA_PUBLIC_PRIVATE_KEY) )
            { PyErr_SetString( SSLErrorObject, "cannot perform public encryption with this key" ); goto error; }

         if (!PyArg_ParseTuple(args, "s#", &plain_text, &len))
            goto error;

         size = RSA_size(self->cipher);
         if ( len > size )
            { PyErr_SetString( SSLErrorObject, "plain text is too long" ); goto error; }

         if ( !(cipher_text = malloc( size + 16 ) ) )
            { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

         if ( (len = RSA_public_encrypt( len, plain_text, cipher_text, self->cipher, RSA_PKCS1_PADDING ) ) < 0 )
            { PyErr_SetString( SSLErrorObject, "could not encrypt plain text" ); goto error; }
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "unsupported cipher type" ); goto error; }
   }

   obj = Py_BuildValue("s#", cipher_text, len);
   free( cipher_text );
   return obj;

error:

   if (cipher_text)
      free(cipher_text);

   return NULL;
}

static char asymmetric_object_private_encrypt__doc__[] = "
<method>
   <header>
      <memberof>Asymmetric</memberof>
      <name>privateEncrypt</name>
      <parameter>plaintext</parameter>
   </header>
   <body>
      <para>
         This method is used to encrypt the <parameter>plaintext</parameter>
         using a private key. It should be noted; in practice this
         function would be used almost exclusively to encrypt symmetric cipher
         keys and not data since asymmetric cipher operations are very slow.
      </para>
   </body>
</method>";

static PyObject *
asymmetric_object_private_encrypt(asymmetric_object *self, PyObject *args)
{
   char *plain_text=NULL, *cipher_text=NULL;
   int len=0, size=0;
   PyObject *obj=NULL;

   switch( self->cipher_type )
   {
      case RSA_CIPHER:
      {
         if ( !(self->key_type == RSA_PRIVATE_KEY || self->key_type == RSA_PUBLIC_PRIVATE_KEY) ) 
            { PyErr_SetString( SSLErrorObject, "cannot perform private encryption with this key" ); goto error; }

         if (!PyArg_ParseTuple(args, "s#", &plain_text, &len) )
            goto error;

         size = RSA_size(self->cipher);
         if ( len > size )
            { PyErr_SetString( SSLErrorObject, "plain text is too long" ); goto error; }

         if ( !(cipher_text = malloc( size + 16 ) ) )
            { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

         if ( (len = RSA_private_encrypt( len, plain_text, cipher_text, self->cipher, RSA_PKCS1_PADDING ) ) < 0 )
            { PyErr_SetString( SSLErrorObject, "could not encrypt plain text" ); goto error; }
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "unsupported cipher type" ); goto error; }
   }

   obj = Py_BuildValue("s#", cipher_text, len);
   free( cipher_text );
   return obj;

error:

   if (cipher_text)
      free(cipher_text);

   return NULL;
}

static char asymmetric_object_public_decrypt__doc__[] = "
<method>
   <header>
      <memberof>Asymmetric</memberof>
      <name>publicDecrypt</name>
      <parameter>ciphertext</parameter>
   </header>
   <body>
      <para>
         This method is used to decrypt the
         <parameter>ciphertext</parameter> which has been encrypted
         using the corresponding private key and the
         <function>privateEncrypt</function> function.  
      </para>
   </body>
</method>";


static PyObject *
asymmetric_object_public_decrypt(asymmetric_object *self, PyObject *args)
{
   char *plain_text=NULL, *cipher_text=NULL;
   int len=0, size=0;
   PyObject *obj=NULL;

   switch( self->cipher_type )
   {
      case RSA_CIPHER:
      {
         if ( !(self->key_type == RSA_PUBLIC_KEY || self->key_type == RSA_PUBLIC_PRIVATE_KEY) ) 
            { PyErr_SetString( SSLErrorObject, "cannot perform public decryption with this key" ); goto error; }

         if (!PyArg_ParseTuple(args, "s#", &cipher_text, &len))
            goto error;
   
         size = RSA_size(self->cipher);
         if ( len > size )
            { PyErr_SetString( SSLErrorObject, "cipher text is too long" ); goto error; }

         if ( !(plain_text = malloc( size + 16 ) ) )
            { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

         if ( (len = RSA_public_decrypt( len, cipher_text, plain_text, self->cipher, RSA_PKCS1_PADDING ) ) < 0 )
            { PyErr_SetString( SSLErrorObject, "could not decrypt cipher text" ); goto error; }
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "unsupported cipher type" ); goto error; }
   }

   obj = Py_BuildValue("s#", plain_text, len);
   free( plain_text );
   return obj;

error:

   if (plain_text)
      free(plain_text);

   return NULL;
}

static char asymmetric_object_private_decrypt__doc__[] = "
<method>
   <header>
      <memberof>Asymmetric</memberof>
      <name>privateDecrypt</name>
      <parameter>ciphertext</parameter>
   </header>
   <body>
      <para>
         This method is used to decrypt ciphertext which has been encrypted
         using the corresponding public key and the
         <function>publicEncrypt</function> function.  
      </para>
   </body>
</method>";

static PyObject *
asymmetric_object_private_decrypt(asymmetric_object *self, PyObject *args)
{
   char *plain_text=NULL, *cipher_text=NULL;
   int len=0, size=0;
   PyObject *obj=NULL;
   switch( self->cipher_type )
   {
      case RSA_CIPHER:
      {
         if ( !(self->key_type == RSA_PRIVATE_KEY || self->key_type == RSA_PUBLIC_PRIVATE_KEY) ) 
            { PyErr_SetString( SSLErrorObject, "cannot perform private decryption with this key" ); goto error; }

         if (!PyArg_ParseTuple(args, "s#", &cipher_text, &len))
            goto error;
   
         size = RSA_size(self->cipher);
         if ( len > size )
            { PyErr_SetString( SSLErrorObject, "cipher text is too long" ); goto error; }

         if ( !(plain_text = malloc( size + 16 ) ) )
            { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

         if ( (len = RSA_private_decrypt( len, cipher_text, plain_text, self->cipher, RSA_PKCS1_PADDING ) ) < 0 )
            { PyErr_SetString( SSLErrorObject, "could not decrypt cipher text" ); goto error; }
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "unsupported cipher type" ); goto error; }
   }

   obj = Py_BuildValue("s#", plain_text, len);
   free( plain_text );
   return obj;

error:

   if (plain_text)
      free(plain_text);
   return NULL;
}

static char asymmetric_object_sign__doc__[] = "
<method>
   <header>
      <memberof>Asymmetric</memberof>
      <name>sign</name>
      <parameter>digesttext</parameter>
      <parameter>digesttype</parameter>
   </header>
   <body>
      <para>
         This method is used to produce a signed digest text.  
         This instance of
         <classname>Asymmetric</classname> should be a private key used for
         signing.  The parameter
         <parameter>digesttext</parameter> should be a digest of the 
         data to protect against alteration and
         finally <parameter>digesttype</parameter> should be one of the
         following:
      </para>
      <simplelist>
         <member><constant>MD2_DIGEST</constant></member>
         <member><constant>MD5_DIGEST</constant></member>
         <member><constant>SHA_DIGEST</constant></member>
         <member><constant>SHA1_DIGEST</constant></member>
         <member><constant>RIPEMD160_DIGEST</constant></member>
      </simplelist>
      <para>
         If the procedure was successful, a string containing the signed
         digest is returned. 
      </para>
   </body>
</method>";


static PyObject *
asymmetric_object_sign(asymmetric_object *self, PyObject *args)
{
   char *digest_text=NULL, *signed_text=NULL; 
   int digest_len=0, digest_type=0, digest_nid=0, signed_len=0;
   PyObject *obj=NULL;

	if (!PyArg_ParseTuple(args, "s#i", &digest_text, &digest_len, &digest_type))
		return NULL;

   switch( self->key_type )
   {
      case RSA_PRIVATE_KEY:
      {
         if ( !(signed_text = malloc( RSA_size(self->cipher) ) ) )
            { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

         switch(digest_type)
         {
            case MD2_DIGEST:
               { digest_nid = NID_md2; digest_len = MD2_DIGEST_LENGTH; break; }
            case MD5_DIGEST:
               { digest_nid = NID_md5; digest_len = MD5_DIGEST_LENGTH; break; }
            case SHA_DIGEST:
               { digest_nid = NID_sha; digest_len = SHA_DIGEST_LENGTH; break; }
            case SHA1_DIGEST:
               { digest_nid = NID_sha1; digest_len = SHA_DIGEST_LENGTH; break; }
            case RIPEMD160_DIGEST:
               { digest_nid = NID_ripemd160; digest_len = RIPEMD160_DIGEST_LENGTH; break; }
            default:
               { PyErr_SetString( SSLErrorObject, "unsupported digest" ); goto error; }
         }
         if ( !(RSA_sign( digest_nid, digest_text, digest_len, signed_text, &signed_len, self->cipher ) ) )
            { PyErr_SetString( SSLErrorObject, "could not sign digest" ); goto error; }
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "unsupported key type" ); goto error; }
   }

   obj = Py_BuildValue("s#", signed_text, signed_len);
   free(signed_text);
   return obj;

error:

   if (signed_text)
      free(signed_text);

   return NULL;
}

static char asymmetric_object_verify__doc__[] = " 
<method>
   <header>
      <memberof>Asymmetric</memberof>
      <name>verify</name>
      <parameter>signedtext</parameter>
      <parameter>digesttext</parameter>
      <parameter>digesttype</parameter>
   </header>
   <body>
      <para>
         This method is used to verify a signed digest text.  
      </para>
       <example>
         <title><function>verify</function> method usage</title>
         <programlisting>
      plain_text = 'Hello World!'
      print '\tPlain text:', plain_text
      digest = POW.Digest( POW.RIPEMD160_DIGEST )
      digest.update( plain_text )
      print '\tDigest text:', digest.digest()

      privateFile = open('test/private.key', 'r')
      privateKey = POW.pemRead( POW.RSA_PRIVATE_KEY, privateFile.read(), 'pass' )
      privateFile.close()
      signed_text =  privateKey.sign(digest.digest(), POW.RIPEMD160_DIGEST)
      print '\tSigned text:', signed_text

      digest2 = POW.Digest( POW.RIPEMD160_DIGEST )
      digest2.update( plain_text )
      publicFile = open('test/public.key', 'r')
      publicKey = POW.pemRead( POW.RSA_PUBLIC_KEY, publicFile.read() )
      publicFile.close()
      if publicKey.verify( signed_text, digest2.digest(), POW.RIPEMD160_DIGEST ):
         print 'Signing verified!'
      else:
         print 'Signing gone wrong!'
         </programlisting>
      </example>
      <para>
         The parameter <parameter>signedtext</parameter> should be a 
         signed digest text.  This instance of
         <classname>Asymmetric</classname> should correspond to the private
         key used to sign the digest.  The parameter
         <parameter>digesttext</parameter> should be a digest of the same
         data used to produce the <parameter>signedtext</parameter> and
         finally <parameter>digesttype</parameter> should be one of the
         following:
      </para>
      <simplelist>
         <member><constant>MD2_DIGEST</constant></member>
         <member><constant>MD5_DIGEST</constant></member>
         <member><constant>SHA_DIGEST</constant></member>
         <member><constant>SHA1_DIGEST</constant></member>
         <member><constant>RIPEMD160_DIGEST</constant></member>
      </simplelist>
      <para>
         If the procedure was successful, 1 is returned, otherwise 0 is
         returned.
      </para>
   </body>
</method>";

static PyObject *
asymmetric_object_verify(asymmetric_object *self, PyObject *args)
{
   char *digest_text=NULL, *signed_text=NULL; 
   int digest_len=0, digest_type=0, digest_nid=0, signed_len=0, result=0;

	if (!PyArg_ParseTuple(args, "s#s#i", &signed_text, &signed_len, &digest_text, &digest_len, &digest_type))
		return NULL;

   switch( self->key_type )
   {
      case RSA_PUBLIC_KEY:
      {
         switch(digest_type)
         {
            case MD2_DIGEST:
               { digest_len = MD2_DIGEST_LENGTH; digest_nid = NID_md2; break; }
            case MD5_DIGEST:
               { digest_len = MD5_DIGEST_LENGTH; digest_nid = NID_md5; break; }
            case SHA_DIGEST:
               { digest_len = SHA_DIGEST_LENGTH; digest_nid = NID_sha; break; }
            case SHA1_DIGEST:
               { digest_len = SHA_DIGEST_LENGTH; digest_nid = NID_sha1; break; }
            case RIPEMD160_DIGEST:
               { digest_len = RIPEMD160_DIGEST_LENGTH; digest_nid = NID_ripemd160; break; }
            default:
               { PyErr_SetString( SSLErrorObject, "unsupported digest" ); goto error; }
         }
         result = RSA_verify( digest_nid, digest_text, digest_len, signed_text, signed_len, self->cipher );
         break;
      }
      default:
         { PyErr_SetString( SSLErrorObject, "unsupported key type" ); goto error; }
   }

   return Py_BuildValue("i", result);

error:

   return NULL;
}

static struct PyMethodDef asymmetric_object_methods[] = {
   {"pemWrite",      (PyCFunction)asymmetric_object_pem_write,  
                           METH_VARARGS,  asymmetric_object_pem_write__doc__}, 
   {"publicEncrypt", (PyCFunction)asymmetric_object_public_encrypt,  
                           METH_VARARGS,  asymmetric_object_public_encrypt__doc__}, 
   {"privateEncrypt",(PyCFunction)asymmetric_object_private_encrypt,  
                           METH_VARARGS,  asymmetric_object_private_encrypt__doc__}, 
   {"privateDecrypt",(PyCFunction)asymmetric_object_private_decrypt,  
                           METH_VARARGS,  asymmetric_object_private_decrypt__doc__}, 
   {"publicDecrypt", (PyCFunction)asymmetric_object_public_decrypt,  
                           METH_VARARGS,  asymmetric_object_public_decrypt__doc__}, 
   {"sign",          (PyCFunction)asymmetric_object_sign,  
                           METH_VARARGS,  asymmetric_object_sign__doc__}, 
   {"verify",        (PyCFunction)asymmetric_object_verify,  
                           METH_VARARGS,  asymmetric_object_verify__doc__}, 
 
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
asymmetric_object_getattr(asymmetric_object *self, char *name)
{
	return Py_FindMethod(asymmetric_object_methods, (PyObject *)self, name);
}

static void
asymmetric_object_dealloc(asymmetric_object *self, char *name)
{
   switch( self->cipher_type )
   {
      case RSA_CIPHER: 
      {
         RSA_free( self->cipher ); 
         break;
      }
   }
   PyObject_Del(self);
}

static char asymmetrictype__doc__[] = "
<class>
   <header>
      <name>Asymmetric</name>
   </header>
   <body>
      <para>
         This class provides access to RSA asymmetric ciphers in OpenSSL.
         Other ciphers will probably be supported in the future but this is
         not a priority.
      </para>
   </body>
</class>";

static PyTypeObject asymmetrictype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				                           /*ob_size*/
	"Asymmetric",			                  /*tp_name*/
	sizeof(asymmetric_object),		         /*tp_basicsize*/
	0,				                           /*tp_itemsize*/
	(destructor)asymmetric_object_dealloc,	/*tp_dealloc*/
	(printfunc)0,		                     /*tp_print*/
	(getattrfunc)asymmetric_object_getattr,	/*tp_getattr*/
	(setattrfunc)0,	                     /*tp_setattr*/
	(cmpfunc)0,		                        /*tp_compare*/
	(reprfunc)0,      		               /*tp_repr*/
	0,			                              /*tp_as_number*/
	0,		                                 /*tp_as_sequence*/
	0,		                                 /*tp_as_mapping*/
	(hashfunc)0,		                     /*tp_hash*/
	(ternaryfunc)0,		                  /*tp_call*/
	(reprfunc)0,		                     /*tp_str*/
	0,
   0,
   0,
   0,
	asymmetrictype__doc__                   /* Documentation string */
};
/*========== asymmetric Code ==========*/

/*========== symmetric Code ==========*/
static symmetric_object *
symmetric_object_new(int cipher_type)
{
   symmetric_object *self=NULL;

   self = PyObject_New( symmetric_object, &symmetrictype );
   if (self == NULL)
      goto error;

   self->cipher_type = cipher_type;
   EVP_CIPHER_CTX_init( &self->cipher_ctx );

   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

static char symmetric_object_encrypt_init__doc__[] = "
<method>
   <header>
      <memberof>Symmetric</memberof>
      <name>encryptInit</name>
      <parameter>key</parameter>
      <parameter>initialvalue=''</parameter>
   </header>
   <body>
      <para>
         This method sets up the cipher object to start encrypting a stream
         of data.  The first parameter is the key used to encrypt the
         data.  The second, the <parameter>initialvalue</parameter> serves
         a similar purpose the the salt supplied to the Unix
         <function>crypt</function> function.
         The <parameter>initialvalue</parameter> is normally chosen at random and 
         often transmitted with the encrypted data, its purpose is to prevent 
         two identical plain texts resulting in two identical cipher texts.
      </para>
   </body>
</method>";


static PyObject *
symmetric_object_encrypt_init(symmetric_object *self, PyObject *args)
{
   char *key=NULL, *iv=NULL, nulliv [] = "";
   EVP_CIPHER *cipher=NULL;

	if (!PyArg_ParseTuple(args, "s|s", &key, &iv))
		goto error;

   if (!iv)
      iv = nulliv;

   if ( !(cipher = evp_cipher_factory( self->cipher_type ) ) )
         { PyErr_SetString( SSLErrorObject, "unsupported cipher" ); goto error; }

   if ( !EVP_EncryptInit( &self->cipher_ctx, cipher, key, iv ) )
      { PyErr_SetString( SSLErrorObject, "could not initialise cipher" ); goto error; }
   
   return Py_BuildValue("");

error:

   return NULL;
}

static char symmetric_object_decrypt_init__doc__[] = "
<method>
   <header>
      <memberof>Symmetric</memberof>
      <name>decryptInit</name>
      <parameter>key</parameter>
      <parameter>initialvalue=''</parameter>
   </header>
   <body>
      <para>
         This method sets up the cipher object to start decrypting a stream
         of data.  The first value must be the key used to encrypt the
         data.  The second parameter is the <parameter>initialvalue</parameter> 
         used to encrypt the data.
      </para>
   </body>
</method>";

static PyObject *
symmetric_object_decrypt_init(symmetric_object *self, PyObject *args)
{
   char *key=NULL, *iv=NULL, nulliv [] = "";
   EVP_CIPHER *cipher=NULL;

	if (!PyArg_ParseTuple(args, "s|s", &key, &iv))
		goto error;

   if (!iv)
      iv = nulliv;

   if ( !(cipher = evp_cipher_factory( self->cipher_type ) ) )
         { PyErr_SetString( SSLErrorObject, "unsupported cipher" ); goto error; }

   if ( !EVP_DecryptInit( &self->cipher_ctx, cipher, key, iv ) )
      { PyErr_SetString( SSLErrorObject, "could not initialise cipher" ); goto error; }
   
   return Py_BuildValue("");

error:

   return NULL;
}

static char symmetric_object_update__doc__[] = "
<method>
   <header>
      <memberof>Symmetric</memberof>
      <name>update</name>
      <parameter>data</parameter>
   </header>
   <body>
      <para>
         This method is used to process the bulk of data being encrypted
         or decrypted by the cipher object.  <parameter>data</parameter>
         should be a string.
      </para>
   </body>
</method>";

static PyObject *
symmetric_object_update(symmetric_object *self, PyObject *args)
{
   int inl=0, outl=0;
   char *in=NULL, *out=NULL;
   PyObject *py_out=NULL;

	if (!PyArg_ParseTuple(args, "s#", &in, &inl))
		goto error;

   if ( !(out = malloc( inl + EVP_CIPHER_CTX_block_size( &self->cipher_ctx) ) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   if ( !EVP_CipherUpdate( &self->cipher_ctx, out, &outl, in, inl ) )
      { PyErr_SetString( SSLErrorObject, "could not update cipher" ); goto error; }

   if ( !(py_out = Py_BuildValue("s#", out, outl) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   free(out);
   return py_out;

error:

   if (out)
      free(out);

   return NULL;
}

static char symmetric_object_final__doc__[] = "
<method>
   <header>
      <memberof>Symmetric</memberof>
      <name>final</name>
      <parameter>size=1024</parameter>
   </header>
   <body>
      <para>
         Most ciphers are block ciphers, that is they encrypt or decrypt a block of
         data at a time.  Often the data being processed will not fill an
         entire block, this method processes these half-empty blocks.  A
         string is returned of a maximum length <parameter>size</parameter>. 
      </para>
   </body>
</method>";


static PyObject *
symmetric_object_final(symmetric_object *self, PyObject *args)
{
   int outl=0, size=1024;
   char *out=NULL;
   PyObject *py_out=NULL;

	if (!PyArg_ParseTuple(args, "|i", &size))
		goto error;

   if ( !(out = malloc( size + EVP_CIPHER_CTX_block_size( &self->cipher_ctx) ) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   if ( !EVP_CipherFinal( &self->cipher_ctx, out, &outl ) )
      { PyErr_SetString( SSLErrorObject, "could not update cipher" ); goto error; }

   if ( !(py_out = Py_BuildValue("s#", out, outl) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   free(out);
   return py_out;

error:

   if (out)
      free(out);

   return NULL;
}

static struct PyMethodDef symmetric_object_methods[] = {
   {"encryptInit",      (PyCFunction)symmetric_object_encrypt_init,  
                           METH_VARARGS,  symmetric_object_encrypt_init__doc__}, 
   {"decryptInit",      (PyCFunction)symmetric_object_decrypt_init,  
                           METH_VARARGS,  symmetric_object_decrypt_init__doc__}, 
   {"update",      (PyCFunction)symmetric_object_update,  
                           METH_VARARGS,  symmetric_object_update__doc__}, 
   {"final",      (PyCFunction)symmetric_object_final,  
                           METH_VARARGS,  symmetric_object_final__doc__}, 
 
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
symmetric_object_getattr(symmetric_object *self, char *name)
{
	return Py_FindMethod(symmetric_object_methods, (PyObject *)self, name);
}

static void
symmetric_object_dealloc(symmetric_object *self, char *name)
{
   PyObject_Del(self);
}

static char symmetrictype__doc__[] = "
<class>
   <header>
      <name>Symmetric</name>
   </header>
   <body>
      <para>
         This class provides access to all the symmetric ciphers in OpenSSL.
         Initialisation of the cipher structures is performed late, only
         when <function>encryptInit</function> or
         <function>decryptInit</function> is called, the
         constructor only records the cipher type.  It is possible to reuse
         the <classname>Symmetric</classname> objects by calling
         <function>encryptInit</function> or <function>decryptInit</function>
         again.
      </para>
      <example>
         <title><classname>Symmetric</classname> class usage</title>
         <programlisting>
      passphrase = 'my silly passphrase'
      md5 = POW.Digest( POW.MD5_DIGEST )
      md5.update( passphrase )
      password = md5.digest()[:8]

      plaintext = 'cast test message'
      cast = POW.Symmetric( POW.CAST5_CFB ) 
      cast.encryptInit( password )
      ciphertext = cast.update(plaintext) + cast.final()
      print 'Cipher text:', ciphertext

      cast.decryptInit( password )
      out = cast.update( ciphertext ) + cast.final()
      print 'Deciphered text:', out
         </programlisting>
      </example>
   </body>
</class>";


static PyTypeObject symmetrictype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				                           /*ob_size*/
	"Symmetric",			                     /*tp_name*/
	sizeof(symmetric_object),		         /*tp_basicsize*/
	0,				                           /*tp_itemsize*/
	(destructor)symmetric_object_dealloc,	/*tp_dealloc*/
	(printfunc)0,		                     /*tp_print*/
	(getattrfunc)symmetric_object_getattr,	/*tp_getattr*/
	(setattrfunc)0,	                     /*tp_setattr*/
	(cmpfunc)0,		                        /*tp_compare*/
	(reprfunc)0,      		               /*tp_repr*/
	0,			                              /*tp_as_number*/
	0,		                                 /*tp_as_sequence*/
	0,		                                 /*tp_as_mapping*/
	(hashfunc)0,		                     /*tp_hash*/
	(ternaryfunc)0,		                  /*tp_call*/
	(reprfunc)0,		                     /*tp_str*/
	0,
   0,
   0,
   0,
	symmetrictype__doc__                    /* Documentation string */
};
/*========== symmetric Code ==========*/

/*========== digest Code ==========*/
static digest_object *
digest_object_new(int digest_type)
{
   digest_object *self=NULL;

   self = PyObject_New( digest_object, &digesttype );
   if (self == NULL)
      goto error;

   switch(digest_type)
   {
      case MD2_DIGEST: 
         { self->digest_type = MD2_DIGEST; EVP_DigestInit( &self->digest_ctx, EVP_md2() ); break; }
      case MD5_DIGEST: 
         { self->digest_type = MD5_DIGEST; EVP_DigestInit( &self->digest_ctx, EVP_md5() ); break; }
      case SHA_DIGEST: 
         { self->digest_type = SHA_DIGEST; EVP_DigestInit( &self->digest_ctx, EVP_sha() ); break; }
      case SHA1_DIGEST: 
         { self->digest_type = SHA1_DIGEST; EVP_DigestInit( &self->digest_ctx, EVP_sha1() ); break; }
      case RIPEMD160_DIGEST: 
         { self->digest_type = RIPEMD160_DIGEST; EVP_DigestInit( &self->digest_ctx, EVP_ripemd160() ); break; }
      default:
         { PyErr_SetString( SSLErrorObject, "unsupported digest" ); goto error; }
   }

   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

static char digest_object_update__doc__[] = "
<method>
   <header>
      <memberof>Digest</memberof>
      <name>update</name>
      <parameter>data</parameter>
   </header>
   <body>
      <para>
         This method updates the internal structures of the 
         <classname>Digest</classname> object with <parameter>data</parameter>.
         <parameter>data</parameter> should be a string.
      </para>
   </body>
</method>";

static PyObject *
digest_object_update(digest_object *self, PyObject *args)
{
   char *data=NULL;
   int len=0;

	if (!PyArg_ParseTuple(args, "s#", &data, &len))
		goto error;

   EVP_DigestUpdate( &self->digest_ctx, data, len );

   return Py_BuildValue("");

error:

   return NULL;
}

static char digest_object_copy__doc__[] = "
<method>
   <header>
      <memberof>Digest</memberof>
      <name>copy</name>
   </header>
   <body>
      <para>
         This method returns a copy of the <classname>Digest</classname>
         object.
      </para>
   </body>
</method>";

static PyObject *
digest_object_copy(digest_object *self, PyObject *args)
{
   digest_object *new=NULL;

   if ( !(new = PyObject_New( digest_object, &digesttype ) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   new->digest_type = self->digest_type;
   memcpy( &new->digest_ctx, &self->digest_ctx, sizeof(EVP_MD_CTX) );

   return (PyObject*)new;

error:

   Py_XDECREF(new);
   return NULL;
}

static char digest_object_digest__doc__[] = "
<method>
   <header>
      <memberof>Digest</memberof>
      <name>digest</name>
   </header>
   <body>
      <para>
         This method returns the digest of all the data which has been
         processed.  This function can be called at any time and will not
         effect the internal structure of the <classname>digest</classname>
         object.
      </para>
   </body>
</method>";

static PyObject *
digest_object_digest(digest_object *self, PyObject *args)
{
   char digest_text[EVP_MAX_MD_SIZE];
   void *md_copy=NULL;
   int digest_len=0;

	if (!PyArg_ParseTuple(args, ""))
		goto error;

   if ( !(md_copy = malloc( sizeof(EVP_MD_CTX) ) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }
   memcpy( md_copy, &self->digest_ctx, sizeof(EVP_MD_CTX) );
   EVP_DigestFinal( md_copy, digest_text, &digest_len );

   free(md_copy);
   return Py_BuildValue("s#", digest_text, digest_len);

error:

   if (md_copy)
      free(md_copy);

   return NULL;
}


static struct PyMethodDef digest_object_methods[] = {
   {"update",           (PyCFunction)digest_object_update,  
                           METH_VARARGS,  digest_object_update__doc__}, 
   {"digest",           (PyCFunction)digest_object_digest,  
                           METH_VARARGS,  digest_object_digest__doc__}, 
   {"copy",             (PyCFunction)digest_object_copy,  
                           METH_VARARGS,  digest_object_copy__doc__}, 
 
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
digest_object_getattr(digest_object *self, char *name)
{
	return Py_FindMethod(digest_object_methods, (PyObject *)self, name);
}

static void
digest_object_dealloc(digest_object *self, char *name)
{
   PyObject_Del(self);
}

static char digesttype__doc__[] = "
<class>
   <header>
      <name>Digest</name>
   </header>
   <body>
      <para>
         This class provides access to the digest functionality of OpenSSL.
         It emulates the digest modules in the Python Standard Library but
         does not currently support the <function>hexdigest</function>
         function.
      </para>
      <example>
         <title><classname>digest</classname> class usage</title>
         <programlisting>
      plain_text = 'Hello World!'
      sha1 = POW.Digest( POW.SHA1_DIGEST )
      sha1.update( plain_text )
      print '\tPlain text: Hello World! =>', sha1.digest()
         </programlisting>
      </example>
   </body>
</class>";

static PyTypeObject digesttype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				                        /*ob_size*/
	"Digest",		                     /*tp_name*/
	sizeof(digest_object),		         /*tp_basicsize*/
	0,				                        /*tp_itemsize*/
	(destructor)digest_object_dealloc,	/*tp_dealloc*/
	(printfunc)0,		                  /*tp_print*/
	(getattrfunc)digest_object_getattr,	/*tp_getattr*/
	(setattrfunc)0,	                  /*tp_setattr*/
	(cmpfunc)0,		                     /*tp_compare*/
	(reprfunc)0,      		            /*tp_repr*/
	0,			                           /*tp_as_number*/
	0,		                              /*tp_as_sequence*/
	0,		                              /*tp_as_mapping*/
	(hashfunc)0,		                  /*tp_hash*/
	(ternaryfunc)0,		               /*tp_call*/
	(reprfunc)0,		                  /*tp_str*/
	0,
   0,
   0,
   0,
	digesttype__doc__                   /* Documentation string */
};
/*========== digest Code ==========*/

/*========== hmac Code ==========*/
static hmac_object *
hmac_object_new(int digest_type, char *key, int key_len)
{
   hmac_object *self=NULL;
   EVP_MD *md=NULL;

   self = PyObject_New( hmac_object, &hmactype );
   if (self == NULL)
      goto error;

   switch(digest_type)
   {
      case MD2_DIGEST: 
         { md = EVP_md2(); break; }
      case MD5_DIGEST: 
         { md = EVP_md5(); break; }
      case SHA_DIGEST: 
         { md = EVP_sha(); break; }
      case SHA1_DIGEST: 
         { md = EVP_sha1(); break; }
      case RIPEMD160_DIGEST: 
         { md = EVP_ripemd160(); break; }
      default:
         { PyErr_SetString( SSLErrorObject, "unsupported digest" ); goto error; }
   }

   HMAC_Init( &self->hmac_ctx, key, key_len, md );

   return self;

error:

   Py_XDECREF(self);
   return NULL;
}

static char hmac_object_update__doc__[] = "
<method>
   <header>
      <memberof>Hmac</memberof>
      <name>update</name>
      <parameter>data</parameter>
   </header>
   <body>
      <para>
         This method updates the internal structures of the 
         <classname>Hmac</classname> object with <parameter>data</parameter>.
         <parameter>data</parameter> should be a string.
      </para>
   </body>
</method>";

static PyObject *
hmac_object_update(hmac_object *self, PyObject *args)
{
   char *data=NULL;
   int len=0;

	if (!PyArg_ParseTuple(args, "s#", &data, &len))
		goto error;

   HMAC_Update( &self->hmac_ctx, data, len );

   return Py_BuildValue("");

error:

   return NULL;
}

static char hmac_object_copy__doc__[] = "
<method>
   <header>
      <memberof>Hmac</memberof>
      <name>copy</name>
   </header>
   <body>
      <para>
         This method returns a copy of the <classname>Hmac</classname>
         object.
      </para>
   </body>
</method>";

static PyObject *
hmac_object_copy(hmac_object *self, PyObject *args)
{
   hmac_object *new=NULL;

   if ( !(new = PyObject_New( hmac_object, &hmactype ) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   memcpy( &new->hmac_ctx, &self->hmac_ctx, sizeof(HMAC_CTX) );

   return (PyObject*)new;

error:

   Py_XDECREF(new);
   return NULL;
}

static char hmac_object_mac__doc__[] = "
<method>
   <header>
      <memberof>Hmac</memberof>
      <name>mac</name>
   </header>
   <body>
      <para>
         This method returns the MAC of all the data which has been
         processed.  This function can be called at any time and will not
         effect the internal structure of the <classname>Hmac</classname>
         object.
      </para>
   </body>
</method>";

static PyObject *
hmac_object_mac(hmac_object *self, PyObject *args)
{
   char hmac_text[EVP_MAX_MD_SIZE];
   void *hmac_copy=NULL;
   int hmac_len=0;

	if (!PyArg_ParseTuple(args, ""))
		goto error;

   if ( !(hmac_copy = malloc( sizeof(HMAC_CTX) ) ) )
      { PyErr_SetString( SSLErrorObject, "could not allocate memory" ); goto error; }

   memcpy( hmac_copy, &self->hmac_ctx, sizeof(HMAC_CTX) );
   HMAC_Final( hmac_copy, hmac_text, &hmac_len );

   free(hmac_copy);
   return Py_BuildValue("s#", hmac_text, hmac_len);

error:

   if (hmac_copy)
      free(hmac_copy);

   return NULL;
}


static struct PyMethodDef hmac_object_methods[] = {
   {"update",           (PyCFunction)hmac_object_update,  
                           METH_VARARGS,  hmac_object_update__doc__}, 
   {"mac",              (PyCFunction)hmac_object_mac,  
                           METH_VARARGS,  digest_object_digest__doc__}, 
   {"copy",             (PyCFunction)hmac_object_copy,  
                           METH_VARARGS,  hmac_object_copy__doc__}, 
 
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
hmac_object_getattr(hmac_object *self, char *name)
{
	return Py_FindMethod(hmac_object_methods, (PyObject *)self, name);
}

static void
hmac_object_dealloc(hmac_object *self, char *name)
{
   PyObject_Del(self);
}

static char hmactype__doc__[] = "
<class>
   <header>
      <name>Hmac</name>
   </header>
   <body>
      <para>
         This class provides access to the HMAC functionality of OpenSSL.  
         HMAC's are a varient on digest based MACs, which have the 
         interesting property of a provable level of security.  HMAC is 
         discussed further in RFC 2104.  
      </para>
   </body>
</class>";

static PyTypeObject hmactype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				                        /*ob_size*/
	"Hmac",		                        /*tp_name*/
	sizeof(hmac_object),		            /*tp_basicsize*/
	0,				                        /*tp_itemsize*/
	(destructor)hmac_object_dealloc,	   /*tp_dealloc*/
	(printfunc)0,		                  /*tp_print*/
	(getattrfunc)hmac_object_getattr,	/*tp_getattr*/
	(setattrfunc)0,	                  /*tp_setattr*/
	(cmpfunc)0,		                     /*tp_compare*/
	(reprfunc)0,      		            /*tp_repr*/
	0,			                           /*tp_as_number*/
	0,		                              /*tp_as_sequence*/
	0,		                              /*tp_as_mapping*/
	(hashfunc)0,		                  /*tp_hash*/
	(ternaryfunc)0,		               /*tp_call*/
	(reprfunc)0,		                  /*tp_str*/
	0,
   0,
   0,
   0,
	hmactype__doc__                     /* Documentation string */
};
/*========== hmac Code ==========*/

/*========== module functions ==========*/
static char pow_module_new_ssl__doc__[] = "
<constructor>
   <header>
      <memberof>Ssl</memberof>
      <parameter>protocol=SSLV23METHOD</parameter>
   </header>
   <body>
      <para>
         This constructor creates a new <classname>Ssl</classname> object which will behave as a client
         or server, depending on the <parameter>protocol</parameter> value passed.  The
         <parameter>protocol</parameter> also determines the protocol type
         and version and should be one of the following:
      </para>

      <simplelist>
         <member><constant>SSLV2_SERVER_METHOD</constant></member>
         <member><constant>SSLV2_CLIENT_METHOD</constant></member>
         <member><constant>SSLV2_METHOD</constant></member>
         <member><constant>SSLV3_SERVER_METHOD</constant></member>
         <member><constant>SSLV3_CLIENT_METHOD</constant></member>
         <member><constant>SSLV3_METHOD</constant></member>
         <member><constant>TLSV1_SERVER_METHOD</constant></member>
         <member><constant>TLSV1_CLIENT_METHOD</constant></member>
         <member><constant>TLSV1_METHOD</constant></member>
         <member><constant>SSLV23_SERVER_METHOD</constant></member>
         <member><constant>SSLV23_CLIENT_METHOD</constant></member>
         <member><constant>SSLV23_METHOD</constant></member>
      </simplelist>
   </body>
</constructor>";

static PyObject *
pow_module_new_ssl (PyObject *self, PyObject *args)
{
   ssl_object *ssl=NULL;
   int ctxtype=SSLV23_METHOD;

	if (!PyArg_ParseTuple(args, "|i", &ctxtype))
   	goto error;

   if ( !(ssl = newssl_object(ctxtype) ) )
      goto error;

   return (PyObject*)ssl;

error:

   return NULL;
}

static char pow_module_new_x509__doc__[] = "
<constructor>
   <header>
      <memberof>X509</memberof>
   </header>
   <body>
      <para>
         This constructor creates a skeletal X509 certificate object.
         It won't be any use at all until several structures 
         have been created using it's member functions.  
      </para>
   </body>
</constructor>";

static PyObject *
pow_module_new_x509 (PyObject *self, PyObject *args)
{
   x509_object *x509=NULL;

	if (!PyArg_ParseTuple(args, ""))
		goto error;
   
   if ( !(x509 = X509_object_new() ) ) 
      { PyErr_SetString( SSLErrorObject, "could not create new x509 object" ); goto error; }

   return (PyObject*)x509;
 
error:

   return NULL;
}

static char pow_module_new_asymmetric__doc__[] = "
<constructor>
   <header>
      <memberof>Asymmetric</memberof>
      <parameter>ciphertype=RSA_CIPHER</parameter>
      <parameter>keylength=1024</parameter>
   </header>
   <body>
      <para>
         This constructor builds a new cipher object.  Only RSA ciphers
         are currently support, so the first argument should always be
         <constant>RSA_CIPHER</constant>.  The second argument,
         <parameter>keylength</parameter>,
         is normally 512, 768, 1024 or 2048.  Key lengths as short as 512
         bits are generally considered weak, and can be cracked by
         determined attackers without tremendous expense.
      </para>
      <example>
         <title><classname>asymmetric</classname> class usage</title>
         <programlisting>
      privateFile = open('test/private.key', 'w')
      publicFile = open('test/public.key', 'w')

      passphrase = 'my silly passphrase'
      md5 = POW.Digest( POW.MD5_DIGEST )
      md5.update( passphrase )
      password = md5.digest()

      rsa = POW.Asymmetric( POW.RSA_CIPHER, 1024 )
      privateFile.write( rsa.pemWrite( 
               POW.RSA_PRIVATE_KEY, POW.DES_EDE3_CFB, password ) )
      publicFile.write( rsa.pemWrite( POW.RSA_PUBLIC_KEY ) )

      privateFile.close()
      publicFile.close()
         </programlisting>
      </example>
   </body>
</constructor>";

static PyObject *
pow_module_new_asymmetric (PyObject *self, PyObject *args)
{
   int cipher_type=RSA_CIPHER, key_size=1024;

	if (!PyArg_ParseTuple(args, "|ii", &cipher_type, &key_size))
		goto error;

   return (PyObject*)asymmetric_object_new( cipher_type, key_size );

error:

   return NULL;
}

static char pow_module_new_digest__doc__[] = "
<constructor>
   <header>
      <memberof>Digest</memberof>
      <parameter>type</parameter>
   </header>
   <body>
      <para>
         This constructor creates a new <classname>Digest</classname>
         object.  The parameter <parameter>type</parameter> specifies what kind
         of digest to create and should be one of the following: 
      </para>
      <simplelist>
         <member><constant>MD2_DIGEST</constant></member>
         <member><constant>MD5_DIGEST</constant></member>
         <member><constant>SHA_DIGEST</constant></member>
         <member><constant>SHA1_DIGEST</constant></member>
         <member><constant>RIPEMD160_DIGEST</constant></member>
      </simplelist>
   </body>
</constructor>";

static PyObject *
pow_module_new_digest (PyObject *self, PyObject *args)
{
   int digest_type=0;

	if (!PyArg_ParseTuple(args, "i", &digest_type))
		goto error;

   return (PyObject*)digest_object_new( digest_type );

error:

   return NULL;
}

static char pow_module_new_hmac__doc__[] = "
<constructor>
   <header>
      <memberof>Hmac</memberof>
      <parameter>type</parameter>
      <parameter>key</parameter>
   </header>
   <body>
      <para>
         This constructor creates a new <classname>Hmac</classname>
         object.  The parameter <parameter>key</parameter> should be a
         string and <parameter>type</parameter> should be one of the following: 
      </para>
      <simplelist>
         <member><constant>MD2_DIGEST</constant></member>
         <member><constant>MD5_DIGEST</constant></member>
         <member><constant>SHA_DIGEST</constant></member>
         <member><constant>SHA1_DIGEST</constant></member>
         <member><constant>RIPEMD160_DIGEST</constant></member>
      </simplelist>
   </body>
</constructor>";

static PyObject *
pow_module_new_hmac (PyObject *self, PyObject *args)
{
   int digest_type=0, key_len=0;
   char *key=NULL;

	if (!PyArg_ParseTuple(args, "is#", &digest_type, &key, &key_len))
		goto error;

   return (PyObject*)hmac_object_new( digest_type, key, key_len );

error:

   return NULL;
}

static char pow_module_pem_read__doc__[] = "
<modulefunction>
   <header>
      <name>pemRead</name>
      <parameter>type</parameter>
      <parameter>string</parameter>
      <parameter>pass=None</parameter>
   </header>
   <body>
      <para>
         This function attempts to parse the <parameter>string</parameter> according to the PEM
         type passed. <parameter>type</parameter> should be one of the
         following:
      </para>
      <simplelist>
         <member><constant>RSA_PUBLIC_KEY</constant></member>
         <member><constant>RSA_PRIVATE_KEY</constant></member>
         <member><constant>X509_CERTIFICATE</constant></member>
         <member><constant>X509_CRL</constant></member>
      </simplelist>
      <para>
         <parameter>pass</parameter> should only be provided if an encrypted
         <classname>Asymmetric</classname> is being loaded.  If the password
         is incorrect an exception will be raised, if no password is provided
         and the PEM file is encrypted the user will be prompted.  If this is
         not desirable, always supply a password.  The object returned will be 
         and instance of <classname>Asymmetric</classname>, 
         <classname>X509</classname> or <classname>X509Crl</classname>.
      </para>
   </body>
</modulefunction>";

static PyObject *
pow_module_pem_read (PyObject *self, PyObject *args)
{
   BIO *in=NULL;
   PyObject *obj=NULL;
   int object_type=0, len=0;
   char *pass=NULL, *src=NULL;

	if (!PyArg_ParseTuple(args, "is#|s", &object_type, &src, &len, &pass))
		goto error;

   if ( !(in = BIO_new_mem_buf(src, -1) ) )
      { PyErr_SetString( SSLErrorObject, "unable to create new BIO" ); goto error; }

   if ( !BIO_write( in, src, len ) )
      { PyErr_SetString( SSLErrorObject, "unable to write to BIO" ); goto error; }

   switch(object_type)
   {
      case RSA_PRIVATE_KEY:
         { obj = (PyObject*)asymmetric_object_pem_read( object_type, in, pass ); break; }
      case RSA_PUBLIC_KEY:
         { obj = (PyObject*)asymmetric_object_pem_read( object_type, in, pass ); break; }
      case X509_CERTIFICATE:
         { obj = (PyObject*)X509_object_pem_read( in ); break ; }
      case X_X509_CRL:
         { obj = (PyObject*)x509_crl_object_pem_read( in ); break ; }

      default:
         { PyErr_SetString( SSLErrorObject, "unknown pem encoding" ); goto error; }
   }

   BIO_free(in);

   if (obj)
      return obj;

error:

   return NULL;
}

static char pow_module_new_x509_store__doc__[] = "
<constructor>
   <header>
      <memberof>X509Store</memberof>
   </header>
   <body>
      <para>
         This constructor takes no arguments.  The
         <classname>X509Store</classname> returned cannot be used for
         verifying certificates until at least one trusted certificate has been
         added.
      </para>
   </body>
</constructor>";

static PyObject *
pow_module_new_x509_store (PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ""))
		goto error;

   return (PyObject*)x509_store_object_new();

error:

   return NULL;
}

static char pow_module_new_symmetric__doc__[] = "
<constructor>
   <header>
      <memberof>Symmetric</memberof>
      <parameter>type</parameter>
   </header>
   <body>
      <para>
         This constructor creates a new <classname>Symmetric</classname>
         object.  The parameter <parameter>type</parameter> specifies which kind
         of cipher to create. <constant>type</constant> should be one of the following: 
      </para>
      <simplelist columns=\"2\">
         <member><constant>DES_ECB</constant></member>     
         <member><constant>DES_EDE</constant></member>
         <member><constant>DES_EDE3</constant></member>   
         <member><constant>DES_CFB</constant></member>    
         <member><constant>DES_EDE_CFB</constant></member> 
         <member><constant>DES_EDE3_CFB</constant></member>
         <member><constant>DES_OFB</constant></member>
         <member><constant>DES_EDE_OFB</constant></member>
         <member><constant>DES_EDE3_OFB</constant></member>
         <member><constant>DES_CBC</constant></member>
         <member><constant>DES_EDE_CBC</constant></member>
         <member><constant>DES_EDE3_CBC</constant></member>
         <member><constant>DESX_CBC</constant></member>
         <member><constant>RC4</constant></member>
         <member><constant>RC4_40</constant></member>
         <member><constant>IDEA_ECB</constant></member>
         <member><constant>IDEA_CFB</constant></member>
         <member><constant>IDEA_OFB</constant></member>
         <member><constant>IDEA_CBC</constant></member>
         <member><constant>RC2_ECB</constant></member>
         <member><constant>RC2_CBC</constant></member>
         <member><constant>RC2_40_CBC</constant></member>
         <member><constant>RC2_CFB</constant></member>
         <member><constant>RC2_OFB</constant></member>
         <member><constant>BF_ECB</constant></member>
         <member><constant>BF_CBC</constant></member>
         <member><constant>BF_CFB</constant></member>
         <member><constant>BF_OFB</constant></member>
         <member><constant>CAST5_ECB</constant></member>
         <member><constant>CAST5_CBC</constant></member>
         <member><constant>CAST5_CFB</constant></member>
         <member><constant>CAST5_OFB</constant></member>
         <member><constant>RC5_32_12_16_CBC</constant></member>
         <member><constant>RC5_32_12_16_CFB</constant></member>
         <member><constant>RC5_32_12_16_ECB</constant></member>
         <member><constant>RC5_32_12_16_OFB</constant></member>
      </simplelist>
      <para>
         Please note your version of OpenSSL might not have been compiled with
         all the ciphers listed above.  If that is the case, which is very
         likely if you are using a stock binary, the unsuported ciphers will not even
         be in the module namespace.
      </para>
   </body>
</constructor>";

static PyObject *
pow_module_new_symmetric (PyObject *self, PyObject *args)
{
   int cipher_type=0; 

	if (!PyArg_ParseTuple(args, "i", &cipher_type))
		goto error; 

   return (PyObject*)symmetric_object_new(cipher_type);

error:

   return NULL;
}

static char pow_module_new_x509_crl__doc__[] = " ;
<constructor>
   <header>
      <memberof>x509_crl</memberof>
   </header>
   <body>
      <para>
         This constructor builds an empty CRL.
      </para>
   </body>
</constructor>";

static PyObject *
pow_module_new_x509_crl (PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ""))
		goto error; 

   return (PyObject*)x509_crl_object_new();

error:

   return NULL;
}

static char pow_module_new_x509_revoked__doc__[] = "
<constructor>
   <header>
      <memberof>X509Revoked</memberof>
      <parameter>serial</parameter>
      <parameter>date</parameter>
   </header>
   <body>
      <para>
         This constructor builds a X509 Revoked structure.  Both arguments
         are integers, <parameter>date</parameter> is the same as the C
         <constant>time_t</constant> typedef and can be
         generated with the Python Standard Library function
         <function>time</function>.
      </para>
   </body>
</constructor>";

static PyObject *
pow_module_new_x509_revoked (PyObject *self, PyObject *args)
{
   int serial=-1, date=-1;
   x509_revoked_object *revoke=NULL;

	if (!PyArg_ParseTuple(args, "|ii", &serial, &date))
		goto error; 

   revoke = x509_revoked_object_new();

   if (serial != -1)
      if ( !ASN1_INTEGER_set( revoke->revoked->serialNumber, serial ) )
         { PyErr_SetString( SSLErrorObject, "unable to set serial number" ); goto error; }

   if (date != -1)
      if (!ASN1_UTCTIME_set( revoke->revoked->revocationDate, date ))
         { PyErr_SetString( PyExc_TypeError, "could not set revocationDate" ); goto error; }

   return (PyObject*)revoke;

error:

   return NULL;
}

static char pow_module_get_error__doc__[] = " ;
<modulefunction>
   <header>
      <name>getError</name>
   </header>
   <body>
      <para>
         Pops an error off the global error stack and returns it as a string.
      </para>
   </body>
</modulefunction>";

static PyObject *
pow_module_get_error(PyObject *self, PyObject *args)
{
   unsigned long error;
   char buf[256];
   PyObject *msg;
   
	if (!PyArg_ParseTuple(args, ""))
		goto error;
   
   error = ERR_get_error();
   ERR_error_string( error, buf );

   if ( !(msg = Py_BuildValue("s", buf) ) )
      {PyErr_SetString(SSLErrorObject, "unable to allocate memory"); goto error;}

   return msg;

error:

   return NULL;
}

static char pow_module_clear_error__doc__[] = "
<modulefunction>
   <header>
      <name>clearError</name>
   </header>
   <body>
      <para>
         Removes all errors from the global error stack.
      </para>
   </body>
</modulefunction>";

static PyObject *
pow_module_clear_error(PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ""))
		goto error;
   
   ERR_clear_error();

   return Py_BuildValue("");

error:

   return NULL;
}

static char pow_module_seed__doc__[] = "
<modulefunction>
   <header>
      <name>seed</name>
      <parameter>data</parameter>
   </header>
   <body>
      <para>
         The <function>seed</function> function adds data to OpenSSLs PRNG
         state.  It is often said the hardest part of cryptography is
         getting good random data, after all if you don't have good random
         data, a 1024 bit key is no better than a 512 bit key and neither
         would provide protection from a targeted brute force attack.
         The <function>seed</function> and <function>add</function> are very
         similar, except the entropy of the data is assumed to be equal to
         the length for <function>seed</function>.  I final point to be aware 
         of, only systems which support /dev/urandom are automatically seeded.  
         If your system does not support /dev/urandom it is your responsibility 
         to seed OpenSSL's PRNG.
      </para>
   </body>
</modulefunction>";

static PyObject *
pow_module_seed(PyObject *self, PyObject *args)
{
   char *in=NULL;
   int inl=0;

	if (!PyArg_ParseTuple(args, "s#", &in, &inl))
		goto error;
   
   RAND_seed( in, inl );

   return Py_BuildValue("");

error:

   return NULL;
}

static char pow_module_add__doc__[] = "
<modulefunction>
   <header>
      <name>add</name>
      <parameter>data</parameter>
      <parameter>entropy</parameter>
   </header>
   <body>
      <para>
         The <function>add</function> function adds data to OpenSSLs PRNG
         state.  <parameter>data</parameter> should be data obtained from a
         random source and <parameter>entropy</parameter> is an estimation of the number of random
         bytes in <parameter>data</parameter>.
      </para>
   </body>
</modulefunction>";

static PyObject *
pow_module_add(PyObject *self, PyObject *args)
{
   char *in=NULL;
   int inl=0;
   double entropy=0;

	if (!PyArg_ParseTuple(args, "s#d", &in, &inl, &entropy))
		goto error;
   
   RAND_add( in, inl, entropy );

   return Py_BuildValue("");

error:

   return NULL;
}

static char pow_module_write_random_file__doc__[] = "
<modulefunction>
   <header>
      <name>writeRandomFile</name>
      <parameter>filename</parameter>
   </header>
   <body>
      <para>
         This function writes the current random state to a file.  Clearly
         this function should be used in conjunction with
         <function>readRandomFile</function>.
      </para>
   </body>
</modulefunction>";

static PyObject *
pow_module_write_random_file(PyObject *self, PyObject *args)
{
   char *file=NULL;

	if (!PyArg_ParseTuple(args, "s", &file))
		goto error;
   
   if ( RAND_write_file( file ) == -1 )
      {PyErr_SetString(SSLErrorObject, "could not write random file"); goto error;}

   return Py_BuildValue("");

error:

   return NULL;
}

static char pow_module_read_random_file__doc__[] = "
<modulefunction>
   <header>
      <name>readRandomFile</name>
      <parameter>filename</parameter>
   </header>
   <body>
      <para>
         This function reads a previously saved random state.  It can be very
         useful to improve the quality of random data used by an application.
         The random data should be added to, using the
         <function>add</function> function, with data from other
         suitable random sources.
      </para>
   </body>
</modulefunction>";

static PyObject *
pow_module_read_random_file(PyObject *self, PyObject *args)
{
   char *file=NULL;
   int len=0;

	if (!PyArg_ParseTuple(args, "si", &file, &len))
		goto error;
   
   if (!RAND_load_file( file, len ) )
      {PyErr_SetString(SSLErrorObject, "could not load random file"); goto error;}

   return Py_BuildValue("");

error:

   return NULL;
}

static char pow_module___doclist____doc__[] = "
<modulefunction>
   <header>
      <name>__doclist__</name>
   </header>
   <body>
      <para>
         This function returns a list of all the doc strings in this
         module.  The doc strings contain a mixture of DocBook markup and
         custom tags which semi-formally describe the class or function.  The
         list of strings was used to generate this documentation, it was
         processed by a simple and pretty raw script which produced a valid
         DobBook article.  Finally Openjade was used process the DocBook article
         to produce this document.
      </para>
   </body>
</modulefunction>";

static PyObject *
pow_module___doclist__(PyObject *self, PyObject *args)
{
   PyObject *docset, *doctuple;

	if (!PyArg_ParseTuple(args, ""))
		goto error;
   
   docset = PyList_New(0);
   // module documentation
   docset_helper_add( docset, pow_module__doc__ );
   // constructors
   docset_helper_add( docset, pow_module_new_symmetric__doc__ );
   docset_helper_add( docset, pow_module_new_asymmetric__doc__ );
   docset_helper_add( docset, pow_module_new_digest__doc__ );
   docset_helper_add( docset, pow_module_new_hmac__doc__ );
   docset_helper_add( docset, pow_module_new_ssl__doc__ );
   docset_helper_add( docset, pow_module_new_x509__doc__ );
   docset_helper_add( docset, pow_module_new_x509_store__doc__ );
   docset_helper_add( docset, pow_module_new_x509_crl__doc__ );
   docset_helper_add( docset, pow_module_new_x509_revoked__doc__ );
   // functions
   docset_helper_add( docset, pow_module_pem_read__doc__ );
   docset_helper_add( docset, pow_module_seed__doc__ );
   docset_helper_add( docset, pow_module_add__doc__ );
   docset_helper_add( docset, pow_module_read_random_file__doc__ );
   docset_helper_add( docset, pow_module_write_random_file__doc__ );
   docset_helper_add( docset, pow_module_get_error__doc__ );
   docset_helper_add( docset, pow_module_clear_error__doc__ );
   docset_helper_add( docset, pow_module___doclist____doc__ );

   // ssl documentation
   docset_helper_add( docset, ssltype__doc__ );
   docset_helper_add( docset, ssl_object_set_fd__doc__ );
   docset_helper_add( docset, ssl_object_accept__doc__ );
   docset_helper_add( docset, ssl_object_connect__doc__ );
   docset_helper_add( docset, ssl_object_write__doc__ );
   docset_helper_add( docset, ssl_object_read__doc__ );
   docset_helper_add( docset, ssl_object_peer_certificate__doc__ );
   docset_helper_add( docset, ssl_object_use_certificate__doc__ );
   docset_helper_add( docset, ssl_object_use_key__doc__ );
   docset_helper_add( docset, ssl_object_check_key__doc__ );
   docset_helper_add( docset, ssl_object_clear__doc__ );
   docset_helper_add( docset, ssl_object_shutdown__doc__ );
   docset_helper_add( docset, ssl_object_get_shutdown__doc__ );
   docset_helper_add( docset, ssl_object_get_ciphers__doc__ );
   docset_helper_add( docset, ssl_object_set_ciphers__doc__ );
   docset_helper_add( docset, ssl_object_get_cipher__doc__ );
   docset_helper_add( docset, ssl_object_set_verify_mode__doc__ );

   // x509 documentation
   docset_helper_add( docset, x509type__doc__ );
   docset_helper_add( docset, X509_object_pem_write__doc__ );
   docset_helper_add( docset, X509_object_sign__doc__ );
   docset_helper_add( docset, X509_object_set_public_key__doc__ );
   docset_helper_add( docset, X509_object_get_version__doc__ );
   docset_helper_add( docset, X509_object_set_version__doc__ );
   docset_helper_add( docset, X509_object_get_serial__doc__ );
   docset_helper_add( docset, X509_object_set_serial__doc__ );
   docset_helper_add( docset, X509_object_get_issuer__doc__ );
   docset_helper_add( docset, X509_object_set_issuer__doc__ );
   docset_helper_add( docset, X509_object_get_subject__doc__ );
   docset_helper_add( docset, X509_object_set_subject__doc__ );
   docset_helper_add( docset, X509_object_get_not_before__doc__ );
   docset_helper_add( docset, X509_object_set_not_before__doc__ );
   docset_helper_add( docset, X509_object_get_not_after__doc__ );
   docset_helper_add( docset, X509_object_set_not_after__doc__ );
   docset_helper_add( docset, x509_object_pprint__doc__ );

   // x509_crl documentation
   docset_helper_add( docset, x509_crltype__doc__ );
   docset_helper_add( docset, x509_crl_object_pem_write__doc__ );
   docset_helper_add( docset, x509_crl_object_get_version__doc__ );
   docset_helper_add( docset, x509_crl_object_set_version__doc__ );
   docset_helper_add( docset, x509_crl_object_get_issuer__doc__ );
   docset_helper_add( docset, x509_crl_object_set_issuer__doc__ );
   docset_helper_add( docset, x509_crl_object_get_this_update__doc__ );
   docset_helper_add( docset, x509_crl_object_set_this_update__doc__ );
   docset_helper_add( docset, x509_crl_object_get_next_update__doc__ );
   docset_helper_add( docset, x509_crl_object_set_next_update__doc__ );
   docset_helper_add( docset, x509_crl_object_get_revoked__doc__ );
   docset_helper_add( docset, x509_crl_object_set_revoked__doc__ );
   docset_helper_add( docset, x509_crl_object_verify__doc__ );
   docset_helper_add( docset, x509_crl_object_sign__doc__ );
   docset_helper_add( docset, x509_crl_object_pprint__doc__ );

   // x509_revoked documentation
   docset_helper_add( docset, x509_revokedtype__doc__ );
   docset_helper_add( docset, x509_revoked_object_get_date__doc__ );
   docset_helper_add( docset, x509_revoked_object_set_date__doc__ );
   docset_helper_add( docset, x509_revoked_object_get_serial__doc__ );
   docset_helper_add( docset, x509_revoked_object_set_serial__doc__ );

   // x509_store documentation
   docset_helper_add( docset, x509_storetype__doc__ );
   docset_helper_add( docset, x509_store_object_verify__doc__ );
   docset_helper_add( docset, x509_store_object_verify_chain__doc__ );
   docset_helper_add( docset, x509_store_object_add_trust__doc__ );
   docset_helper_add( docset, x509_store_object_add_crl__doc__ );

   // digest documentation
   docset_helper_add( docset, digesttype__doc__ );
   docset_helper_add( docset, digest_object_update__doc__ );
   docset_helper_add( docset, digest_object_copy__doc__ );
   docset_helper_add( docset, digest_object_digest__doc__ );

    // digest documentation
   docset_helper_add( docset, hmactype__doc__ );
   docset_helper_add( docset, hmac_object_update__doc__ );
   docset_helper_add( docset, hmac_object_copy__doc__ );
   docset_helper_add( docset, hmac_object_mac__doc__ );
    
   // symmetric documentation
   docset_helper_add( docset, symmetrictype__doc__ );
   docset_helper_add( docset, symmetric_object_encrypt_init__doc__ );
   docset_helper_add( docset, symmetric_object_decrypt_init__doc__ );
   docset_helper_add( docset, symmetric_object_update__doc__ );
   docset_helper_add( docset, symmetric_object_final__doc__ );

   // asymmetric documentation
   docset_helper_add( docset, asymmetrictype__doc__ );
   docset_helper_add( docset, asymmetric_object_pem_write__doc__ );
   docset_helper_add( docset, asymmetric_object_public_encrypt__doc__ );
   docset_helper_add( docset, asymmetric_object_public_decrypt__doc__ );
   docset_helper_add( docset, asymmetric_object_private_encrypt__doc__ );
   docset_helper_add( docset, asymmetric_object_private_decrypt__doc__ );
   docset_helper_add( docset, asymmetric_object_sign__doc__ );
   docset_helper_add( docset, asymmetric_object_verify__doc__ );

	doctuple = PyList_AsTuple( docset );
   Py_DECREF(docset);
 
   return Py_BuildValue("O", doctuple);

error:

   return NULL;
}

static struct PyMethodDef pow_module_methods[] = {
   {"Ssl",	         (PyCFunction)pow_module_new_ssl,	
                           METH_VARARGS,	pow_module_new_ssl__doc__},
   {"X509",       (PyCFunction)pow_module_new_x509, 
                           METH_VARARGS,  pow_module_new_x509__doc__}, 
   {"pemRead",       (PyCFunction)pow_module_pem_read, 
                           METH_VARARGS,  pow_module_pem_read__doc__}, 
   {"Digest",     (PyCFunction)pow_module_new_digest,
                           METH_VARARGS,  pow_module_new_digest__doc__}, 
   {"Hmac",     (PyCFunction)pow_module_new_hmac,
                           METH_VARARGS,  pow_module_new_hmac__doc__}, 
   {"Asymmetric",  (PyCFunction)pow_module_new_asymmetric,     
                           METH_VARARGS,  pow_module_new_asymmetric__doc__}, 
   {"Symmetric",   (PyCFunction)pow_module_new_symmetric,     
                           METH_VARARGS,  pow_module_new_symmetric__doc__}, 
   {"X509Store",  (PyCFunction)pow_module_new_x509_store,  
                           METH_VARARGS,  pow_module_new_x509_store__doc__}, 
   {"X509Crl",  (PyCFunction)pow_module_new_x509_crl,  
                           METH_VARARGS,  pow_module_new_x509_crl__doc__}, 
   {"X509Revoked",  (PyCFunction)pow_module_new_x509_revoked,  
                           METH_VARARGS,  pow_module_new_x509_revoked__doc__}, 
   {"getError",      (PyCFunction)pow_module_get_error,	
                           METH_VARARGS,	pow_module_get_error__doc__},
   {"clearError",    (PyCFunction)pow_module_clear_error,	
                           METH_VARARGS,	pow_module_clear_error__doc__},
   {"seed",    (PyCFunction)pow_module_seed,	
                           METH_VARARGS,	pow_module_seed__doc__},
   {"add",    (PyCFunction)pow_module_add,	
                           METH_VARARGS,	pow_module_add__doc__},
   {"readRandomFile",    (PyCFunction)pow_module_read_random_file,	
                           METH_VARARGS,	pow_module_read_random_file__doc__},
   {"writeRandomFile",    (PyCFunction)pow_module_write_random_file,	
                           METH_VARARGS,	pow_module_write_random_file__doc__},
   {"__doclist__",    (PyCFunction)pow_module___doclist__,	
                           METH_VARARGS,	pow_module___doclist____doc__},
 
	{NULL,	 (PyCFunction)NULL, 0, NULL}		/* sentinel */
};
/*========== module functions ==========*/


/*==========================================================================*/
void
initPOW(void)
{
	PyObject *m, *d;

	m = Py_InitModule4("POW", pow_module_methods,
		pow_module__doc__,
		(PyObject*)NULL,PYTHON_API_VERSION);

	d = PyModule_GetDict(m);
	SSLErrorObject = PyString_FromString("POW.SSLError");
	PyDict_SetItemString(d, "SSLError", SSLErrorObject);

   // constants for SSL_get_error()
   install_int_const( d, "SSL_ERROR_NONE",            SSL_ERROR_NONE );
   install_int_const( d, "SSL_ERROR_ZERO_RETURN",     SSL_ERROR_ZERO_RETURN );
   install_int_const( d, "SSL_ERROR_WANT_READ",       SSL_ERROR_WANT_READ );
   install_int_const( d, "SSL_ERROR_WANT_WRITE",      SSL_ERROR_WANT_WRITE );
   install_int_const( d, "SSL_ERROR_WANT_X509_LOOKUP",SSL_ERROR_WANT_X509_LOOKUP );
   install_int_const( d, "SSL_ERROR_SYSCALL",         SSL_ERROR_SYSCALL );
   install_int_const( d, "SSL_ERROR_SSL",             SSL_ERROR_SSL );

   // constants for different types of connection methods
   install_int_const( d, "SSLV2_SERVER_METHOD",       SSLV2_SERVER_METHOD );
   install_int_const( d, "SSLV2_CLIENT_METHOD",       SSLV2_CLIENT_METHOD );
   install_int_const( d, "SSLV2_METHOD",              SSLV2_METHOD );
   install_int_const( d, "SSLV3_SERVER_METHOD",       SSLV3_SERVER_METHOD );
   install_int_const( d, "SSLV3_CLIENT_METHOD",       SSLV3_CLIENT_METHOD );
   install_int_const( d, "SSLV3_METHOD",              SSLV3_METHOD );
   install_int_const( d, "SSLV23_SERVER_METHOD",      SSLV23_SERVER_METHOD );
   install_int_const( d, "SSLV23_CLIENT_METHOD",      SSLV23_CLIENT_METHOD );
   install_int_const( d, "SSLV23_METHOD",             SSLV23_METHOD );
   install_int_const( d, "TLSV1_SERVER_METHOD",       TLSV1_SERVER_METHOD );
   install_int_const( d, "TLSV1_CLIENT_METHOD",       TLSV1_CLIENT_METHOD );
   install_int_const( d, "TLSV1_METHOD",              TLSV1_METHOD );

   install_int_const( d, "SSL_NO_SHUTDOWN",           0 );
   install_int_const( d, "SSL_SENT_SHUTDOWN",         SSL_SENT_SHUTDOWN );
   install_int_const( d, "SSL_RECIEVED_SHUTDOWN",     SSL_RECEIVED_SHUTDOWN );

   // ssl verification mode
   install_int_const( d, "SSL_VERIFY_NONE",           SSL_VERIFY_NONE );
   install_int_const( d, "SSL_VERIFY_PEER",           SSL_VERIFY_PEER );

   // object format types
   install_int_const( d, "LONGNAME_FORMAT",           LONGNAME_FORMAT );
   install_int_const( d, "SHORTNAME_FORMAT",          SHORTNAME_FORMAT );

   // PEM encoded types
#ifndef NO_RSA
   install_int_const( d, "RSA_PUBLIC_KEY",            RSA_PUBLIC_KEY );
   install_int_const( d, "RSA_PRIVATE_KEY",           RSA_PRIVATE_KEY );
#endif
#ifndef NO_DSA
   install_int_const( d, "DSA_PUBLIC_KEY",            DSA_PUBLIC_KEY );
   install_int_const( d, "DSA_PRIVATE_KEY",           DSA_PRIVATE_KEY );
#endif
#ifndef NO_DH
   install_int_const( d, "DH_PUBLIC_KEY",             DH_PUBLIC_KEY );
   install_int_const( d, "DH_PRIVATE_KEY",            DH_PRIVATE_KEY );
#endif
   install_int_const( d, "X509_CERTIFICATE",          X509_CERTIFICATE );
   install_int_const( d, "X509_CRL",                  X_X509_CRL );

   // asymmetric ciphers
#ifndef NO_RSA
   install_int_const( d, "RSA_CIPHER",                RSA_CIPHER );
#endif
#ifndef NO_DSA
   install_int_const( d, "DSA_CIPHER",                DSA_CIPHER );
#endif
#ifndef NO_DH
   install_int_const( d, "DH_CIPHER",                 DH_CIPHER );
#endif

   // symmetric ciphers
#ifndef NO_DES
   install_int_const( d, "DES_ECB",                   DES_ECB );
   install_int_const( d, "DES_EDE",                   DES_EDE );
   install_int_const( d, "DES_EDE3",                  DES_EDE3 );
   install_int_const( d, "DES_CFB",                   DES_CFB );
   install_int_const( d, "DES_EDE_CFB",               DES_EDE_CFB );
   install_int_const( d, "DES_EDE3_CFB",              DES_EDE3_CFB );
   install_int_const( d, "DES_OFB",                   DES_OFB );
   install_int_const( d, "DES_EDE_OFB",               DES_EDE_OFB );
   install_int_const( d, "DES_EDE3_OFB",              DES_EDE3_OFB );
   install_int_const( d, "DES_CBC",                   DES_CBC );
   install_int_const( d, "DES_EDE_CBC",               DES_EDE_CBC );
   install_int_const( d, "DES_EDE3_CBC",              DES_EDE3_CBC );
   install_int_const( d, "DESX_CBC",                  DESX_CBC );
#endif
#ifndef NO_RC4
   install_int_const( d, "RC4",                       RC4 );
   install_int_const( d, "RC4_40",                    RC4_40 );
#endif
#ifndef NO_IDEA
   install_int_const( d, "IDEA_ECB",                  IDEA_ECB );
   install_int_const( d, "IDEA_CFB",                  IDEA_CFB );
   install_int_const( d, "IDEA_OFB",                  IDEA_OFB );
   install_int_const( d, "IDEA_CBC",                  IDEA_CBC );
#endif
#ifndef NO_RC2
   install_int_const( d, "RC2_ECB",                   RC2_ECB );
   install_int_const( d, "RC2_CBC",                   RC2_CBC );
   install_int_const( d, "RC2_40_CBC",                RC2_40_CBC );
   install_int_const( d, "RC2_CFB",                   RC2_CFB );
   install_int_const( d, "RC2_OFB",                   RC2_OFB );
#endif
#ifndef NO_BF
   install_int_const( d, "BF_ECB",                    BF_ECB );
   install_int_const( d, "BF_CBC",                    BF_CBC );
   install_int_const( d, "BF_CFB",                    BF_CFB );
   install_int_const( d, "BF_OFB",                    BF_OFB );
#endif
   install_int_const( d, "CAST5_ECB",                 CAST5_ECB );
   install_int_const( d, "CAST5_CBC",                 CAST5_CBC );
   install_int_const( d, "CAST5_CFB",                 CAST5_CFB );
   install_int_const( d, "CAST5_OFB",                 CAST5_OFB );
#ifndef NO_RC5_32_12_16
   install_int_const( d, "RC5_32_12_16_CBC",          RC5_32_12_16_CBC );
   install_int_const( d, "RC5_32_12_16_CFB",          RC5_32_12_16_CFB );
   install_int_const( d, "RC5_32_12_16_ECB",          RC5_32_12_16_ECB );
   install_int_const( d, "RC5_32_12_16_OFB",          RC5_32_12_16_OFB );
#endif

   // message digests
   install_int_const( d, "MD2_DIGEST",                MD2_DIGEST );
   install_int_const( d, "MD5_DIGEST",                MD5_DIGEST );
   install_int_const( d, "SHA_DIGEST",                SHA_DIGEST );
   install_int_const( d, "SHA1_DIGEST",               SHA1_DIGEST );
   install_int_const( d, "RIPEMD160_DIGEST",          RIPEMD160_DIGEST );

   // initialise library
   SSL_library_init();
   OpenSSL_add_all_algorithms();
   OpenSSL_add_all_ciphers();
   OpenSSL_add_all_digests();

   // load error strings
   SSL_load_error_strings();

	if (PyErr_Occurred())
		Py_FatalError("can't initialize module pow");
}
/*==========================================================================*/
