/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/**************************** motion estimation shape padding ****************/

/*  pad                                                                      */
/*                                                                           */
/*  Description:                                                             */
/*    Perform repetitive padding of arbitrary shape for motion estimation.   */
/*                                                                           */
/*  Arguments:                                                               */
/*    int width: width of the frame                                          */
/*    int height: height of the frame                                        */
/*    fame_yuv_t frame: the frame to pad                                     */
/*    unsigned char *shape: shape of the frame                               */
/*    unsigned char *padded: temporary shape buffer                          */
/*    fame_encoder_t *encoder: the encoder                                   */
/*    fame_box_t box: bounding box                                           */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void inline pad(int width,
		       int height,
		       fame_yuv_t *frame,
		       unsigned char *shape,
		       unsigned char *padded,
		       fame_box_t *box)
{
  int x, y, r1, r2, e;
  unsigned char *s, *sp;
  unsigned char *Y, *U, *V;
  unsigned char p1, p2;
  unsigned char ps;
  unsigned char p1u, p2u;
  unsigned char p1v, p2v;
  int bx, by, bw, bh;
  int pitch;

  pitch = width;

  memcpy(padded, shape, width*height);

  bx = box->x;
  by = box->y;
  bw = box->w;
  bh = box->h;

  /* horizontal padding Y */
  s = shape + by*pitch;
  sp = padded + by*pitch;
  Y = frame->y + by*pitch;
  for(y = by; y < bh; y++)
  {
    x = bx;
    while(x < bw)
    {
      p1 = p2 = 128;
      ps = 0;
      e = fame_min(x+32, bw);
      r2 = x;
      for(r1 = x; r1 < e; r1++)
	if(s[r1]) {
	  p1 = Y[r1];
	  p2 = p1;
	  ps = 255;
	  for(r2 = r1 + 1; r2 < bw; r2++)
	    if(s[r2]) {
	      p2 = (unsigned char) (((short) p1 + (short) Y[r2]) >> 1);
	      break;
	    }
	  if(p1 == p2)
	    r2 = fame_min((r1 & (~15))+32, r2);
	  break;
	}
      for(; x < r1; x++) {
	Y[x] = p1;
	/* compatibility with extended padding */
	if((x & (~15)) == (r1 & (~15)) &&
	   (sp < padded+16*pitch || sp[-16*pitch] == 0))
	  sp[x] = ps;
      }
      for(; x < r2; x++) {
	Y[x] = p2;
	sp[x] = ps;
      }
    }
    s += pitch;
    sp += pitch;
    Y += pitch;
  }

  /* vertical padding Y */
  s = padded + bx;
  Y = frame->y + bx;
  for(x = bx; x < bw; x++)
  {
    y = by;
    while(y < bh)
    {
      p1 = p2 = 128;
      e = fame_min(y+32, bh);
      r2 = y;
      for(r1 = y; r1 < e; r1++)
	if(s[r1*pitch]) {
	  p1 = Y[r1*pitch];
	  p2 = p1;
	  for(r2 = r1 + 1; r2 < bh; r2++)
	    if(s[r2*pitch]) {
	      p2 = (unsigned char) (((short) p1 + (short) Y[r2*pitch]) >> 1);
	      break;
	    }
	  if(p1 == p2)
	    r2 = fame_min((r1 & (~15))+32, r2);
	  break;
	}
      for(; y < r1; y++)
	Y[y*pitch] = p1;
      for(; y < r2; y++)
	Y[y*pitch] = p2;
    }
    s ++;
    Y ++;
  }
  


  bx >>= 1;
  by >>= 1;
  bw >>= 1;
  bh >>= 1;

  /* horizontal padding Cr, Cb */
  s = shape + (by<<1)*pitch;
  U = frame->u + by*(pitch>>1);
  V = frame->v + by*(pitch>>1);
  for(y = by; y < bh; y++)
  {
    x = bx;
    while(x < bw)
    {
      p1u = p2u = 128;
      p1v = p2v = 128;
      e = fame_min(x+16, bw);
      r2 = x;
      for(r1 = x; r1 < e; r1++)
	if(s[(r1 << 1)] | s[(r1 << 1) + 1] |
	   s[pitch + (r1 << 1)] | s[pitch + (r1 << 1) + 1]) {
	  p1u = U[r1];
	  p1v = V[r1];
	  p2u = p1u;
	  p2v = p1v;
	  for(r2 = r1 + 1; r2 < bw; r2++)
	    if(s[(r2 << 1)] | s[(r2 << 1) + 1] |
	       s[pitch + (r2 << 1)] | s[pitch + (r2 << 1) + 1]) {
	      p2u = (unsigned char) (((short) p1u + (short) U[r2]) >> 1);
	      p2v = (unsigned char) (((short) p1v + (short) V[r2]) >> 1);
	      break;
	    }
	  if(p1u == p2u && p1v == p2v)
	    r2 = fame_min((r1 & (~7))+16, r2);
	  break;
	}
      for(; x < r1; x++) {
	U[x] = p1u;
	V[x] = p1v;
      }
      for(; x < r2; x++) {
	U[x] = p2u;
	V[x] = p2v;
      }
    }
    s += pitch << 1;
    U += pitch >> 1;
    V += pitch >> 1;
  }

  /* vertical padding Cr, Cb */
  s = padded + (bx<<1);
  U = frame->u + bx;
  V = frame->v + bx;
  for(x = bx; x < bw; x++)
  {
    y = by;
    while(y < bh)
    {
      p1u = p2u = 128;
      p1v = p2v = 128;
      e = fame_min(y+16, bh);
      r2 = y;
      for(r1 = y; r1 < e; r1++)
	if(s[(r1 << 1)*pitch] | s[(r1 << 1)*pitch + pitch] |
	   s[(r1 << 1)*pitch + 1] | s[(r1 << 1)*pitch + pitch + 1]) {
	  p1u = U[r1*(pitch >> 1)];
	  p1v = V[r1*(pitch >> 1)];
	  p2u = p1u;
	  p2v = p1v;
	  for(r2 = r1 + 1; r2 < bh; r2++)
	    if(s[(r2 << 1)*pitch] | s[(r2 << 1)*pitch + pitch] |
	       s[(r2 << 1)*pitch + 1] | s[(r2 << 1)*pitch + pitch + 1]) {
	      p2u = (unsigned char) (((short)p1u+(short)U[r2*(pitch>>1)])>>1);
	      p2v = (unsigned char) (((short)p1v+(short)V[r2*(pitch>>1)])>>1);
	      break;
	    }
	  if(p1u == p2u && p1v == p2v)
	    r2 = fame_min((r1 & (~7))+16, r2);
	  break;
	}
      for(; y < r1; y++) {
	U[y*(pitch >> 1)] = p1u;
	V[y*(pitch >> 1)] = p1v;
      }
      for(; y < r2; y++) {
	U[y*(pitch >> 1)] = p2u;
	V[y*(pitch >> 1)] = p2v;
      }
    }
    s += 2;
    U ++;
    V ++;
  }
}
