/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Damien Vincent

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include "fame.h"
#include "fame_rate.h"
#ifdef HAS_MMX
#include "mad_mmx.h"
#else
#include "mad_int.h"
#endif

static void rate_init(fame_rate_t *rate,
		      int mb_width,
		      int mb_height,
		      int bitrate,
		      char *coding,
		      unsigned int flags);
static void rate_close(fame_rate_t *rate);
static void rate_enter(fame_rate_t *rate,
		       fame_yuv_t **ref,
		       fame_yuv_t *current,
		       unsigned char *shape,
		       char coding);
static int rate_global_estimation(fame_rate_t *rate);
static int rate_local_estimation(fame_rate_t *rate,
				 int mb_x, int mb_y,
				 short blocks[6][64]);
static void rate_leave(fame_rate_t *rate,
		       int spent);

FAME_CONSTRUCTOR(fame_rate_t)
{
  FAME_OBJECT(this)->name = "rate estimation";
  FAME_RATE(this)->init = rate_init;
  FAME_RATE(this)->close = rate_close;
  FAME_RATE(this)->enter = rate_enter;
  FAME_RATE(this)->global_estimation = rate_global_estimation;
  FAME_RATE(this)->local_estimation = rate_local_estimation;
  FAME_RATE(this)->leave = rate_leave;
  FAME_RATE(this)->flags = 0xffffffff;
  return(this);
}

/*  rate_init                                                                */
/*                                                                           */
/*  Description:                                                             */
/*    Initialise rate estimation.                                            */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_rate_t *rate: the rate estimation                                 */
/*    int mb_width: width in macroblocks                                     */
/*    int mb_height: height in macroblocks                                   */
/*                                                                           */
/*  Return value:                                                            */
/*    Rate.                                                                  */

static void rate_init(fame_rate_t *rate,
		      int mb_width,
		      int mb_height,
		      int bitrate,
		      char *coding,
		      unsigned int flags)
{
  int ni, np;
  int ratio;
  int i;

  rate->mb_width = mb_width;
  rate->mb_height = mb_height;
  rate->flags &= flags;
  rate->available = 0;
  rate->global_scale = 8;
  ni = np = 0;
  for(i = 0; i < strlen(coding); i++) {
    switch(coding[i]) {
      case 'I': ni++; break;
      case 'P': np++; break;
    }
  }
  ratio = 1;
  rate->P_bits = bitrate * (np + ni) / (np + ratio * ni);
  rate->I_bits = ratio * rate->P_bits;
  rate->quant_step = 16;
  rate->I_factor = (3 << 8);
  rate->P_factor = (6 << 8);
}

/*  rate_close                                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Release rate estimation.                                               */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_rate_t *rate: the rate estimation                                 */
/*                                                                           */
/*  Return value:                                                            */
/*    Rate.                                                                  */

static void rate_close(fame_rate_t *rate)
{
}

/*  rate_enter                                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Prepare for a new frame.                                               */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_rate_t *rate: the rate estimation                                 */
/*    fame_yuv_t **ref: the reference frames (half-pel)                      */
/*    fame_yuv_t *current: the current frame                                 */
/*    unsigned char *shape: the current shape                                */
/*                                                                           */
/*  Return value:                                                            */
/*    Rate.                                                                  */

static void rate_enter(struct _fame_rate_t_ *rate,
		       fame_yuv_t **ref,
		       fame_yuv_t *current,
		       unsigned char *shape,
		       char coding)
{
  rate->ref = ref;
  rate->current = current;
  rate->shape = shape;
  rate->coding = coding;
  /* estimate activity */
  { 
    int bx, by;
    int a, p;
    unsigned long m;
    unsigned char *input;
      
    a = 0;
    p = rate->mb_width*16;
    input = rate->current->y;
    for(by = 0; by < rate->mb_height*2; by++) {
      for(bx = 0; bx < rate->mb_width*2; bx++) {
	mad_withoutmask(input, p, &m);
	a+=m;
	input+=8;
      }
      input += (p << 3) - p;
    }
    rate->activity = a;
  }

  switch(coding) {
    case 'I': rate->available += rate->I_bits; break;
    case 'P': rate->available += rate->P_bits; break;
  };
}

/*  rate_leave                                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Finish estimating a frame.                                             */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_rate_t *rate: the rate estimation                                 */
/*                                                                           */
/*  Return value:                                                            */
/*    Rate.                                                                  */

static void rate_leave(fame_rate_t *rate, int spent)
{
  switch(rate->coding) {
    case 'I' : rate->I_factor = (rate->activity << 8)/(rate->global_scale*spent); break;
    case 'P' : rate->P_factor = (rate->activity << 8)/(rate->global_scale*spent); break;
  }
  rate->spent = spent;
  rate->available -= spent;
}

static int rate_global_estimation(fame_rate_t *rate)
{
  int quant_scale;
  unsigned int factor;

  switch(rate->coding) {
    case 'I': factor = rate->I_factor; break;
    case 'P': factor = rate->P_factor; break;
    default: factor = 0;
  }
  if(rate->available > 0 && factor > 0) 
    quant_scale = (rate->activity << 8)/(factor*rate->available);
  else
    quant_scale = 31;
 
  if(quant_scale < rate->global_scale/2) quant_scale = rate->global_scale/2;
  if(quant_scale > rate->global_scale*2) quant_scale = rate->global_scale*2;
  if(quant_scale < 2) quant_scale = 2;
  if(quant_scale > 31) quant_scale = 31;
  rate->global_scale = quant_scale;
  return(quant_scale);
}

static int rate_local_estimation(fame_rate_t *rate,
				 int mb_x, int mb_y,
				 short blocks[6][64])
{
  return(rate->global_scale);
}
