/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Damien Vincent

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include "fame.h"
#include "fame_motion.h"
#if defined(HAS_MMX)
#include "mad_mmx.h"
#include "mae_mmx.h"
#else
#include "mad_int.h"
#include "mae_int.h"
#endif
    
static void  motion_init(fame_motion_t *motion,
		       int mb_width,
		       int mb_height,
		       unsigned int flags);
static void  motion_close(fame_motion_t *motion);
static void  motion_enter(fame_motion_t *motion,
			fame_yuv_t **ref,
			fame_yuv_t *current,
			unsigned char *shape,
			int search_range);
static void motion_leave(fame_motion_t *motion);

FAME_CONSTRUCTOR(fame_motion_t)
{
  FAME_OBJECT(this)->name = "motion estimation";
  FAME_MOTION(this)->init = motion_init;
  FAME_MOTION(this)->close = motion_close;
  FAME_MOTION(this)->enter = motion_enter;
  FAME_MOTION(this)->estimation = NULL;
  FAME_MOTION(this)->leave = motion_leave;
  FAME_MOTION(this)->flags = 0xffffffff;
  return(this);
}

/*  motion_init                                                              */
/*                                                                           */
/*  Description:                                                             */
/*    Initialise motion estimation.                                          */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_motion_t *motion: the motion estimation                           */
/*    int mb_width: width in macroblocks                                     */
/*    int mb_height: height in macroblocks                                   */
/*                                                                           */
/*  Return value:                                                            */
/*    Motion.                                                                */

static void motion_init(fame_motion_t *motion,
			int mb_width,
			int mb_height,
			unsigned int flags)
{
  motion->mb_width = mb_width;
  motion->mb_height = mb_height;
  motion->flags &= flags;
}

/*  motion_close                                                             */
/*                                                                           */
/*  Description:                                                             */
/*    Release motion estimation.                                             */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_motion_t *motion: the motion estimation                           */
/*                                                                           */
/*  Return value:                                                            */
/*    Motion.                                                                */

static void motion_close(fame_motion_t *motion)
{
}

/*  motion_enter                                                             */
/*                                                                           */
/*  Description:                                                             */
/*    Prepare for a new frame.                                               */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_motion_t *motion: the motion estimation                           */
/*    fame_yuv_t **ref: the reference frames (half-pel)                      */
/*    fame_yuv_t *current: the current frame                                 */
/*    unsigned char *shape: the current shape                                */
/*                                                                           */
/*  Return value:                                                            */
/*    Motion.                                                                */

static void motion_enter(struct _fame_motion_t_ *motion,
		       fame_yuv_t **ref,
		       fame_yuv_t *current,
		       unsigned char *shape,
		       int search_range)
{
  motion->ref = ref;
  motion->current = current;
  motion->search_range = search_range;
  motion->shape = shape;

  if(shape == NULL)
    motion->MAE8x8 = MAE8x8_withoutmask;
  else
    motion->MAE8x8 = MAE8x8_withmask;
}

/*  motion_leave                                                             */
/*                                                                           */
/*  Description:                                                             */
/*    Finish estimating a frame.                                             */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_motion_t *motion: the motion estimation                           */
/*                                                                           */
/*  Return value:                                                            */
/*    Motion.                                                                */

static void motion_leave(fame_motion_t *motion)
{
}
