/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/**************************** mpeg encoder ***********************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "fame.h"
#include "fame_encoder.h"
#include "fame_encoder_mpeg.h"
#include "table_scale.h"
#if defined(HAS_MMX)
#define arch_enter_state()
#define arch_leave_state() asm("emms")
#include "transpose_mmx.h"
#include "dct_mmx.h"
#include "quantise_mmx.h"
#include "fetch_mmx.h"
#else
#define arch_enter_state() 
#define arch_leave_state() 
#include "dct_float.h"
#include "quantise_float.h"
#include "fetch_float.h"
#endif

static void mpeg_init(fame_encoder_t *encoder,
		      int width,
		      int height,
		      unsigned char *intra_quantisation_table,
		      unsigned char *inter_quantisation_table,
		      fame_mismatch_t mismatch_type);
static void mpeg_enter(fame_encoder_t *encoder,
			fame_yuv_t **past_ref,
			fame_yuv_t **new_ref,
			fame_yuv_t **future_ref,
			fame_yuv_t *yuv,
			unsigned char *shape);
static void mpeg_set_quantisation(fame_encoder_t *encoder,
				  unsigned char quant_scale,
				  unsigned char intra_y_scale,
				  unsigned char intra_c_scale);
static void mpeg_encode_intra_mb(fame_encoder_t *encoder,
				  short x,
				  short y,
				  short *blocks[6],
				  fame_bab_t bab_type);
static void mpeg_encode_inter_mb(fame_encoder_t *encoder,
				  short x,
				  short y,
				  short *blocks[6],
				  fame_motion_vector_t *forward,
				  fame_motion_vector_t *backward,
				  fame_motion_coding_t motion_coding,
				  fame_bab_t bab_type);
static void mpeg_leave(fame_encoder_t *encoder);
static void mpeg_close(fame_encoder_t *encoder);

FAME_CONSTRUCTOR(fame_encoder_mpeg_t)
{
  FAME_OBJECT(this)->name = "MPEG encoder";
  FAME_ENCODER(this)->init = mpeg_init;
  FAME_ENCODER(this)->enter = mpeg_enter;
  FAME_ENCODER(this)->set_quantisation = mpeg_set_quantisation;
  FAME_ENCODER(this)->encode_intra_mb = mpeg_encode_intra_mb;
  FAME_ENCODER(this)->encode_inter_mb = mpeg_encode_inter_mb;
  FAME_ENCODER(this)->leave = mpeg_leave;
  FAME_ENCODER(this)->close = mpeg_close;
  return(this);
}

/*  mpeg_init                                                                */
/*                                                                           */
/*  Description:                                                             */
/*    Initialize the encoder.                                                */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_encoder_t *encoder: the encoder to initialize                     */
/*    int width: width of the frame                                          */
/*    int height: height of the frame                                        */
/*    unsigned char *intra_quantisation_table: quantisation matrix for intra */
/*    unsigned char *inter_quantisation_table: quantisation matrix for inter */
/*    fame_mismatch_t mismatch_type: type of mismatch control                */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_init(fame_encoder_t *encoder,
		      int width,
		      int height,
		      unsigned char *intra_quantisation_table,
		      unsigned char *inter_quantisation_table,
		      fame_mismatch_t mismatch_type)
{
  fame_encoder_mpeg_t *encoder_mpeg = FAME_ENCODER_MPEG(encoder);

  /* set width and height */
  encoder_mpeg->width = width;
  encoder_mpeg->height = height;

  /* allocate padded shape buffer */
  encoder_mpeg->padded = (unsigned char *) malloc(encoder_mpeg->width*
						   encoder_mpeg->height);
  encoder_mpeg->intra_quantisation_table = intra_quantisation_table;
  encoder_mpeg->inter_quantisation_table = inter_quantisation_table;
  encoder_mpeg->mismatch = mismatch_type;
}

/*  mpeg_enter                                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Start encoding a new picture.                                          */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_encoder_t *encoder: the encoder                                   */
/*    fame_yuv_t **past_ref: past reference images                           */
/*    fame_yuv_t **new_ref: new reconstructed reference images               */
/*    fame_yuv_t **future_ref: future reference images                       */
/*    fame_yuv_t *yuv: source image                                          */
/*    unsigned char *shape: shape binary mask                                */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */
  
static void mpeg_enter(fame_encoder_t *encoder,
			fame_yuv_t **past_ref,
			fame_yuv_t **new_ref,
			fame_yuv_t **future_ref,
			fame_yuv_t *yuv,
			unsigned char *shape)
{
  fame_encoder_mpeg_t *encoder_mpeg = FAME_ENCODER_MPEG(encoder);

  /* Make pointers on the input frame and reference frame */
  encoder_mpeg->input = yuv;
  encoder_mpeg->past_ref = past_ref;
  encoder_mpeg->new_ref = new_ref;
  encoder_mpeg->future_ref = future_ref;
  encoder_mpeg->shape = shape;
  arch_enter_state();
}

/*  mpeg_set_quantisation                                                    */
/*                                                                           */
/*  Description:                                                             */
/*    Updates the matrices used for quantisation.                            */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_encoder_t *encoder: the encoder                                   */
/*    unsigned char quant_scale: quantisation scale                          */
/*    unsigned char intra_y_scale: intra DC scaler for Y component           */
/*    unsigned char intra_c_scale: intra DC scaler for U and V components    */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_set_quantisation(fame_encoder_t *encoder,
				  unsigned char quant_scale,
				  unsigned char intra_y_scale,
				  unsigned char intra_c_scale)
{
  fame_encoder_mpeg_t *encoder_mpeg = FAME_ENCODER_MPEG(encoder);
  unsigned char *iqtable = encoder_mpeg->intra_quantisation_table;
  unsigned char *niqtable = encoder_mpeg->inter_quantisation_table;
  dct_t *yiqmatrix = encoder_mpeg->yiqmatrix;
  dct_t *ciqmatrix = encoder_mpeg->ciqmatrix;
  dct_t *niqmatrix = encoder_mpeg->niqmatrix;
  dct_t *yidqmatrix = encoder_mpeg->yidqmatrix;
  dct_t *cidqmatrix = encoder_mpeg->cidqmatrix;
  dct_t *nidqmatrix = encoder_mpeg->nidqmatrix;
  dct_t *psmatrix = encoder_mpeg->psmatrix;
  int i;

  /* initialize quantisation matrices */
  encoder_mpeg->quant_scale = quant_scale;

  /* compute the intra quantisation and dequantisation DC scaler */
#ifdef HAS_MMX
  asm("emms");
  yiqmatrix[0] = (short) ((double)(1UL<<16)*postscale[0]/intra_y_scale); /*16*/
  yidqmatrix[0] =  (short) (intra_y_scale << 3);
  ciqmatrix[0] = (short) ((double)(1UL<<16)*postscale[0]/intra_c_scale); /*16*/
  cidqmatrix[0] =  (short) (intra_c_scale << 3);
#else
  yiqmatrix[0] = postscale[0] / intra_y_scale;
  yidqmatrix[0] =  intra_y_scale;
  ciqmatrix[0] = postscale[0] / intra_c_scale;
  cidqmatrix[0] =  intra_c_scale;
#endif

  /* compute the intra quantisation and dequantisation matrix */
  for(i = 1; i < 64; i++)
  {
#ifdef HAS_MMX
    yiqmatrix[i] = ciqmatrix[i] =
      (short) ((double)(1UL<<19)*postscale[i] / (quant_scale*iqtable[i]));
    yidqmatrix[i] = cidqmatrix[i] = (short) quant_scale*iqtable[i];
#else
    yiqmatrix[i] = ciqmatrix[i] =
      8.0 * postscale[i] / (quant_scale * iqtable[i]);
    yidqmatrix[i] = cidqmatrix[i] = quant_scale*iqtable[i];
#endif
  }

  /* compute the inter quantisation and dequantisation matrix */
  for(i = 0; i < 64; i++)
  {
#ifdef HAS_MMX
    niqmatrix[i] = (short) ((double)(1UL<<19)*postscale[i]/
			   (quant_scale*niqtable[i]));
    nidqmatrix[i] = (short) quant_scale*niqtable[i];
    psmatrix[i] = (short) ((double)(1UL << 16) * prescale[i]);
#else
    niqmatrix[i] = 8.0 * postscale[i] / (quant_scale * niqtable[i]);
    nidqmatrix[i] = quant_scale*niqtable[i];
    psmatrix[i] = prescale[i];
#endif
  }		     
}
  
/*  mpeg_encode_intra_mb                                                    */
/*                                                                           */
/*  Description:                                                             */
/*    Encode an intra macroblock.                                            */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_encoder_t *encoder: the encoder                                   */
/*    bitbuffer_t *bb: a bit buffer to write the resulting encoded data to.  */
/*    short x: the x location of the macroblock in macroblock units          */
/*    short y: the y location of the macroblock in macroblock units          */
/*    short *blocks[6]: the DCT coded blocks                                 */
/*    fame_bab_t bab_type: binary alpha block type                           */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */
  
static void mpeg_encode_intra_mb(fame_encoder_t *encoder,
				  short x,
				  short y,
				  short *blocks[6],
				  fame_bab_t bab_type)
{
  fame_encoder_mpeg_t *encoder_mpeg = FAME_ENCODER_MPEG(encoder);
  unsigned long offset[6];
  int i, pitch;

  pitch = encoder_mpeg->width;

  /* Make offsets to blocks */
  offset[0] = (y << 4) * pitch + (x << 4);         /* Y(0,0) */
  offset[1] = offset[0] + 8;                       /* Y(0,1) */
  offset[2] = offset[0] + (pitch << 3);            /* Y(1,0) */
  offset[3] = offset[2] + 8;                       /* Y(1,1) */
  offset[4] = (y << 3) * (pitch >> 1) + (x << 3);  /* Cb     */
  offset[5] = (y << 3) * (pitch >> 1) + (x << 3);  /* Cr     */

  /* Encode blocks */
  for(i = 0; i < 6; i++)
    blocks[i] = encoder_mpeg->blocks[i];

  if(bab_type != bab_all_coded) 
  {
    for(i = 0; i < 4; i++) { /* Y */
      prefetch_Y_withmask(encoder_mpeg->input->y + offset[i],
			  encoder_mpeg->tmpblock,
			  encoder_mpeg->shape + offset[i],
			  pitch);
      dct(encoder_mpeg->tmpblock);
      quantise(encoder_mpeg->blocks[i],
	       encoder_mpeg->tmpblock,
	       encoder_mpeg->yiqmatrix);
    }
    /* U */
    prefetch_C_withmask(encoder_mpeg->input->u + offset[4],
			encoder_mpeg->tmpblock,
			encoder_mpeg->shape + offset[0], /*top left corner of mb*/
			pitch >> 1);
    dct(encoder_mpeg->tmpblock);
    quantise(encoder_mpeg->blocks[4],
	     encoder_mpeg->tmpblock,
	     encoder_mpeg->ciqmatrix);
    /* V */
    prefetch_C_withmask(encoder_mpeg->input->v + offset[5],
			encoder_mpeg->tmpblock,
			encoder_mpeg->shape + offset[0], /*top left corner of mb*/
			pitch >> 1);
    dct(encoder_mpeg->tmpblock);
    quantise(encoder_mpeg->blocks[5],
	     encoder_mpeg->tmpblock,
	     encoder_mpeg->ciqmatrix);
  } else {
    for(i = 0; i < 4; i++) { /* Y */
      prefetch_Y_withoutmask(encoder_mpeg->input->y + offset[i],
			     encoder_mpeg->tmpblock,
			     pitch);
      dct(encoder_mpeg->tmpblock);
      quantise(encoder_mpeg->blocks[i],
	       encoder_mpeg->tmpblock,
	       encoder_mpeg->yiqmatrix);
    }
    /* U */
    prefetch_C_withoutmask(encoder_mpeg->input->u + offset[4],
			   encoder_mpeg->tmpblock,
			   pitch >> 1);
    dct(encoder_mpeg->tmpblock);
    quantise(encoder_mpeg->blocks[4],
	     encoder_mpeg->tmpblock,
	     encoder_mpeg->ciqmatrix);
    /* V */
    prefetch_C_withoutmask(encoder_mpeg->input->v + offset[5],
			   encoder_mpeg->tmpblock,
			   pitch >> 1);
    dct(encoder_mpeg->tmpblock);
    quantise(encoder_mpeg->blocks[5],
	     encoder_mpeg->tmpblock,
	     encoder_mpeg->ciqmatrix);
  }
}

/*  mpeg_encode_inter_mb                                                    */
/*                                                                           */
/*  Description:                                                             */
/*    Encode an inter macroblock.                                            */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_encoder_t *encoder: the encoder                                   */
/*    bitbuffer_t *bb: a bit buffer to write the resulting encoded data to.  */
/*    short x: the x location of the macroblock in macroblock units          */
/*    short y: the y location of the macroblock in macroblock units          */
/*    short *blocks[6]: the DCT coded blocks                                 */
/*    fame_bab_t bab_type: binary alpha block type                           */
/*    fame_motion_vector_t *forward: forward motion vectors                  */
/*    fame_motion_vector_t *backward: backward motion vectors                */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */
  
static void mpeg_encode_inter_mb(fame_encoder_t *encoder,
				 short x,
				 short y,
				 short *blocks[6],
				 fame_motion_vector_t *forward,
				 fame_motion_vector_t *backward,
				 fame_motion_coding_t motion_coding,
				 fame_bab_t bab_type)
{
  fame_encoder_mpeg_t *encoder_mpeg = FAME_ENCODER_MPEG(encoder);
  unsigned long offset[6];
  signed long motion[6];
  signed long residual[6];
  int i, pitch;

  pitch = encoder_mpeg->width;

  /* Make offsets to blocks */
  offset[0] = (y << 4) * pitch + (x << 4);         /* Y(0,0) */
  offset[1] = offset[0] + 8;                       /* Y(0,1) */
  offset[2] = offset[0] + (pitch << 3);            /* Y(1,0) */
  offset[3] = offset[2] + 8;                       /* Y(1,1) */
  offset[4] = (y << 3) * (pitch >> 1) + (x << 3);  /* Cb     */
  offset[5] = (y << 3) * (pitch >> 1) + (x << 3);  /* Cr     */

  /* Compute motion offsets (motion is half-pixel coded) */
  for(i = 0; i < 4; i++) {
    /* full-pel motion */
    motion[i] = (forward[i].dy >> 1) * pitch + (forward[i].dx >> 1);
    /* half-pel motion */
    residual[i] = ((forward[i].dy & 1) << 1) | (forward[i].dx & 1);
  }
  for(i = 4; i < 6; i++) {
    /* full-pel motion */
    motion[i] = (forward[i].dy >> 1) * (pitch >> 1) + (forward[i].dx >> 1);
    /* half-pel motion */
    residual[i] = ((forward[i].dy & 1) << 1) | (forward[i].dx & 1);
  }

  /* Encode blocks */
  for(i = 0; i < 6; i++)
    blocks[i] = encoder_mpeg->blocks[i];

  /* might sound strange but yes, this is future_ref in case */
  /* of P frames */
  for(i = 0; i < 4; i++) {
    /* Y */
    if(forward[i].error < encoder_mpeg->quant_scale*16)
      blocks[i] = NULL;
    else {
      diff(encoder_mpeg->input->y + offset[i],
	   encoder_mpeg->future_ref[residual[i]]->y + offset[i] + motion[i],
	   encoder_mpeg->tmpblock,
	   pitch);
      dct(encoder_mpeg->tmpblock);
      quantise(encoder_mpeg->blocks[i],
	       encoder_mpeg->tmpblock,
	       encoder_mpeg->niqmatrix);
    }
  }
  /* U */
  /* TODO: skip block with error < quant_scale*16 */
  diff(encoder_mpeg->input->u + offset[4],
       encoder_mpeg->future_ref[residual[4]]->u + offset[4] + motion[4],
       encoder_mpeg->tmpblock,
       pitch >> 1);
  dct(encoder_mpeg->tmpblock);
  quantise(encoder_mpeg->blocks[4],
	   encoder_mpeg->tmpblock,
	   encoder_mpeg->niqmatrix);

  /* V */
  /* TODO: skip block with error < quant_scale*16 */
  diff(encoder_mpeg->input->v + offset[5],
       encoder_mpeg->future_ref[residual[5]]->v + offset[5] + motion[5],
       encoder_mpeg->tmpblock,
       pitch >> 1);
  dct(encoder_mpeg->tmpblock);
  quantise(encoder_mpeg->blocks[5],
	   encoder_mpeg->tmpblock,
	   encoder_mpeg->niqmatrix);
}

/*  mpeg_leave                                                              */
/*                                                                           */
/*  Description:                                                             */
/*    End the encoding of a picture.                                         */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_encoder_t *encoder: the encoder                                   */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_leave(fame_encoder_t *encoder)
{
  arch_leave_state();
}

/*  mpeg_close                                                              */
/*                                                                           */
/*  Description:                                                             */
/*    Release the encoder.                                                   */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_encoder_t *encoder: the encoder                                   */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_close(fame_encoder_t *encoder)
{
  fame_encoder_mpeg_t *encoder_mpeg = FAME_ENCODER_MPEG(encoder);

  /* free shape padding buffer */
  free(encoder_mpeg->padded);
}
