/*(GPL)
------------------------------------------------------------
   Kobo Deluxe - Wrapper for Sound Control
------------------------------------------------------------
 * Copyright (C) 2007 David Olofson
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "sound.h"

#include "kobo.h"
#include "kobolog.h"
#include "random.h"
#include "audio.h"

int	KOBO_sound::sounds_loaded = 0;
int	KOBO_sound::music_loaded = 0;
int	KOBO_sound::time = 0;
int	KOBO_sound::_period = 30;
int	KOBO_sound::sfx2d_tag = 0;
int	KOBO_sound::listener_x = 0;
int	KOBO_sound::listener_y = 0;
int	KOBO_sound::wrap_x = 0;
int	KOBO_sound::wrap_y = 0;
int	KOBO_sound::scale = 65536 / 1000;
int	KOBO_sound::panscale = 65536 / 700;


KOBO_sound::KOBO_sound()
{
}


KOBO_sound::~KOBO_sound()
{
	close();
}


void KOBO_sound::play(int ch, int wid, float pitch, float vol)
{
	audio_channel_control(ch, AVT_FUTURE, ACC_PATCH, wid);
	audio_channel_play(ch, 0, (int)((60<<16) * pitch), (int)(65536 * vol));
}


/*--------------------------------------------------
	Open/close
--------------------------------------------------*/


int KOBO_sound::load(int (*prog)(int pc, const char *msg), int force)
{
	int res;
	int save_to_disk = 0;
	const char *ap = fmap->get("SFX>>", FM_DIR);
	if(!ap)
	{
		log_printf(ELOG, "Couldn't find audio data directory!\n");
		return -1;
	}
	audio_set_path(ap);

	if(!sounds_loaded || force)
	{
		if(prog(5, "Loading sound effects"))
			return -999;
		res = -1;
		if(prefs->cached_sounds && !force)
		{
			res = audio_wave_load(0, "sfx_c.agw", 0);
			if(res < 0)
				save_to_disk = 1;
		}
		if(res < 0)
			res = audio_wave_load(0, "sfx.agw", 0);
		if(res >= 0)
			sounds_loaded = 1;
		else
			log_printf(ELOG, "Could not load sound effects!\n");
	}

	if(prefs->use_music && !music_loaded || force)
	{
		if(save_to_disk)
		{
			if(prog(20, "Loading music"))
				return -999;
		}
		else
		{
			if(prog(30, "Loading music"))
				return -999;
		}
		res = -1;
		if(prefs->cached_sounds && !force)
		{
			res = audio_wave_load(0, "music_c.agw", 0);
			if(res < 0)
				save_to_disk = 1;
		}
		if(res < 0)
			res = audio_wave_load(0, "music.agw", 0);
		if(res >= 0)
			music_loaded = 1;
		else
			log_printf(ELOG, "Could not load music!\n");
	}

	if(save_to_disk)
	{
		if(prog(60, "Preparing audio engine"))
			return -999;
	}
	else
	{
		if(prog(80, "Preparing audio engine"))
			return -999;
	}

	audio_wave_prepare(-1);

	if(save_to_disk)
	{
		if(prog(80, "Writing sounds to disk"))
			return -999;
		if(audio_wave_load(0, "save_waves.agw", 0) < 0)
			log_printf(ELOG, "Could not save sounds to disk!\n");
	}

	return prog(100, NULL);
}


void KOBO_sound::prefschange()
{
	/* Levels */
//	audio_master_volume((float)prefs->volume/100.0);
	audio_group_controlf(SOUND_GROUP_UI, ACC_VOLUME,
			(float)prefs->intro_vol/100.0);
	audio_group_controlf(SOUND_GROUP_UIMUSIC, ACC_VOLUME,
			(float)prefs->intro_vol/100.0);
	audio_group_controlf(SOUND_GROUP_SFX, ACC_VOLUME,
			(float)prefs->sfx_vol/100.0);
	audio_group_controlf(SOUND_GROUP_BGMUSIC, ACC_VOLUME,
			(float)prefs->music_vol/100.0);
	set_boost(prefs->vol_boost);
	audio_quality((audio_quality_t)prefs->mixquality);

	// Bus 7: Our "Master Reverb Bus"
	master_reverb((float)prefs->reverb/100.0);
}
	

int KOBO_sound::open()
{
	if(!prefs->use_sound)
	{
		log_printf(WLOG, "Sound disabled!\n");
		return 0;
	}

	if(audio_start(prefs->samplerate, prefs->latency, prefs->use_oss, prefs->cmd_midi,
			prefs->cmd_pollaudio) < 0)
	{
		log_printf(ELOG, "Couldn't initialize audio;"
				" disabling sound effects.\n");
		return -1;
	}

	// Channel grouping. We use only one chanel per group here, so we
	// just assign the first channels to the available groups.
	for(int i = 0; i < AUDIO_MAX_GROUPS; ++i)
		audio_channel_control(i, -1, ACC_GROUP, i);

	// Dirty hack for the music; the sequencer uses channels 16..31.
	for(int i = 16; i < 32; ++i)
		audio_channel_control(i, -1, ACC_GROUP, SOUND_GROUP_BGMUSIC);

	// Higher priority for music and UI effects
	audio_channel_control(SOUND_GROUP_BGMUSIC, AVT_ALL, ACC_PRIORITY, 1);
	audio_channel_control(SOUND_GROUP_UIMUSIC, AVT_ALL, ACC_PRIORITY, 2);
	audio_channel_control(SOUND_GROUP_UI, AVT_ALL, ACC_PRIORITY, 3);
	audio_channel_control(SOUND_GROUP_SFX, AVT_ALL, ACC_PRIORITY, 4);

	g_wrap(MAP_SIZEX*CHIP_SIZEX, MAP_SIZEY*CHIP_SIZEY);
	g_scale(VIEWLIMIT * 3 / 2, VIEWLIMIT);

	// For the noise "bzzzt" effect :-)
	audio_channel_control(3, -1, ACC_PRIM_BUS, 1);

	// Bus 0: Sound effects
	audio_bus_control(0, 1, ABC_FX_TYPE, AFX_NONE);
	audio_bus_controlf(0, 0, ABC_SEND_MASTER, 1.0);
	audio_bus_controlf(0, 0, ABC_SEND_BUS_7, 0.5); // Always a little rvb!

	// Bus 1: Sound effects with less reverb
	audio_bus_control(1, 1, ABC_FX_TYPE, AFX_NONE);
	audio_bus_controlf(1, 0, ABC_SEND_MASTER, 1.0);
	audio_bus_controlf(1, 0, ABC_SEND_BUS_7, 0.1);

	prefschange();

	time = SDL_GetTicks();
	return 0;
}


void KOBO_sound::stop()
{
	audio_stop();
}


void KOBO_sound::close()
{
	audio_close();
	sounds_loaded = 0;
	music_loaded = 0;
}



/*--------------------------------------------------
	Main controls
--------------------------------------------------*/

void KOBO_sound::master_reverb(float rvb)
{
	int master_rvb = (int)(rvb * 65536.0);
	audio_bus_controlf(7, 0, ABC_SEND_MASTER, 0.0);
	audio_bus_control(7, 1, ABC_SEND_MASTER, master_rvb);
	if(master_rvb)
		audio_bus_control(7, 1, ABC_FX_TYPE, AFX_REVERB);
	else
		audio_bus_control(7, 1, ABC_FX_TYPE, AFX_NONE);
}


void KOBO_sound::set_boost(int boost)
{
	switch(boost)
	{
	  case 0:
		audio_set_limiter(1.0, 5.0);
		break;
	  case 1:
		audio_set_limiter(.9, 4.5);
		break;
	  case 2:
	  default:
		audio_set_limiter(.75, 4.0);
		break;
	  case 3:
		audio_set_limiter(.5, 3.5);
		break;
	  case 4:
		audio_set_limiter(.25, 3.0);
		break;
	}
}


void KOBO_sound::sfx_volume(float vol)
{
	audio_group_controlf(SOUND_GROUP_SFX, ACC_VOLUME,
			(float)prefs->sfx_vol * vol / 100.0f);
}


void KOBO_sound::intro_volume(float vol)
{
	audio_group_controlf(SOUND_GROUP_UIMUSIC, ACC_VOLUME,
			(float)prefs->intro_vol * vol / 100.0f);
}


void KOBO_sound::music_volume(float vol)
{
	audio_group_controlf(SOUND_GROUP_BGMUSIC, ACC_VOLUME,
			(float)prefs->music_vol * vol / 100.0f);
}


void KOBO_sound::period(int ms)
{
	if(ms > 0)
		_period = ms;
	time = 0;
}


void KOBO_sound::frame()
{
	int nc = audio_next_callback();
	if(time < nc)
		time = nc;
	else
		time -= 1 + (labs(time - nc) >> 5);
	audio_bump(time);
	time += _period;
}


void KOBO_sound::run()
{
	audio_run();
}


/*--------------------------------------------------
	In-game sound
--------------------------------------------------*/

void KOBO_sound::g_music(int wid)
{
	play(SOUND_GROUP_BGMUSIC, wid);
}


void KOBO_sound::g_position(int x, int y)
{
	listener_x = x;
	listener_y = y;
}


void KOBO_sound::g_wrap(int w, int h)
{
	wrap_x = w;
	wrap_y = h;
}


void KOBO_sound::g_scale(int maxrange, int pan_maxrange)
{
	scale = 65536 / maxrange;
	panscale = 65536 / pan_maxrange;
}


void KOBO_sound::g_play(int wid, int x, int y, int energy)
{
	int volume, vx, vy, pan;
	/* Calculate volume */
	x -= listener_x;
	y -= listener_y;
	if(wrap_x)
		x %= wrap_x;
	if(wrap_y)
		y %= wrap_y;

	/* Approximation of distance attenuation */
	vx = abs(x * scale);
	vy = abs(y * scale);
	if((vx | vy) & 0xffff0000)
		return;

	vx = (65536 - vx) >> 1;
	vy = (65536 - vy) >> 1;
	volume = vx * vy >> 14;
	volume = (volume>>1) * (volume>>1) >> 14;

	pan = x * panscale;
	if(pan < -65536)
		pan = -65536;
	else if(pan > 65536)
		pan = 65536;

	audio_channel_control(SOUND_GROUP_SFX, AVT_FUTURE, ACC_PATCH, wid);
	audio_channel_control(SOUND_GROUP_SFX, AVT_FUTURE, ACC_PAN, pan);
	audio_channel_control(SOUND_GROUP_SFX, AVT_FUTURE, ACC_VOLUME, volume);
	audio_channel_play(SOUND_GROUP_SFX, sfx2d_tag,
			((60 - 12)<<16) + energy * 12, energy);
	sfx2d_tag = (sfx2d_tag + 1) & 0xffff;
}


void KOBO_sound::g_play(int wid)
{
	audio_channel_control(SOUND_GROUP_SFX, AVT_FUTURE, ACC_PATCH, wid);
	audio_channel_control(SOUND_GROUP_SFX, AVT_FUTURE, ACC_PAN, 0);
	audio_channel_control(SOUND_GROUP_SFX, AVT_FUTURE, ACC_VOLUME, 65536);
	audio_channel_play(SOUND_GROUP_SFX, sfx2d_tag, 60<<16, 65536);
	sfx2d_tag = (sfx2d_tag + 1) & 0xffff;
}


void KOBO_sound::g_player_fire()
{
	g_play(SOUND_SHOT);
}


void KOBO_sound::g_player_damage()
{
	g_play(SOUND_METALLIC);
}


void KOBO_sound::g_player_death()
{
	g_play(SOUND_EXPL);
}


/*--------------------------------------------------
	UI sound effects
--------------------------------------------------*/

void KOBO_sound::ui_music(int wid)
{
// KLUDGE UNTIL MIDI SONGS AND FX CONTROL IS FIXED!
	audio_bus_control(0, 1, ABC_FX_TYPE, AFX_NONE);
	audio_bus_controlf(0, 0, ABC_SEND_MASTER, 1.0);
	audio_bus_controlf(0, 0, ABC_SEND_BUS_7, 0.5);
// KLUDGE UNTIL MIDI SONGS AND FX CONTROL IS FIXED!
	play(SOUND_GROUP_UIMUSIC, wid);
}


void KOBO_sound::ui_noise(int n)
{
	if(n < 0)
	{
		sound.sfx_volume(0.0f);
		audio_channel_control(SOUND_GROUP_UIMUSIC,
				AVT_FUTURE, ACC_PATCH, SOUND_BZZZT);
		audio_channel_play(SOUND_GROUP_UIMUSIC, 0, 60<<16,
				10000 + pubrand.get(14));
		return;
	}
	if(n && pubrand.get(1))
	{
		sound.sfx_volume(0.0f);
		audio_channel_control(SOUND_GROUP_UIMUSIC,
				AVT_FUTURE, ACC_PATCH, SOUND_BZZZT);
		audio_channel_play(SOUND_GROUP_UIMUSIC, 0, 60<<16,
				20000 + pubrand.get(14));
	}
	else
		sound.sfx_volume(1.0f);
}


void KOBO_sound::ui_ok()
{
	play(SOUND_GROUP_UI, SOUND_EXPL);
}


void KOBO_sound::ui_cancel()
{
	play(SOUND_GROUP_UI, SOUND_SHOT);
}


void KOBO_sound::ui_move()
{
	play(SOUND_GROUP_UI, SOUND_BEAM);
}


void KOBO_sound::ui_tick()
{
	play(SOUND_GROUP_UI, SOUND_SHOT);
}


void KOBO_sound::ui_error()
{
	play(SOUND_GROUP_UI, SOUND_METALLIC);
}


void KOBO_sound::ui_play()
{
	play(SOUND_GROUP_UI, SOUND_PLAY);
}


void KOBO_sound::ui_ready()
{
	play(SOUND_GROUP_UI, SOUND_READY);
}


void KOBO_sound::ui_countdown(int remain)
{
	play(SOUND_GROUP_UI, SOUND_METALLIC);
}


void KOBO_sound::ui_oneup()
{
	play(SOUND_GROUP_UI, SOUND_ONEUP);
}


void KOBO_sound::ui_gameover()
{
	play(SOUND_GROUP_UI, SOUND_GAMEOVER);
}
