/*
  Copyright 1999-2007 ImageMagick Studio LLC, a non-profit organization
  dedicated to making software imaging solutions freely available.

  You may not use this file except in compliance with the License.
  obtain a copy of the License at

    http://www.imagemagick.org/script/license.php

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

  MagickCore quantum inline methods.
*/
#ifndef _MAGICKCORE_QUANTUM_PRIVATE_H
#define _MAGICKCORE_QUANTUM_PRIVATE_H

#if defined(__cplusplus) || defined(c_plusplus)
extern "C" {
#endif

static inline void PopCharPixel(const unsigned char pixel,
  unsigned char **pixels)
{
  *(*pixels)++=(unsigned char) (pixel);
}

static inline void PopDoublePixel(const QuantumInfo *quantum_info,
  const EndianType endian,const double pixel,unsigned char **pixels)
{
  unsigned char
    quantum[8];

  *((double *) quantum)=(double) (pixel*quantum_info->scale+
    quantum_info->minimum); 
  if (endian != LSBEndian)
    {
      *(*pixels)++=quantum[7];
      *(*pixels)++=quantum[6];
      *(*pixels)++=quantum[5];
      *(*pixels)++=quantum[4];
      *(*pixels)++=quantum[3];
      *(*pixels)++=quantum[2];
      *(*pixels)++=quantum[1];
      *(*pixels)++=quantum[0];
      return;
    }
  *(*pixels)++=quantum[0];
  *(*pixels)++=quantum[1];
  *(*pixels)++=quantum[2];
  *(*pixels)++=quantum[3];
  *(*pixels)++=quantum[4];
  *(*pixels)++=quantum[5];
  *(*pixels)++=quantum[6];
  *(*pixels)++=quantum[7];
}

static inline void PopFloatPixel(const QuantumInfo *quantum_info,
  const EndianType endian,const float pixel,unsigned char **pixels)
{
  unsigned char
    quantum[4];

  *((float *) quantum)=(float) ((double) pixel*quantum_info->scale+
    quantum_info->minimum); 
  if (endian != LSBEndian)
    {
      *(*pixels)++=quantum[3];
      *(*pixels)++=quantum[2];
      *(*pixels)++=quantum[1];
      *(*pixels)++=quantum[0];
      return;
    }
  *(*pixels)++=quantum[0];
  *(*pixels)++=quantum[1];
  *(*pixels)++=quantum[2];
  *(*pixels)++=quantum[3];
}

static inline void PopLongPixel(const EndianType endian,const unsigned long pixel,
  unsigned char **pixels)
{
  if (endian != LSBEndian)
    {
      *(*pixels)++=(unsigned char) ((pixel) >> 24);
      *(*pixels)++=(unsigned char) ((pixel) >> 16);
      *(*pixels)++=(unsigned char) ((pixel) >> 8);
      *(*pixels)++=(unsigned char) (pixel);
      return;
    }
  *(*pixels)++=(unsigned char) (pixel);
  *(*pixels)++=(unsigned char) ((pixel) >> 8);
  *(*pixels)++=(unsigned char) ((pixel) >> 16);
  *(*pixels)++=(unsigned char) ((pixel) >> 24);
}

static inline void PopQuantumPixel(const unsigned long depth,
  const unsigned long quantum,unsigned char *pixels)
{
  register long
    i;

  register unsigned long
    data_bits;

  static unsigned char
    *p;

  static unsigned long
    quantum_bits;

  if (depth == 0UL)
    {
      p=pixels;
      quantum_bits=8UL;
    }
  for (i=(long) depth; i > 0L; )
  {
    data_bits=(unsigned long) i;
    if (data_bits > quantum_bits)
      data_bits=quantum_bits;
    i-=data_bits;
    if (quantum_bits == 8)
      *p='\0';
    quantum_bits-=data_bits;
    *p|=(((quantum >> i) &~ ((~0UL) << data_bits)) << quantum_bits);
    if (quantum_bits == 0UL)
      {
        p++;
        quantum_bits=8UL;
      }
  }
}

static inline void PopShortPixel(const EndianType endian,const unsigned short pixel,
  unsigned char **pixels)
{
  if (endian != LSBEndian)
    {
      *(*pixels)++=(unsigned char) ((pixel) >> 8);
      *(*pixels)++=(unsigned char) (pixel);
      return;
    }
  *(*pixels)++=(unsigned char) (pixel);
  *(*pixels)++=(unsigned char) ((pixel) >> 8);
}

static inline unsigned char PushCharPixel(const unsigned char **pixels)
{
  unsigned char
    pixel;

  pixel=(unsigned char) *(*pixels)++;
  return(pixel);
}

static inline IndexPacket PushColormapIndex(Image *image,
  const unsigned long index)
{
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (index < image->colors)
    return((IndexPacket) index);
  (void) ThrowMagickException(&image->exception,GetMagickModule(),
    CorruptImageError,"InvalidColormapIndex","`%s'",image->filename);
  return((IndexPacket) 0);
}

static inline double PushDoublePixel(const QuantumInfo *quantum_info,
  const EndianType endian,const unsigned char **pixels)
{
  double
    pixel;

  unsigned char
    quantum[8];

  if (endian != LSBEndian)
    {
      quantum[7]=(*(*pixels)++);
      quantum[6]=(*(*pixels)++);
      quantum[5]=(*(*pixels)++);
      quantum[5]=(*(*pixels)++);
      quantum[3]=(*(*pixels)++);
      quantum[2]=(*(*pixels)++);
      quantum[1]=(*(*pixels)++);
      quantum[0]=(*(*pixels)++);
      pixel=(*((double *) quantum));
      pixel-=quantum_info->minimum;
      pixel*=quantum_info->scale;
      return(pixel);
    }
  quantum[0]=(*(*pixels)++);
  quantum[1]=(*(*pixels)++);
  quantum[2]=(*(*pixels)++);
  quantum[3]=(*(*pixels)++);
  quantum[4]=(*(*pixels)++);
  quantum[5]=(*(*pixels)++);
  quantum[6]=(*(*pixels)++);
  quantum[7]=(*(*pixels)++);
  pixel=(*((double *) quantum));
  pixel-=quantum_info->minimum;
  pixel*=quantum_info->scale;
  return(pixel);
}

static inline float PushFloatPixel(const QuantumInfo *quantum_info,
  const EndianType endian,const unsigned char **pixels)
{
  float
    pixel;

  unsigned char
    quantum[4];

  if (endian != LSBEndian)
    {
      quantum[3]=(*(*pixels)++);
      quantum[2]=(*(*pixels)++);
      quantum[1]=(*(*pixels)++);
      quantum[0]=(*(*pixels)++);
      pixel=(*((float *) quantum));
      pixel-=quantum_info->minimum;
      pixel*=quantum_info->scale;
      return(pixel);
    }
  quantum[0]=(*(*pixels)++);
  quantum[1]=(*(*pixels)++);
  quantum[2]=(*(*pixels)++);
  quantum[3]=(*(*pixels)++);
  pixel=(*((float *) quantum));
  pixel-=quantum_info->minimum;
  pixel*=quantum_info->scale;
  return(pixel);
}

static inline unsigned long PushLongPixel(const EndianType endian,
  const unsigned char **pixels)
{
  unsigned long
    pixel;

  if (endian != LSBEndian)
    {
      pixel=(unsigned long) (*(*pixels)++ << 24);
      pixel|=(unsigned long) (*(*pixels)++ << 16);
      pixel|=(unsigned long) (*(*pixels)++ << 8);
      pixel|=(unsigned long) (*(*pixels)++);
      return(pixel);
    }
  pixel=(unsigned long) (*(*pixels)++);
  pixel|=(unsigned long) (*(*pixels)++ << 8);
  pixel|=(unsigned long) (*(*pixels)++ << 16);
  pixel|=(unsigned long) (*(*pixels)++ << 24);
  return(pixel);
}

static inline unsigned long PushQuantumPixel(const unsigned char *pixels,
  const unsigned long depth)
{
  register long
    i;

  register unsigned long
    quantum_bits,
    quantum;

  static const unsigned char
    *p;

  static unsigned long
    data_bits;

  if (depth == 0UL)
    {
      p=pixels;
      data_bits=8UL;
    }
  quantum=0UL;
  for (i=(long) depth; i > 0L; )
  {
    quantum_bits=(unsigned long) i;
    if (quantum_bits > data_bits)
      quantum_bits=data_bits;
    i-=quantum_bits;
    data_bits-=quantum_bits;
    quantum=(quantum << quantum_bits) |
      ((*p >> data_bits) &~ ((~0UL) << quantum_bits));
    if (data_bits == 0UL)
      {
        p++;
        data_bits=8UL;
      }
  }
  return(quantum);
}

static inline unsigned short PushShortPixel(const EndianType endian,
  const unsigned char **pixels)
{
  unsigned short
    pixel;

  if (endian != LSBEndian)
    {
      pixel=(unsigned short) (*(*pixels)++ << 8);
      pixel|=(unsigned short) (*(*pixels)++);
      return(pixel);
    }
  pixel=(unsigned short) (*(*pixels)++);
  pixel|=(unsigned short) (*(*pixels)++ << 8);
  return(pixel);
}

#if defined(__cplusplus) || defined(c_plusplus)
}
#endif

#endif
