/*--------------------------------------------------------------------
 *	$Id: gmtmath_main.c,v 1.76 2006/05/09 04:35:51 pwessel Exp $
 *
 *	Copyright (c) 1991-2006 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: gmt.soest.hawaii.edu
 *--------------------------------------------------------------------*/
/*
 * gmtmath.c is a reverse polish calculator that operates on table files 
 * (and constants) and perform basic mathematical operations
 * on them like add, multiply, etc.
 * Some operators only work on one operand (e.g., log, exp)
 *
 * Author:	Paul Wessel
 * Date:	10-NOV-1998
 * Version:	1.0 based on 3.1 grdmath and sample1d
 *		3.1.2 PW 03/26/99 Added -H capability
 *		3.1.2 PW 04/07/99 Added -Q for quick scalar calculator
 *		3.3.2 PW 09/10/99 Added erfinv
 *		3.3.3 PW 12/10/99 Added RAND and NRAND
 *		3.3.4 PW 03/14/00 Fix problems with EXCH and DUP for constants
 *		3.3.5 PW 07/14/00 Changed STEP to STEPT and added new STEP
 *		3.3.5 PW 07/31/00 Empty -T means there are no time columns.
 *		3.3.6 PW 08/16/00 Add LT, LE, EQ, GE, GT, NAN, CHIDIST, FDIST, TDIST
 *		3.3.6 PW 08/18/00 Add INT, ISNAN, XOR, MODE, MAD, LMSSCL, SUM
 *		3.3.6 PW 08/18/00 Added -S to just return first row
 *		3.3.6 PW 08/23/00 Added LOWER and UPPER
 *		3.4   PW 03/01/01
 *		4.0   PW 11/28/01 Added Critical values for Chi2, F, T, and Z distributions
 *		      PW 11/30/01 Added LSQFIT to solve a general least squares system
 *		      PW 12/08/01 Allow for -T<filename> with irregular time coordinates
 *		      PW 01/27/04 Added SINC and NEQ
 *		      PW 03/24/04 Added ROOTS
 *		      PW 03/28/04 Added FLIPUD, ROTT
 *		      PW 07/01/04 Added LRAND
 *		      PW 07/17/04 Added LOG2
 *		      PW 07/27/05 Added TN (Chebyshev)
 *		      PW 08/05/05 Added -I to output descending times (reverse t)
 *		      PW 08/10/05 Added Tn for normalized T coordinates [-1 | +1 ]
 *		      PW 09/07/05 Added CORRCOEFF
 *		      PW 02/16/06 If STDIN is given, read <stdin> and put it on the stack
 *				  Also added -F to select which columns should be output [all]
 *		      PW 03/22/06 Added CPOISS
 *		      PW 03/25/06 Removed use of global variables, added ZDIST
 *
 */
 
#include "gmt.h"

#define GMTMATH_N_OPERATORS	110
#define GMTMATH_STACK_SIZE	100

#define GMTMATH_ARG_IS_OPERATOR	 0
#define GMTMATH_ARG_IS_FILE	-1
#define GMTMATH_ARG_IS_NUMBER	-2
#define GMTMATH_ARG_IS_PI	-3
#define GMTMATH_ARG_IS_E	-4
#define GMTMATH_ARG_IS_T_MATRIX	-5
#define GMTMATH_ARG_IS_t_MATRIX	-6

struct GMTMATH_CTRL {	/* All control options for this program (except common args) */
	/* active is TRUE if the option has been activated */
	struct A {	/* -A<t_f(t).d> */
		BOOLEAN active;
		char *file;
	} A;
	struct C {	/* -C<cols> */
		BOOLEAN active;
		int *cols;
	} C;
	struct F {	/* -F<cols> */
		BOOLEAN active;
		int *cols;
	} F;
	struct I {	/* -I */
		BOOLEAN active;
	} I;
	struct N {	/* -N<n_col>/<t_col> */
		BOOLEAN active;
		int ncol, tcol;
	} N;
	struct Q {	/* -Q */
		BOOLEAN active;
	} Q;
	struct S {	/* -S[f|l] */
		BOOLEAN active;
		int mode;
	} S;
	struct T {	/* -T[<tmin/tmax/t_inc>] | -T<file> */
		BOOLEAN active;
		BOOLEAN notime;
		double min, max, inc;
		char *file;
	} T;
};

struct TABLE_HEADER {
	int n_row;		/* Number of time-nodes (rows) */
	int n_col;		/* Number of columns */
	double t_min;		/* Minimum t value */
	double t_max;		/* Maximum t value */
	double t_inc;		/* t increment */
};

struct GMTMATH_INFO {
	BOOLEAN irregular;	/* TRUE if t_inc varies */
	BOOLEAN roots_found;	/* TRUE if roots have been solved for */
	BOOLEAN very_first;	/* TRUE the very first time */
	BOOLEAN *skip_row;	/* TRUE for each row to be skipped */
	int n_roots;		/* Number of roots found */
	int r_col;		/* The column used to find roots */
	double *t_coordinates;	/* Array with t values */
	double *tn_coordinates;	/* Array with t normalized values [-1,+1] */
	struct TABLE_HEADER header;
	char **segment_header;	/* List of segment headers to output when -M is in effect */
	char head_record[BUFSIZ];
};

#include "gmtmath_def.h"

void new_table (double ***s, int n_col, int n);

int main (int argc, char **argv)
{
	int i, j, arg, op = 0, nstack = 0, new_stack = -1, last_arg, ok = 1;
	int use_t_col = 0, first_last_all = 0, nm = 0;
	int consumed_operands[GMTMATH_N_OPERATORS], produced_operands[GMTMATH_N_OPERATORS];

	BOOLEAN constant[GMTMATH_STACK_SIZE], error = FALSE, set_t = FALSE, got_t_from_file = FALSE;
	BOOLEAN set_q = FALSE, read_stdin = FALSE;

	double **stack[GMTMATH_STACK_SIZE], **rhs, **tmp_stack, **stdin_stack;

	double factor[GMTMATH_STACK_SIZE], t_noise, value, off, scale;

	char *outfile = CNULL, file[BUFSIZ];

	struct TABLE_HEADER tbl[GMTMATH_STACK_SIZE], stdin_header, rhs_header;

	struct GMT_HASH *p, *current, localhashnode[GMT_HASH_SIZE];
	struct GMTMATH_INFO info;
	struct GMTMATH_CTRL *Ctrl;

	FILE *fp;

	PFV call_operator[GMTMATH_N_OPERATORS];

	int decode_argument (char *txt, double *value, struct GMT_HASH *H);
	void gmtmath_init(PFV ops[], int n_args[], int n_out[]);
	void GMT_read_table (struct GMTMATH_INFO *info, char *file, struct TABLE_HEADER *h, double ***p, int t_col);
	void GMT_write_table (struct GMTMATH_INFO *info, char *file, struct TABLE_HEADER *h, double **p, int first_last_all, int cols[]);
	void decode_columns (char *txt, int *skip, int n_col, int t_col, int mode);
	void solve_LSQFIT (struct GMTMATH_INFO *info, double **stack[], int last, int n_col, int n_row, int skip[], char *file);
	void *New_Gmtmath_Ctrl (), Free_Gmtmath_Ctrl (struct GMTMATH_CTRL *C);

	argc = GMT_begin (argc, argv);

	Ctrl = (struct GMTMATH_CTRL *) New_Gmtmath_Ctrl ();		/* Allocate and initialize defaults in a new control structure */

	memset ((void *)&info, 0, sizeof (struct GMTMATH_INFO));
	info.very_first = TRUE;

	if (argc == 2 && !strcmp (argv[1], "-")) error = GMT_give_synopsis_and_exit = TRUE;

	if (argc == 1 || GMT_give_synopsis_and_exit) {
		fprintf (stderr, "gmtmath %s - Reverse Polish Notation (RPN) calculator for table data\n\n", GMT_VERSION);
		fprintf (stderr, "usage: gmtmath [-A<t_f(t).d>] [-C<cols>] [-F<cols>] [%s] [-I] [%s] [-N<n_col>/<t_col>] [-Q]\n", GMT_H_OPT, GMT_M_OPT);
		fprintf (stderr, "\t[-S[f|l]] [-T[<tmin/tmax/t_inc>]] [-V] [-%s]\n\t[%s] A B op C op ... = [outfile]\n\n", GMT_b_OPT, GMT_f_OPT);

		if (GMT_give_synopsis_and_exit) exit (EXIT_FAILURE);

		fprintf (stderr, "\tA, B, etc are table files, constants, or symbols (see below)\n");
		fprintf (stderr, "\tTo read stdin give filename as STDIN (which can appear more than once)\n");
		fprintf (stderr, "\tThe stack can hold up to %d entries (given enough memory)\n", GMTMATH_STACK_SIZE);
		fprintf (stderr, "\tTrigonometric operators expect radians.  The operators are:\n\n");
		fprintf (stderr, "\tName	#args	Returns:\n");
		fprintf (stderr, "\t-----------------------\n");
#include "gmtmath_explain.h"
		fprintf (stderr, "\n\tThe special symbols are:\n\n");
		fprintf (stderr, "\t  PI	= 3.1415926...\n");
		fprintf (stderr, "\t  E	= 2.7182818...\n");
		fprintf (stderr, "\t  T	= table with t-coordinates\n");
		fprintf (stderr, "\t  Tn	= table with normalized [-1 to +1] t-coordinates\n");
		fprintf (stderr, "\n\tOPTIONS:\n\n");
		fprintf (stderr, "\t-A Requires -N and will initialize table with file containing t and f(t) only.\n");
		fprintf (stderr, "\t   t goes into column <t_col> while f(t) goes into column <n_col> - 1.\n");
		fprintf (stderr, "\t-C change which columns to operate on [Default is all except time]\n");
		fprintf (stderr, "\t   -C reverts to the default, -Cr toggles current settings, and -Ca selects all columns.\n");
		fprintf(stderr,"\t-F Give comma-separated list of desired columns or ranges to output (0 is first column) [Default is all]\n");
		GMT_explain_option ('H');
		fprintf (stderr, "\t-I Reverses the output sequence into descending order [ascending]\n");
		GMT_explain_option ('M');
		fprintf (stderr, "\t-N sets the number of columns and the id of the time column (0 is first) [2/0]\n");
		fprintf (stderr, "\t-Q quick scalar calculator. Shorthand for -Ca -N1/0 -T0/0/1\n");
		fprintf (stderr, "\t-S Only write first row upon completion of calculations [write all rows]\n");
		fprintf (stderr, "\t   Optionally, append l for last row or f for first row [Default]\n");
		fprintf (stderr, "\t-T Set domain from t_min to t_max in steps of t_inc\n");
		fprintf (stderr, "\t   If a filename is given instead we read t coordinates from first column.\n");
		fprintf (stderr, "\t   If no domain is given we assume no time, i.e. only data columns are present.\n");
		fprintf (stderr, "\t   This choice also implies -Ca.\n");
		GMT_explain_option ('V');
		GMT_explain_option ('i');
		GMT_explain_option ('n');
		GMT_explain_option ('o');
		GMT_explain_option ('n');
		GMT_explain_option ('f');
		exit (EXIT_FAILURE);
	}

	for (i = 1, error = TRUE; error && i < argc; i++) if (argv[i][0] == '=' && argv[i][1] == '\0') error = FALSE;
	if (error) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Usage is <operations> = [outfile]\n", GMT_program);
		exit (EXIT_FAILURE);
	}

	GMT_put_history (argc, argv);	/* Update .gmtcommands4 */

	GMT_hash_init (localhashnode, operator, GMT_HASH_SIZE, GMTMATH_N_OPERATORS);

	for (i = 0; i < GMTMATH_STACK_SIZE; i++) {
		constant[i] = FALSE;
		factor[i] = 0.0;
		tbl[i].n_col = tbl[i].n_row = 0;
		stack[i] = (double **)NULL;
	}

	last_arg = 1;
	while (last_arg < argc && !(argv[last_arg][0] == '=' && argv[last_arg][1] == '\0')) last_arg++;	/* Find position of the = argument */
	i = last_arg + 1;	/* This is normally the position of the output file name (if one was given) */
	while (i < argc && argv[i][0] == '-') i++;	/* Skip past any options between = and output file */
	outfile = (i < argc) ? argv[i] : NULL;

	/* Must first scan command line for -b, -f, -H, and -M switches before reading any file */

	for (arg = 1; arg < last_arg; arg++) {
		if (!strncmp (argv[arg], "-H", 2)) error += GMT_parse_common_options (argv[arg], NULL, NULL, NULL, NULL);
		if (!strncmp (argv[arg], "-M", 2)) GMT_multisegment (&argv[arg][2]);
		if (!strncmp (argv[arg], "-b", 2)) error += GMT_parse_common_options (argv[arg], NULL, NULL, NULL, NULL);
		if (!strncmp (argv[arg], "-f", 2)) error += GMT_parse_common_options (argv[arg], NULL, NULL, NULL, NULL);
		if (!strcmp (argv[arg], "-T")) GMT_io.skip_if_NaN[0] = GMT_io.skip_if_NaN[1] = FALSE;	/* Turn off default GMT NaN-handling in x/y columns */
	}
	/* Get header from one file so we can allocate space */

	for (arg = 1; nm == 0 && arg < last_arg; arg++) {

		if (argv[arg][0] == '-' && argv[arg][1] != 0) continue;	/* Command line option */
		if (decode_argument (argv[arg], &value, localhashnode) != GMTMATH_ARG_IS_FILE) continue;

		strcpy (file, argv[arg]);
		if (!strcmp (file, "STDIN")) {
			GMT_read_table (&info, argv[arg], &stdin_header, &stdin_stack, Ctrl->N.tcol);
			memcpy ((void *)&info.header, (void *)&stdin_header, sizeof (struct TABLE_HEADER));
			read_stdin = TRUE;
		}
		else
			GMT_read_table (&info, argv[arg], &info.header, &tmp_stack, Ctrl->N.tcol);

		nm = info.header.n_row * info.header.n_col;
		Ctrl->N.ncol = info.header.n_col;
		got_t_from_file = TRUE;
		use_t_col = Ctrl->N.tcol;
	}

	/* Scan command line for -A, -I, -T, -M, -N, -Q, -S, -V */

	for (arg = 1; arg < last_arg; arg++) {
		if (argv[arg][0] == '-') {

			switch (argv[arg][1]) {

				case 'V':
					error += GMT_parse_common_options (argv[arg], NULL, NULL, NULL, NULL);
					break;

				case 'A':	/* y(x) table for LSQFIT operations */
					Ctrl->A.active = TRUE;
					Ctrl->A.file = strdup (&argv[arg][2]);
					break;
				case 'F':
					decode_columns (&argv[arg][2], Ctrl->F.cols, BUFSIZ, 0, 1);
					Ctrl->F.active = TRUE;
					break;
				case 'I':
					Ctrl->I.active = TRUE;
					break;
				case 'M':               /* Multiple line segments */
					GMT_multisegment (&argv[arg][2]);
					break;
				case 'N':
					Ctrl->N.active = TRUE;
					sscanf (&argv[arg][2], "%d/%d", &Ctrl->N.ncol, &Ctrl->N.tcol);
					break;
				case 'Q':	/* Quick for -Ca -N1/0 -T0/0/1 */
					Ctrl->Q.active = TRUE;
					break;
				case 'S':	/* Only want one row (first or last) */
					Ctrl->S.active = TRUE;
					if (!argv[arg][2] || argv[arg][2] == 'F' || argv[arg][2] == 'f')
						Ctrl->S.mode = -1;
					else if (argv[arg][2] == 'L' || argv[arg][2] == 'l')
						Ctrl->S.mode = +1;
					else {
						fprintf (stderr, "%s: GMT SYNTAX ERROR: Syntax is -S[f|l]\n", GMT_program);
						exit (EXIT_FAILURE);
					}
					break;
				case 'T':	/* Either get a file with time coordinate or a min/max/dt setting */
					Ctrl->T.active = TRUE;
					if (argv[arg][2] && !(GMT_access (&argv[arg][2], R_OK)))	/* Argument given and file can be opened */
						Ctrl->T.file = strdup (&argv[arg][2]);
					else {
						if (sscanf (&argv[arg][2], "%lf/%lf/%lf", &info.header.t_min, &info.header.t_max, &info.header.t_inc) != 3) Ctrl->T.notime = TRUE;
					}
					break;
			}
		}
	}
	if (Ctrl->Q.active && (Ctrl->T.active || Ctrl->N.active || Ctrl->C.active)) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Cannot use -T, -N, or -C when -Q has been set\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	if (GMT_io.binary[GMT_IN] && GMT_io.io_header[GMT_IN]) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data cannot have header -H\n", GMT_program);
		error++;
	}
	if (Ctrl->N.active && (Ctrl->N.ncol <= 0 || Ctrl->N.tcol < 0 || Ctrl->N.tcol >= Ctrl->N.ncol)) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  -N must have positive n_cols and 0 <= t_col < n_col\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	set_t = (Ctrl->T.active && !Ctrl->T.file && !Ctrl->T.notime);
	if (nm && set_t) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Cannot use -T when data files are specified\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	if (Ctrl->A.active) {
		GMT_read_table (&info, Ctrl->A.file, &rhs_header, &rhs, 0);	/* Always store as t and f(t) in cols 0 and 1 */
		if (rhs_header.n_col != 2) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  -A must take a file with 2 (t,f(t)) colums\n", GMT_program);
			exit (EXIT_FAILURE);
		}
	}
	if (Ctrl->N.active) {
		GMT_io.skip_if_NaN[0] = GMT_io.skip_if_NaN[1] = FALSE;
		GMT_io.skip_if_NaN[Ctrl->N.tcol] = TRUE;
	}
	if (Ctrl->Q.active) {
		Ctrl->N.ncol = 1;
		Ctrl->N.tcol = 0;
		Ctrl->N.active = set_t = set_q = TRUE;
		info.header.t_min = info.header.t_max = 0;
		info.header.t_inc = 1.0;
	}
	if (Ctrl->T.active) {
		if (Ctrl->T.file) {	/* Got a filename */
			if (got_t_from_file) {
				fprintf (stderr, "%s: GMT SYNTAX ERROR:  Cannot use -T when data files are specified\n", GMT_program);
				exit (EXIT_FAILURE);
			}
			GMT_read_table (&info, Ctrl->T.file, &info.header, &tmp_stack, 0);
			use_t_col = 0;
			got_t_from_file = TRUE;
		}
		else {
		}
	}
	if (set_t && !set_q) {
		switch (GMT_minmaxinc_verify (info.header.t_min, info.header.t_max, info.header.t_inc, GMT_SMALL)) {
			case 1:
				fprintf (stderr, "%s: GMT SYNTAX ERROR -T:  (max - min) is not a whole multiple of inc\n", GMT_program);
				exit (EXIT_FAILURE);
				break;
			case 2:
				if (info.header.t_inc != 1.0) {	/* Allow for somebody explicitly saying -T0/0/1 */
					fprintf (stderr, "%s: GMT SYNTAX ERROR -T:  (max - min) is <= 0\n", GMT_program);
					exit (EXIT_FAILURE);
				}
				break;
			case 3:
				fprintf (stderr, "%s: GMT SYNTAX ERROR -T:  inc is <= 0\n", GMT_program);
				exit (EXIT_FAILURE);
				break;
			default:	/* OK */
				break;
		}

		info.header.n_row = irint ((info.header.t_max - info.header.t_min) / info.header.t_inc) + 1;
		info.header.n_col = Ctrl->N.ncol;
		nm = info.header.n_row * info.header.n_col;
	}
	first_last_all = Ctrl->S.mode;

	if (Ctrl->A.active) {	/* Get number of rows and time from the file, but not n_cols (that takes -N, which defaults to 2) */
		info.header.n_row = rhs_header.n_row;
		info.header.n_col = Ctrl->N.ncol;
		info.header.t_min = rhs_header.t_min;
		info.header.t_max = rhs_header.t_max;
		info.header.t_inc = rhs_header.t_inc;
		nm = info.header.n_row * info.header.n_col;
	}
	if (set_q) info.header.n_row = info.header.n_col = nm = 1;
	if (Ctrl->T.file) {
		info.header.n_col = Ctrl->N.ncol;
		nm = info.header.n_row * info.header.n_col;
	}
	if (nm == 0) {	/* Neither a file nor -T given; must read data from stdin */
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Expression must contain at least one table file or -T [and -N]\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	new_table (&stack[0], info.header.n_col, info.header.n_row);

	if (!Ctrl->T.notime && info.header.n_col > 1) Ctrl->C.cols[Ctrl->N.tcol] = (set_q) ? FALSE : TRUE;
	Ctrl->F.cols = (int *) GMT_memory ((void *)Ctrl->F.cols, info.header.n_col, sizeof (int), GMT_program);
	if (!Ctrl->F.active) for (i = 0; i < info.header.n_col; i++) Ctrl->F.cols[i] = TRUE;

	/* Get t vector */

	info.t_coordinates = (double *) GMT_memory (VNULL, (size_t)info.header.n_row, sizeof (double), GMT_program);
	info.tn_coordinates = (double *) GMT_memory (VNULL, (size_t)info.header.n_row, sizeof (double), GMT_program);
	if (read_stdin) {
		memcpy ((void *)info.t_coordinates, (void *)stdin_stack[use_t_col], (size_t)(info.header.n_row * sizeof (double)));
		for (i = 1; i < info.header.n_row && (info.skip_row[i] || info.skip_row[i-1]); i++);	/* Find the first real two records in a row */
		info.header.t_inc = (i == info.header.n_row) ? GMT_d_NaN : stdin_stack[use_t_col][i] - stdin_stack[use_t_col][i-1];
		t_noise = fabs (GMT_SMALL * info.header.t_inc);
		for (i = 1; i < info.header.n_row && !info.irregular; i++) if (fabs (fabs (info.t_coordinates[i] - info.t_coordinates[i-1]) - fabs (info.header.t_inc)) > t_noise && !(info.skip_row[i] || info.skip_row[i-1])) info.irregular = TRUE;
	}
	else if (got_t_from_file) {
		memcpy ((void *)info.t_coordinates, (void *)tmp_stack[use_t_col], (size_t)(info.header.n_row * sizeof (double)));
		for (i = 1; i < info.header.n_row && (info.skip_row[i] || info.skip_row[i-1]); i++);	/* Find the first real two records in a row */
		info.header.t_inc = (i == info.header.n_row) ? GMT_d_NaN : tmp_stack[use_t_col][i] - tmp_stack[use_t_col][i-1];
		t_noise = fabs (GMT_SMALL * info.header.t_inc);
		for (i = 1; i < info.header.n_row && !info.irregular; i++) if (fabs (fabs (info.t_coordinates[i] - info.t_coordinates[i-1]) - fabs (info.header.t_inc)) > t_noise && !(info.skip_row[i] || info.skip_row[i-1])) info.irregular = TRUE;
		for (i = 0; i < info.header.n_col; i++) GMT_free ((void *)tmp_stack[i]);
		GMT_free ((void *)tmp_stack);
	}
	else {
		for (i = 0; i < info.header.n_row; i++) info.t_coordinates[i] = (i == (info.header.n_row-1)) ? info.header.t_max: info.header.t_min + i * info.header.t_inc;
		t_noise = fabs (GMT_SMALL * info.header.t_inc);
	}
	off = 0.5 * (info.t_coordinates[info.header.n_row-1] + info.t_coordinates[0]);
	scale = 2.0 / (info.t_coordinates[info.header.n_row-1] - info.t_coordinates[0]);
	if (Ctrl->I.active) for (i = 0; i < info.header.n_row/2; i++) d_swap (info.t_coordinates[i], info.t_coordinates[info.header.n_row-1-i]);	/* Reverse time-series */
	for (i = 0; i < info.header.n_row; i++) info.tn_coordinates[i] = (info.t_coordinates[i] - off) * scale;
	if (!read_stdin) memcpy ((void *)stack[0][Ctrl->N.tcol], (void *)info.t_coordinates, (size_t)(info.header.n_row * sizeof (double)));
	if (Ctrl->A.active) {
		memcpy ((void *)stack[0][Ctrl->N.ncol-1], (void *)rhs[1], (size_t)(info.header.n_row * sizeof (double)));
		GMT_free ((void *)rhs[0]);
		GMT_free ((void *)rhs[1]);
		nstack = 1;
	}
	else
		nstack = 0;
		
	if (!info.skip_row) info.skip_row = (BOOLEAN *) GMT_memory (VNULL, (size_t)info.header.n_row, sizeof (BOOLEAN), GMT_program);	/* All FALSE if generated by -T */

	gmtmath_init (call_operator, consumed_operands, produced_operands);


	for (arg = 1; !error && arg < last_arg; arg++) {

		/* First check if we should skip optional arguments */

		if (!(strncmp (argv[arg], "-T", 2) && strncmp (argv[arg], "-b", 2) && strncmp (argv[arg], "-f", 2) && strncmp (argv[arg], "-N", 2))) continue;
		if (!(strncmp (argv[arg], "-H", 2) && strncmp (argv[arg], "-Q", 2) && strncmp (argv[arg], "-S", 2) && strncmp (argv[arg], "-V", 2))) continue;
		if (!(strncmp (argv[arg], "-A", 2) && strncmp (argv[arg], "-I", 2) && strncmp (argv[arg], "-F", 2) && strncmp (argv[arg], "-M", 2))) continue;

		if (!strncmp (argv[arg], "-C", 2)) {	/* Change affected columns */
			decode_columns (&argv[arg][2], Ctrl->C.cols, Ctrl->N.ncol, Ctrl->N.tcol, 0);
			continue;
		}

		op = decode_argument (argv[arg], &value, localhashnode);

		if (op != GMTMATH_ARG_IS_FILE && !GMT_access(argv[arg], R_OK)) fprintf (stderr, "%s Warning: The number or operator %s may be confused with an existing file %s!\n", GMT_program, argv[arg], argv[arg]);

		if (op < GMTMATH_ARG_IS_OPERATOR) {	/* File name or factor */

			if (nstack == GMTMATH_STACK_SIZE) {	/* Stack overflow */
				error = TRUE;
				continue;
			}

			if (op == GMTMATH_ARG_IS_NUMBER) {
				constant[nstack] = TRUE;
				factor[nstack] = value;
				error = FALSE;
				if (gmtdefs.verbose) fprintf (stderr, "%g ", factor[nstack]);
				nstack++;
				continue;
			}
			else if (op == GMTMATH_ARG_IS_PI) {
				constant[nstack] = TRUE;
				factor[nstack] = M_PI;
				if (gmtdefs.verbose) fprintf (stderr, "%g ", factor[nstack]);
				nstack++;
				continue;
			}
			else if (op == GMTMATH_ARG_IS_E) {
				constant[nstack] = TRUE;
				factor[nstack] = M_E;
				if (gmtdefs.verbose) fprintf (stderr, "%g ", factor[nstack]);
				nstack++;
				continue;
			}

			/* Here we need a matrix */

			if (!stack[nstack]) new_table (&stack[nstack], info.header.n_col, info.header.n_row);

			constant[nstack] = FALSE;

			if (op == GMTMATH_ARG_IS_T_MATRIX) {	/* Need to set up matrix of t-values */
				if (Ctrl->T.notime) {
					fprintf (stderr, "%s: T is not defined for plain data files!\n", GMT_program);
					exit (EXIT_FAILURE);
				}
				if (gmtdefs.verbose) fprintf (stderr, "T ");
				for (j = 0; j < info.header.n_col; j++) memcpy ((void *)stack[nstack][j], (void *)info.t_coordinates, (size_t)(info.header.n_row * sizeof (double)));
			}
			else if (op == GMTMATH_ARG_IS_t_MATRIX) {	/* Need to set up matrix of normalized t-values */
				if (Ctrl->T.notime) {
					fprintf (stderr, "%s: Tn is not defined for plain data files!\n", GMT_program);
					exit (EXIT_FAILURE);
				}
				if (gmtdefs.verbose) fprintf (stderr, "Tn ");
				for (j = 0; j < info.header.n_col; j++) if (j != Ctrl->N.tcol) memcpy ((void *)stack[nstack][j], (void *)info.tn_coordinates, (size_t)(info.header.n_row * sizeof (double)));
			}
			else if (op == GMTMATH_ARG_IS_FILE) {		/* Filename given */
				if (!strcmp (argv[arg], "STDIN")) {	/* stdin file */
					if (gmtdefs.verbose) fprintf (stderr, "<stdin> ");
					memcpy ((void *)&tbl[nstack], (void *)&stdin_header, sizeof (struct TABLE_HEADER));
					for (j = 0; j < info.header.n_col; j++) memcpy ((void *)stack[nstack][j], (void *)stdin_stack[j], (size_t)(info.header.n_row * sizeof (double)));
				}
				else {
					if (gmtdefs.verbose) fprintf (stderr, "%s ", argv[arg]);
					GMT_read_table (&info, argv[arg], &tbl[nstack], &stack[nstack], Ctrl->N.tcol);
				}
				if (tbl[nstack].n_row != info.header.n_row || tbl[nstack].n_col != info.header.n_col) {
					fprintf (stderr, "%s: tables not of same size!\n", GMT_program);
					exit (EXIT_FAILURE);
				}
				else if (!Ctrl->T.notime && (fabs (tbl[nstack].t_min - info.header.t_min) > t_noise || fabs (tbl[nstack].t_max - info.header.t_max) > t_noise)) {
					fprintf (stderr, "%s: tables do not cover the same domain!\n", GMT_program);
					exit (EXIT_FAILURE);
				}
			}
			nstack++;
			continue;
		}

		/* Here we have an operator */

		if (!strncmp (argv[arg], "ROOTS", 5) && !((arg+1) == last_arg && argv[arg+1][0] == '=')) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  Only = may follow operator ROOTS\n", GMT_program);
			exit (EXIT_FAILURE);
		}

		if ((new_stack = nstack - consumed_operands[op] + produced_operands[op]) >= GMTMATH_STACK_SIZE) {
			error = TRUE;
			continue;
		}

		if (nstack < consumed_operands[op]) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  Operation \"%s\" requires %d operands\n", GMT_program, operator[op], consumed_operands[op]);
			exit (EXIT_FAILURE);
		}

		if (gmtdefs.verbose) fprintf (stderr, "%s ", operator[op]);

		for (i = produced_operands[op] - consumed_operands[op]; i > 0; i--) {
			 if (stack[nstack+i-1])	continue;

			/* Must make space for more */

			new_table (&stack[nstack+i-1], info.header.n_col, info.header.n_row);
		}

		/* If operators operates on constants only we may have to make space as well */

		for (j = 0, i = nstack - consumed_operands[op]; j < produced_operands[op]; j++, i++) {
			if (constant[i] && !stack[i]) new_table (&stack[i], info.header.n_col, info.header.n_row);
		}

		if (!strcmp (operator[op], "LSQFIT")) {	/* Special case, solve LSQ system and exit */
			solve_LSQFIT (&info, stack, nstack - 1, Ctrl->N.ncol, info.header.n_row, Ctrl->C.cols, outfile);
			exit (EXIT_SUCCESS);
		}

		for (j = 0; j < Ctrl->N.ncol; j++) {
			if (Ctrl->C.cols[j]) continue;
			(*call_operator[op]) (&info, stack, constant, factor, nstack - 1, j, info.header.n_row);	/* Do it */
		}

		nstack = new_stack;

		for (i = 1; i <= produced_operands[op]; i++)
			constant[nstack-i] = FALSE;	/* Now filled with table */
	}

	if (error && !ok) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Unable to decode constant %s (File not found?)\n", GMT_program, argv[i-1]);
		exit (EXIT_FAILURE);
	}

	if (error) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Stack overflow (%s)\n", GMT_program, argv[i-1]);
		exit (EXIT_FAILURE);
	}

	if (gmtdefs.verbose) {
		(outfile) ? fprintf (stderr, "= %s", outfile) : fprintf (stderr, "= <stdout>");
	}

	if (new_stack < 0 && constant[0]) {	/* Only a constant provided, set table accordingly */
		for (j = 0; j < info.header.n_col; j++) {
			if (Ctrl->C.cols[j]) continue;
			for (i = 0; i < info.header.n_row; i++) stack[0][j][i] = factor[0];
		}
	}

	if (gmtdefs.verbose) fprintf (stderr, "\n");

	if (info.roots_found) {	/* Special treatment of root finding */
		if (outfile) {
			if ((fp = GMT_fopen (outfile, "w")) == NULL) {
				fprintf (stderr, "%s: GMT ERROR:  Could not create file (%s)\n", GMT_program, outfile);
				exit (EXIT_FAILURE);
			}
		}
		else
			fp = GMT_stdout;
		for (i = 0; i < info.n_roots; i++) GMT_output (fp, 1, &stack[0][info.r_col][i]);
		if (fp != GMT_stdout) GMT_fclose (fp);
	}
	else
		GMT_write_table (&info, outfile, &info.header, stack[0], first_last_all, Ctrl->F.cols);

	for (i = 0; i < GMTMATH_STACK_SIZE; i++) if (stack[i]) {
		for (j = 0; j < info.header.n_col; j++) GMT_free ((void *)stack[i][j]);
		GMT_free ((void *)stack[i]);
	}
	if (read_stdin) {
		for (j = 0; j < info.header.n_col; j++) GMT_free ((void *)stdin_stack[j]);
		GMT_free ((void *)stdin_stack);
	}
	GMT_free ((void *)info.t_coordinates);
	if (info.skip_row) GMT_free ((void *)info.skip_row);
	for (i = 0; i < GMT_HASH_SIZE; i++) {
		p = localhashnode[i].next;
		while ((current = p)) {
			p = p->next;
			GMT_free ((void *)current);
		}
	}

	if (nstack > 1) fprintf (stderr, "%s: Warning: %d more operands left on the stack!\n", GMT_program, nstack-1);

	Free_Gmtmath_Ctrl (Ctrl);	/* Deallocate control structure */

	GMT_end (argc, argv);

	exit (EXIT_SUCCESS);
}

void GMT_read_table (struct GMTMATH_INFO *info, char *file, struct TABLE_HEADER *h, double ***p, int t_col)
{
	BOOLEAN init_skip_row;
	int j, n = 0, n_alloc = 0, n_expected_fields, n_fields;
	double *in, **table = NULL;
	char buffer[BUFSIZ];
	FILE *fp;

	n_expected_fields = (GMT_io.binary[GMT_IN]) ? GMT_io.ncol[GMT_IN] : BUFSIZ;

#ifdef SET_IO_MODE
	GMT_setmode (0);
#endif

	if (!strcmp (file, "STDIN"))
		fp = GMT_stdin;
	else if ((fp = GMT_fopen (file, GMT_io.r_mode)) == NULL) {
		fprintf (stderr, "%s: Error opening file %s\n", GMT_program, file);
		exit (EXIT_FAILURE);
	}

	for (j = 0; GMT_io.io_header[GMT_IN] && j < GMT_io.n_header_recs; j++) {
		GMT_fgets (buffer, BUFSIZ, fp);
		if (info->very_first && j == 0) strcpy (info->head_record, buffer);
		info->very_first = FALSE;
	}

	GMT_input (fp, &n_expected_fields, &in);
	if (GMT_io.status & GMT_IO_EOF) {
		fprintf (stderr, "%s: Error reading 1st record of file %s\n", GMT_program, file);
		exit (EXIT_FAILURE);
	}

	init_skip_row = (info->segment_header == NULL);
	if (init_skip_row) {
		info->skip_row = (BOOLEAN *) GMT_memory (VNULL, (size_t)GMT_CHUNK, sizeof (BOOLEAN), GMT_program);	/* All FALSE by default */
		if (!GMT_io.binary[GMT_IN]) info->segment_header = (char **) GMT_memory (VNULL, (size_t)GMT_CHUNK, sizeof (char *), GMT_program);
	}
	do {
		while ((GMT_io.status & GMT_IO_SEGMENT_HEADER) && !(GMT_io.status & GMT_IO_EOF)) {
			if (init_skip_row) {
				info->skip_row[n] = TRUE;
				if (!GMT_io.binary[GMT_IN]) {	/* Only ascii files have useful header info */
					info->segment_header[n] = (char *) GMT_memory (VNULL, (size_t)strlen(GMT_io.segment_header), sizeof (char), GMT_program);
					strcpy (info->segment_header[n], GMT_io.segment_header);	/* Save this header */
				}
			}
			n_fields = GMT_input (fp, &n_expected_fields, &in);
			n++;
		}
		if (n_alloc == 0) {
			n_alloc = GMT_CHUNK;
			h->n_col = n_expected_fields;
			new_table (&table, n_expected_fields, n_alloc);
		}
		for (j = 0; j < h->n_col; j++) table[j][n] = in[j];
		n++;
		if (n == n_alloc) {
			n_alloc += GMT_CHUNK;
			if (init_skip_row) {
				info->skip_row = (BOOLEAN *) GMT_memory ((void *)info->skip_row, (size_t)n_alloc, sizeof (BOOLEAN), GMT_program);
				if (!GMT_io.binary[GMT_IN]) info->segment_header = (char **) GMT_memory ((void *)info->segment_header, (size_t)n_alloc, sizeof (char *), GMT_program);
			}
			for (j = 0; j < h->n_col; j++) table[j] = (double *) GMT_memory ((void *)table[j], (size_t)n_alloc, sizeof (double), GMT_program);
		}
	} while ((n_fields = GMT_input (fp, &n_expected_fields, &in)) >= 0 && !(GMT_io.status & GMT_IO_EOF));

	h->t_min = table[t_col][0];
	h->t_max = table[t_col][n-1];
	h->t_inc = (h->t_max - h->t_min) / (n-1);

	if (p)
		*p = table;
	else {	/* Discard */
		for (j = 0; j < h->n_col; j++) GMT_free ((void *)table[j]);
		GMT_free ((void *)table);
	}
	h->n_row = n;
	if (init_skip_row) {
		info->skip_row = (BOOLEAN *) GMT_memory ((void *)info->skip_row, (size_t)n, sizeof (BOOLEAN), GMT_program);
		if (!GMT_io.binary[GMT_IN]) info->segment_header = (char **) GMT_memory ((void *)info->segment_header, (size_t)n, sizeof (char *), GMT_program);
	}
	/* Fill in NaNs for multisegment headers */
	for (n = 0; n < h->n_row; n++) {
		if (!info->skip_row[n]) continue;
		for (j = 0; j < h->n_col; j++) table[j][n] = GMT_d_NaN;
	}
	if (fp != GMT_stdin) GMT_fclose (fp);
}

void GMT_write_table (struct GMTMATH_INFO *info, char *file, struct TABLE_HEADER *h, double **p, int first_last_all, int cols[])
{	/* first_last_all will write first [-1], last [+1], or all [0] records */
	int i, j, k, start, stop;
	double *out;
	FILE *fp;

	if (!file)
		fp = GMT_stdout;
	else if ((fp = GMT_fopen (file, GMT_io.w_mode)) == NULL) {
		fprintf (stderr, "%s: Error creating file %s\n", GMT_program, file);
		exit (EXIT_FAILURE);
	}

#ifdef SET_IO_MODE
	GMT_setmode (1);
#endif

	out = (double *) GMT_memory (VNULL, (size_t)h->n_col, sizeof (double), GMT_program);

	if (GMT_io.io_header[GMT_OUT] && !GMT_io.binary[GMT_OUT]) {
		fprintf (fp, "%s", info->head_record);
		for (i = 1; i < GMT_io.n_header_recs; i++) fprintf (fp, "# gmtmath header record\n");
	}

	start = (first_last_all == +1) ? h->n_row - 1 : 0;
	stop = (first_last_all == -1) ? 1 : h->n_row;
	for (i = start; i < stop; i++) {
		if (info->skip_row[i]) {
			if (!(GMT_io.binary[GMT_IN] || GMT_io.binary[GMT_OUT])) strcpy (GMT_io.segment_header, info->segment_header[i]);
			GMT_write_segmentheader (GMT_stdout, h->n_col);
		}
		else {	
			for (j = k = 0; j < h->n_col; j++) if (cols[j]) out[k++] = p[j][i];
			GMT_output (fp, k, out);
		}
	}

	if (fp != GMT_stdout) GMT_fclose (fp);
	GMT_free ((void *)out);
}

void new_table (double ***s, int n_col, int n)
{
	int j;
	double **p;
	p = (double **) GMT_memory (VNULL, (size_t)n_col, sizeof (double *), GMT_program);
	for (j = 0; j < n_col; j++) p[j] = (double *) GMT_memory (VNULL, (size_t)n, sizeof (double), GMT_program);
	*s = p;
}

void decode_columns (char *txt, int *skip, int n_col, int t_col, int mode)
{
	int i, start, stop, pos, T, F;
	char p[BUFSIZ];

	if (mode == 0) {	/* For marking columns to skip */
		T = TRUE;
		F = FALSE;
	}
	else {			/* For marking columns to output */
		T = FALSE;
		F = TRUE;
	}
	if (mode == 0 && !txt) {	/* Reset to default */
		for (i = 0; i < n_col; i++) skip[i] = F;
		skip[t_col] = T;
	}
	if (mode == 0 && txt[0] == 'r' && txt[1] == '\0') {	/* Reverse all settings */
		for (i = 0; i < n_col; i++) skip[i] = !skip[i];
	}
	else if (mode == 0 && txt[0] == 'a') {	/* Select all columns */
		for (i = 0; i < n_col; i++) skip[i] = F;
	}
	else {	/* Set the selected columns */
		for (i = 0; i < n_col; i++) skip[i] = T;
		pos = 0;
		while ((GMT_strtok (txt, ",", &pos, p))) {
			if (strchr (p, '-'))
				sscanf (p, "%d-%d", &start, &stop);
			else {
				sscanf (p, "%d", &start);
				stop = start;
			}
			stop = MIN (stop, n_col-1);
			for (i = start; i <= stop; i++) skip[i] = F;
		}
	}
}


/* -----------------------------------------------------------------
 *              Definitions of all operator functions
 * -----------------------------------------------------------------*/

void table_ABS (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand == 0!\n", GMT_program);
	if (constant[last]) a = fabs (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : fabs (stack[last][col][i]);
}

void table_ACOS (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && fabs (factor[last]) > 1.0) fprintf (stderr, "%s: Warning, |operand| > 1 for ACOS!\n", GMT_program);
	if (constant[last]) a = d_acos (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : d_acos (stack[last][col][i]);
}

void table_ACOSH (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && fabs (factor[last]) > 1.0) fprintf (stderr, "%s: Warning, operand < 1 for ACOSH!\n", GMT_program);
	if (constant[last]) a = acosh (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : acosh (stack[last][col][i]);
}

void table_ADD (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = a + b;
	}
}

void table_AND (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = (GMT_is_dnan (a)) ? b : a;
	}
}

void table_ASIN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && fabs (factor[last]) > 1.0) fprintf (stderr, "%s: Warning, |operand| > 1 for ASIN!\n", GMT_program);
	if (constant[last]) a = d_asin (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : d_asin (stack[last][col][i]);
}

void table_ASINH (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = asinh (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : asinh (stack[last][col][i]);
}

void table_ATAN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = atan (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : atan (stack[last][col][i]);
}

void table_ATAN2 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0 for ATAN2!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0 for ATAN2!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = d_atan2 (a, b);
	}
}

void table_ATANH (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && fabs (factor[last]) >= 1.0) fprintf (stderr, "%s: Warning, |operand| >= 1 for ATANH!\n", GMT_program);
	if (constant[last]) a = atanh (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : atanh (stack[last][col][i]);
}

void table_BEI (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_bei (fabs (factor[last]));
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_bei (fabs (stack[last][col][i]));
}

void table_BER (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_ber (fabs (factor[last]));
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_ber (fabs (stack[last][col][i]));
}

void table_CEIL (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = ceil (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : ceil (stack[last][col][i]);
}

void table_CHICRIT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0 for CHICRIT!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0 for CHICRIT!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = GMT_chi2crit (a, b);
	}
}

void table_CHIDIST (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0 for CHIDIST!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0 for CHIDIST!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		GMT_chi2 (a, b, &stack[prev][col][i]);
	}
}

void table_COL (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, k, prev;

	if (!constant[last]) {
		fprintf (stderr, "%s: Error, argument to COL must be a constant column  number (0 <= k < n_col)!\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	prev = last - 1;
	k = irint (factor[last]);
	for (i = 0; i < n_row; i++) {
		stack[last][col][i] = stack[prev][k][i];
	}
}

void table_CORRCOEFF (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double *a, *b, coeff;

	prev = last - 1;
	if (constant[prev] && constant[last]) {	/* Correlation is undefined */
		for (i = 0; i < n_row; i++) stack[prev][col][i] = GMT_d_NaN;
		return;
	}
	
	if (constant[prev]) {		/* Must create the missing (constant) column */
		a = GMT_memory (VNULL, n_row, sizeof (double), GMT_program);
		for (i = 0; i < n_row; i++) a[i] = factor[prev];
		b = stack[last][col];
	}
	else if (constant[last]) {	/* Must create the missing (constant) column */
		a = stack[prev][col];
		b = GMT_memory (VNULL, n_row, sizeof (double), GMT_program);
		for (i = 0; i < n_row; i++) b[i] = factor[last];
	}
	else {
		a = stack[prev][col];
		b = stack[last][col];
	}
	coeff = GMT_corrcoeff (a, b, n_row, 0);
	for (i = 0; i < n_row; i++) stack[prev][col][i] = coeff;
	if (constant[prev]) GMT_free ((void *)a);
	if (constant[last]) GMT_free ((void *)b);
}


void table_COS (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = cos (factor[last]);
	for (i = 0; i < n_row; i++) {
		stack[last][col][i] = (constant[last]) ? a : cos (stack[last][col][i]);
	}
}

void table_COSD (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = cosd (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : cosd (stack[last][col][i]);
}

void table_COSH (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = cosh (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : cosh (stack[last][col][i]);
}

void table_CPOISS (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0 for CPOISS!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		GMT_cumpoisson (a, b, &stack[prev][col][i]);
	}
}

void table_DDT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double c, left, next_left;

	/* Central 1st difference in t */

	if (info->irregular) fprintf (stderr, "%s: Warning, DDT called on irregularly spaced data (not supported)!\n", GMT_program);
	if (gmtdefs.verbose && constant[last]) fprintf (stderr, "%s: Warning, operand to DDT is constant!\n", GMT_program);

	c = 0.5 / info->header.t_inc;
	i = 0;
	while (info->skip_row[i] && i < n_row) i++;	/* Start of first segment */
	while (i < n_row) {	/* Process each segment */
		next_left = 2.0 * stack[last][col][i] - stack[last][col][i+1];
		while (i < n_row - 1 && !info->skip_row[i+1]) {
			left = next_left;
			next_left = stack[last][col][i];
			stack[last][col][i] = (constant[last]) ? 0.0 : c * (stack[last][col][i+1] - left);
			i++;
		}
		stack[last][col][i] = (constant[last]) ? 0.0 : 2.0 * c * (stack[last][col][i] - next_left);
		i++;
		while (info->skip_row[i] && i < n_row) i++;	/* Start of next segment */
	}
		
}

void table_D2DT2 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double c, left, next_left;

	/* Central 2nd difference in t */

	if (info->irregular) fprintf (stderr, "%s: Warning, D2DT2 called on irregularly spaced data (not supported)!\n", GMT_program);
	if (gmtdefs.verbose && constant[last]) fprintf (stderr, "%s: Warning, operand to D2DT2 is constant!\n", GMT_program);

	c = 1.0 / (info->header.t_inc * info->header.t_inc);
	i = 0;
	while (info->skip_row[i] && i < n_row) i++;	/* Start of first segment */
	while (i < n_row) {	/* Process each segment */
		next_left = stack[last][col][i];
		stack[last][col][i] = 0.0;
		i++;
		while (i < n_row - 1 && !info->skip_row[i+1]) {
			left = next_left;
			next_left = stack[last][col][i];
			stack[last][col][i] = (constant[last]) ? 0.0 : c * (stack[last][col][i+1] - 2 * stack[last][col][i] + left);
			i++;
		}
		stack[last][col][i] = 0.0;
		i++;
		while (info->skip_row[i] && i < n_row) i++;	/* Start of next segment */
	}
}

void table_D2R (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = factor[last] * D2R;
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : stack[last][col][i] * D2R;
}

void table_DILOG (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_dilog (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_dilog (stack[last][col][i]);
}

void table_DIV (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;

	if (constant[last] && factor[last] == 0.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Cannot divide by zero\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	if (constant[last]) {	/* Turn divide into multiply */
		a = factor[last];	/* Save old factor */
		factor[last] = 1.0 / factor[last];
		table_MUL (info, stack, constant, factor, last, col, n_row);
		factor[last] = a;	/* Restore factor to original value */
		return;
	}

	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = a / b;
	}
}

void table_DUP (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int next, i;

	next = last + 1;
	factor[next] = factor[last];
	constant[next] = constant[last];
	if (constant[last])
		for (i = 0; i < n_row; i++) stack[next][col][i] = stack[last][col][i] = factor[next];
	else
		memcpy ((void *)stack[next][col], (void *)stack[last][col], (size_t)(n_row * sizeof (double)));
}

void table_ERF (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = erf (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : erf (stack[last][col][i]);
}

void table_ERFC (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = erfc (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : erfc (stack[last][col][i]);
}

void table_ERFINV (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_erfinv (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_erfinv (stack[last][col][i]);
}

void table_EQ (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = (double)(a == b);
	}
}

void table_EXCH (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;

	prev = last - 1;

	for (i = 0; i < n_row; i++) {
		if (constant[last]) stack[last][col][i] = factor[last];
		if (constant[prev]) stack[prev][col][i] = factor[prev];
		d_swap (stack[last][col][i], stack[prev][col][i]);
	}
	i_swap (constant[last], constant[prev]);
	d_swap (factor[last], factor[prev]);
}

void table_EXP (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand == 0!\n", GMT_program);
	if (constant[last]) a = exp (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : exp (stack[last][col][i]);
}

void table_FCRIT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, nu1, nu2, prev1, prev2;
	double alpha;

	prev1 = last - 1;
	prev2 = last - 2;
	if (gmtdefs.verbose && constant[prev2] && factor[prev2] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0 for FCRIT!\n", GMT_program);
	if (gmtdefs.verbose && constant[prev1] && factor[prev1] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0 for FCRIT!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand three == 0 for FCRIT!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		alpha = (constant[prev2]) ? factor[prev2] : stack[prev2][col][i];
		nu1 = irint ((double)((constant[prev1]) ? factor[prev1] : stack[prev1][col][i]));
		nu2 = irint ((double)((constant[last]) ? factor[last] : stack[last][col][i]));
		stack[prev2][col][i] = GMT_Fcrit (alpha, (double)nu1, (double)nu2);
	}
}

void table_FDIST (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, nu1, nu2, prev1, prev2;
	double F, chisq1, chisq2 = 1.0;

	prev1 = last - 1;
	prev2 = last - 2;
	if (gmtdefs.verbose && constant[prev1] && factor[prev1] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0 for FDIST!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand three == 0 for FDIST!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		F = (constant[prev2]) ? factor[prev2] : stack[prev2][col][i];
		nu1 = irint ((double)((constant[prev1]) ? factor[prev1] : stack[prev1][col][i]));
		nu2 = irint ((double)((constant[last]) ? factor[last] : stack[last][col][i]));
		/* Since GMT_f_q needs chisq1 and chisq2, we set chisq2 = 1 and solve for chisq1 */
		chisq1 = F * nu1 / nu2;
		(void) GMT_f_q (chisq1, nu1, chisq2, nu2, &stack[prev2][col][i]);
	}
}

void table_FLIPUD (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, k;
	/* Reverse the order of points in a column */
	if (constant[last]) return;
	if (gmtdefs.verbose && GMT_io.multi_segments[GMT_IN]) {
		fprintf (stderr, "%s: Warning, FLIPUD on multisegment file not supported!\n", GMT_program);
		return;
	}
	for (i = 0, k = n_row-1; i < n_row/2; i++, k--) d_swap (stack[last][col][i], stack[last][col][k]);
}

void table_FLOOR (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = floor (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : floor (stack[last][col][i]);
}

void table_FMOD (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = fmod (a, b);
	}
}

void table_GE (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = (double)(a >= b);
	}
}

void table_GT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = (double)(a > b);
	}
}

void table_HYPOT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = hypot (a, b);
	}
}

void table_I0 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_i0 (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_i0 (stack[last][col][i]);
}

void table_I1 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_i1 (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_i1 (stack[last][col][i]);
}

void table_IN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev, order = 0;
	BOOLEAN simple = FALSE;
	double b = 0.0;

	prev = last - 1;
	if (constant[last]) {
		if (gmtdefs.verbose && factor[last] < 0.0) fprintf (stderr, "%s: Warning, order < 0 for IN!\n", GMT_program);
		if (gmtdefs.verbose && fabs (rint(factor[last]) - factor[last]) > GMT_SMALL) fprintf (stderr, "%s: Warning, order not an integer for IN!\n", GMT_program);
		order = irint (fabs (factor[last]));
		if (constant[prev]) {
			b = GMT_in (order, fabs (factor[prev]));
			simple = TRUE;
		}
	}
	for (i = 0; i < n_row; i++) {
		if (simple)
			stack[prev][col][i] = b;
		else {
			if (!constant[last]) order = irint (fabs (stack[last][col][i]));
			stack[prev][col][i] = GMT_in (order, fabs (stack[prev][col][i]));
		}
	}
}

void table_INT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double f = 0.0, left, right, sum;

	if (constant[last]) {	/* Trivial case */
		sum = factor[last] * info->header.t_inc;
		for (i = 0; i < n_row; i++) stack[last][col][i] = i * sum;
		return;
	}

	/* We use dumb trapezoidal rule - one day we will replace with more sophisticated rules */

	sum = 0.0;
	if (!info->irregular) f = 0.5 * info->header.t_inc;
	i = 0;
	while (info->skip_row[i] && i < n_row) i++;	/* Wind to first segment */
	while (i < n_row) {
		left = stack[last][col][i];
		stack[last][col][i] = sum;
		i++;
		while (i < n_row && !info->skip_row[i]) {	/* Dumb trapezoidal rule */
			if (info->irregular) f = 0.5 * (info->t_coordinates[i] - info->t_coordinates[i-1]);
			right = stack[last][col][i];
			sum += f * (left + right);
			stack[last][col][i] = sum;
			left = right;
			i++;
		}
		while (info->skip_row[i] && i < n_row) i++;	/* Wind to first segment */
	}
}

void table_INV (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last] && factor[last] == 0.0) {
		fprintf (stderr, "%s: Error, Cannot take inverse of zero!\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	if (constant[last]) a = 1.0 / factor[last];
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : 1.0 / stack[last][col][i];
}

void table_ISNAN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = (double)GMT_is_dnan (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : (double)GMT_is_dnan (stack[last][col][i]);
}

void table_J0 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = j0 (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : j0 (stack[last][col][i]);
}

void table_J1 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = j1 (fabs (factor[last]));
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : j1 (fabs (stack[last][col][i]));
}

void table_JN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev, order = 0;
	BOOLEAN simple = FALSE;
	double b = 0.0;

	prev = last - 1;
	if (constant[last]) {
		if (gmtdefs.verbose && factor[last] < 0.0) fprintf (stderr, "%s: Warning, order < 0 for JN!\n", GMT_program);
		if (gmtdefs.verbose && fabs (rint(factor[last]) - factor[last]) > GMT_SMALL) fprintf (stderr, "%s: Warning, order not an integer for JN!\n", GMT_program);
		order = irint (fabs (factor[last]));
		if (constant[prev]) {
			b = jn (order, fabs (factor[prev]));
			simple = TRUE;
		}
	}
	for (i = 0; i < n_row; i++) {
		if (simple)
			stack[prev][col][i] = b;
		else {
			if (!constant[last]) order = irint (fabs (stack[last][col][i]));
			stack[prev][col][i] = jn (order, fabs (stack[prev][col][i]));
		}
	}
}

void table_K0 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_k0 (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_k0 (stack[last][col][i]);
}

void table_K1 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_k1 (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_k1 (stack[last][col][i]);
}

void table_KN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev, order = 0;
	BOOLEAN simple = FALSE;
	double b = 0.0;

	prev = last - 1;
	if (constant[last]) {
		if (gmtdefs.verbose && factor[last] < 0.0) fprintf (stderr, "%s: Warning, order < 0 for KN!\n", GMT_program);
		if (gmtdefs.verbose && fabs (rint(factor[last]) - factor[last]) > GMT_SMALL) fprintf (stderr, "%s: Warning, order not an integer for KN!\n", GMT_program);
		order = irint (fabs (factor[last]));
		if (constant[prev]) {
			b = GMT_kn (order, fabs (factor[prev]));
			simple = TRUE;
		}
	}
	for (i = 0; i < n_row; i++) {
		if (simple)
			stack[prev][col][i] = b;
		else {
			if (!constant[last]) order = irint (fabs (stack[last][col][i]));
			stack[prev][col][i] = GMT_kn (order, fabs (stack[prev][col][i]));
		}
	}
}

void table_KEI (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_kei (fabs (factor[last]));
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_kei (fabs (stack[last][col][i]));
}

void table_KER (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_ker (fabs (factor[last]));
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_ker (fabs (stack[last][col][i]));
}

void table_LE (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = (double)(a <= b);
	}
}

void table_LMSSCL (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	size_t i;
	int GMT_mode_selection = 0, GMT_n_multiples = 0;

	double lmsscl, mode;

	if (constant[last]) {	/* Trivial case */
		memset ((void *)stack[last][col], 0, (size_t)(n_row * sizeof (double)));
		return;
	}

	qsort ((void *)stack[last][col], (size_t)n_row, sizeof (double), GMT_comp_double_asc);
	for (i = n_row; GMT_is_fnan (stack[last][col][i-1]) && i > 1; i--);
	if (i) {
		GMT_mode (stack[last][col], i, i/2, 0, GMT_mode_selection, &GMT_n_multiples, &mode);
		GMT_getmad (stack[last][col], i, mode, &lmsscl);
	}
	else
		lmsscl = GMT_d_NaN;

	for (i = 0; i < (size_t)n_row; i++) stack[last][col][i] = lmsscl;
	if (GMT_n_multiples > 0) fprintf (stderr, "%s: WARNING: %d Multiple modes found\n", GMT_program, GMT_n_multiples);
}

void table_LOG (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, argument to log = 0!\n", GMT_program);

	if (constant[last]) a = d_log (fabs (factor[last]));
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : d_log (fabs (stack[last][col][i]));
}

void table_LOG10 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, argument to log10 = 0!\n", GMT_program);

	if (constant[last]) a = d_log10 (fabs (factor[last]));
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : d_log10 (fabs (stack[last][col][i]));
}

void table_LOG1P (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] < 0.0) fprintf (stderr, "%s: Warning, argument to log1p < 0!\n", GMT_program);

	if (constant[last]) a = d_log1p (fabs (factor[last]));
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : d_log1p (fabs (stack[last][col][i]));
}

void table_LOG2 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, argument to log2 = 0!\n", GMT_program);

	if (constant[last]) a = d_log (fabs (factor[last])) * M_LN2_INV;
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : d_log (fabs (stack[last][col][i])) * M_LN2_INV;
}

void table_LOWER (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double low;

	if (constant[last]) {	/* Trivial case */
		for (i = 0; i < n_row; i++) stack[last][col][i] = factor[last];
		return;
	}

	for (i = 0, low = DBL_MAX; i < n_row; i++) {
		if (GMT_is_fnan (stack[last][col][i])) continue;
		if (stack[last][col][i] < low) low = stack[last][col][i];
	}
	for (i = 0; i < n_row; i++) if (!GMT_is_fnan (stack[last][col][i])) stack[last][col][i] = low;
}

void table_LRAND (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a = 0.0, b = 0.0;

	prev = last - 1;
	if (constant[prev]) a = factor[prev];
	if (constant[last]) b = factor[last];
	for (i = 0; i < n_row; i++) {
		if (!constant[prev]) a = stack[prev][col][i];
		if (!constant[last]) b = stack[last][col][i];
		stack[prev][col][i] = a + b * GMT_lrand ();
	}
}

void table_LSQFIT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	/* Dummy routine needed since the automatically generated include file will have table_LSQFIT
	 * with these parameters just like any other function.  However, when we find LSQFIT we will
	 * instead call solve_LSQFIT which can be found at the end of these functions */
}

void table_LT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = (double)(a < b);
	}
}

void table_MAD (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double mad, med;

	if (constant[last]) {	/* Trivial case */
		memset ((void *)stack[last][col], 0, (size_t)(n_row * sizeof (double)));
		return;
	}

	qsort ((void *)stack[last][col], (size_t)n_row, sizeof (double), GMT_comp_double_asc);
	for (i = n_row; GMT_is_fnan (stack[last][col][i-1]) && i > 1; i--);
	if (i) {
		med = (i%2) ? stack[last][col][i/2] : 0.5 * (stack[last][col][(i-1)/2] + stack[last][col][i/2]);
		GMT_getmad (stack[last][col], i, med, &mad);
	}
	else
		mad = GMT_d_NaN;

	for (i = 0; i < n_row; i++) stack[last][col][i] = mad;
}

void table_MAX (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = MAX (a, b);
	}
}

void table_MEAN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, n_a = 0;
	double sum_a = 0.0;

	if (constant[last]) {	/* Trivial case */
		for (i = 0; i < n_row; i++) stack[last][col][i] = factor[last];
		return;
	}

	for (i = 0; i < n_row; i++) {
		if (GMT_is_fnan (stack[last][col][i])) continue;
		sum_a += stack[last][col][i];
		n_a++;
	}
	sum_a = (n_a) ? sum_a / n_a : 0.0;
	for (i = 0; i < n_row; i++) stack[last][col][i] = sum_a;
}

void table_MED (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double med;

	if (constant[last]) {	/* Trivial case */
		for (i = 0; i < n_row; i++) stack[last][col][i] = factor[last];
		return;
	}

	qsort ((void *)stack[last][col], (size_t)n_row, sizeof (double), GMT_comp_double_asc);
	for (i = n_row; GMT_is_fnan (stack[last][col][i-1]) && i > 1; i--);
	if (i)
		med = (i%2) ? stack[last][col][i/2] : 0.5 * (stack[last][col][(i-1)/2] + stack[last][col][i/2]);
	else
		med = GMT_d_NaN;

	for (i = 0; i < n_row; i++) stack[last][col][i] = med;
}

void table_MIN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = MIN (a, b);
	}
}

void table_MODE (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	size_t i;
	int GMT_mode_selection = 0, GMT_n_multiples = 0;

	double mode;

	if (constant[last]) {	/* Trivial case */
		for (i = 0; i < (size_t)n_row; i++) stack[last][col][i] = factor[last];
		return;
	}

	qsort ((void *)stack[last][col], (size_t)n_row, sizeof (double), GMT_comp_double_asc);
	for (i = n_row; GMT_is_fnan (stack[last][col][i-1]) && i > 1; i--);
	if (i)
		GMT_mode (stack[last][col], i, i/2, 0, GMT_mode_selection, &GMT_n_multiples, &mode);
	else
		mode = GMT_d_NaN;

	for (i = 0; i < (size_t)n_row; i++) stack[last][col][i] = mode;
	if (GMT_n_multiples > 0) fprintf (stderr, "%s: WARNING: %d Multiple modes found\n", GMT_program, GMT_n_multiples);
}

void table_MUL (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = a * b;
	}
}

void table_NAN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a = 0.0, b = 0.0;

	prev = last - 1;
	if (constant[prev]) a = factor[prev];
	if (constant[last]) b = factor[last];
	for (i = 0; i < n_row; i++) {
		if (!constant[prev]) a = stack[prev][col][i];
		if (!constant[last]) b = stack[last][col][i];
		stack[prev][col][i] = ((a == b) ? GMT_d_NaN : a);
	}
}

void table_NEG (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand == 0!\n", GMT_program);
	if (constant[last]) a = -factor[last];
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : -stack[last][col][i];
}

void table_NEQ (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = (double)(a != b);
	}
}

void table_NRAND (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a = 0.0, b = 0.0;

	prev = last - 1;
	if (constant[prev]) a = factor[prev];
	if (constant[last]) b = factor[last];
	for (i = 0; i < n_row; i++) {
		if (!constant[prev]) a = stack[prev][col][i];
		if (!constant[last]) b = stack[last][col][i];
		stack[prev][col][i] = a + b * GMT_nrand ();
	}
}

void table_OR (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = (GMT_is_dnan (a) || GMT_is_dnan (b)) ? GMT_d_NaN : a;
	}
}

void table_PLM (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev, first, L, M;
	double a = 0.0;
				/* last holds the order M */
	prev  = last - 1;	/* prev holds the degree L */
	first = prev - 1;	/* first holds the argument x = cos(colat) */

	if (!(constant[prev] && constant[last])) {
		fprintf (stderr, "%s: L and M must be constants in PLM!\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	L = irint (factor[prev]);
	M = irint (factor[last]);
	if (gmtdefs.verbose && constant[first] && (factor[first] < -1.0 || factor[first] > 1.0)) fprintf (stderr, "%s: Warning, argument to PLM outside domain!\n", GMT_program);

	if (constant[first]) a = GMT_plm (L, M, factor[first]);
	for (i = 0; i < n_row; i++) stack[first][col][i] = (constant[first]) ? a : GMT_plm (L, M, stack[first][col][i]);
}

void table_POP (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	/* Dummy routine that does nothing but consume the top element of stack */
}

void table_POW (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;

	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = pow (a, b);
	}
}

void table_R2 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0!\n", GMT_program);
	if (constant[prev]) factor[prev] *= factor[prev];
	if (constant[last]) factor[last] *= factor[last];
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i] * stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i] * stack[last][col][i];
		stack[prev][col][i] = a + b;
	}
}

void table_R2D (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = factor[last] * R2D;
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : stack[last][col][i] * R2D;
}

void table_RAND (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a = 0.0, b = 0.0;

	prev = last - 1;
	if (constant[prev]) a = factor[prev];
	if (constant[last]) b = factor[last];
	for (i = 0; i < n_row; i++) {
		if (!constant[prev]) a = stack[prev][col][i];
		if (!constant[last]) b = stack[last][col][i];
		stack[prev][col][i] = a + GMT_rand () * (b - a);
	}
}

void table_RINT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = rint (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : rint (stack[last][col][i]);
}

void table_ROTT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, shift, prev;
	double *z;

	if (!constant[last]) {
		fprintf (stderr, "%s: T-shift must be a constant in ROTT\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	prev = last - 1;
	shift = irint (factor[last] / info->header.t_inc);
	if (constant[prev] || !shift) return;	/* Easy, constant or no shift */
	if (shift < 0) shift += n_row;		/* Same thing */

	z = (double *) GMT_memory (VNULL, (size_t)(n_row), sizeof (double), GMT_program);

	for (i = 0; i < n_row; i++) z[(i+shift)%n_row] = stack[prev][col][i];
	memcpy ((void *)stack[prev][col], (void *)z, (size_t)(n_row * sizeof (double)));
	GMT_free ((void *)z);
}

void table_SIGN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand == 0!\n", GMT_program);
	if (constant[last]) a = copysign (1.0, factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : copysign (1.0, stack[last][col][i]);
}

void table_SIN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = sin (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : sin (stack[last][col][i]);
}

void table_SINC (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_sinc (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_sinc (stack[last][col][i]);
}

void table_SIND (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = sind (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : sind (stack[last][col][i]);
}

void table_SINH (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = sinh (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : sinh (stack[last][col][i]);
}

void table_SQRT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0!\n", GMT_program);
	if (constant[last]) a = sqrt (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : sqrt (stack[last][col][i]);
}

void table_STD (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, n_a = 0;
	double sum_a = 0.0, sum_a2 = 0.0, std;

	if (constant[last]) {	/* Trivial case */
		memset ((void *)stack[last][col], 0, (size_t)(n_row * sizeof (double)));
		return;
	}

	for (i = 0; i < n_row; i++) {
		if (GMT_is_fnan (stack[last][col][i])) continue;
		sum_a += stack[last][col][i];
		sum_a2 += (stack[last][col][i] * stack[last][col][i]);
		n_a++;
	}
	if (n_a > 1)
		std = sqrt ((n_a * sum_a2 - sum_a * sum_a) / (n_a * (n_a - 1.0)));
	else
		std = 0.0;
	for (i = 0; i < n_row; i++) stack[last][col][i] = std;
}

void table_STEP (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a;

	for (i = 0; i < n_row; i++) {
		a = (constant[last]) ? factor[last] : stack[last][col][i];
		if (a == 0.0)
			stack[last][col][i] = 0.5;
		else
			stack[last][col][i] = (a < 0.0) ? 0.0 : 1.0;
	}
}

void table_STEPT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a;

	for (i = 0; i < n_row; i++) {
		a = info->t_coordinates[i] - ((constant[last]) ? factor[last] : stack[last][col][i]);
		if (a == 0.0)
			stack[last][col][i] = 0.5;
		else
			stack[last][col][i] = (a < 0.0) ? 0.0 : 1.0;
	}
}

void table_SUB (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = a - b;
	}
}

void table_SUM (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0, sum = 0.0;

	if (constant[last]) a = factor[last];
	for (i = 0; i < n_row; i++) {
		if (!constant[last]) a = stack[last][col][i];
		if (!GMT_is_dnan (a)) sum += a;
		stack[last][col][i] = sum;
	}
}

void table_TAN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = tan (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : tan (stack[last][col][i]);
}

void table_TAND (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = tand (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : tand (stack[last][col][i]);
}

void table_TANH (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = tanh (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : tanh (stack[last][col][i]);
}

void table_TN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev, n;
	double a;

	prev = last - 1;
	for (i = 0; i < n_row; i++) {
		n = irint ((constant[last]) ? factor[last] : stack[last][col][i]);
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		stack[prev][col][i] = GMT_chebyshev (a, n);
	}
}

void table_TCRIT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a, b;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0 for TCRIT!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0 for TCRIT!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = (constant[last]) ? factor[last] : stack[last][col][i];
		stack[prev][col][i] = GMT_tcrit (a, b);
	}
}

void table_TDIST (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, b, prev;
	double a;

	prev = last - 1;
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, operand one == 0 for TDIST!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand two == 0 for TDIST!\n", GMT_program);
	for (i = 0; i < n_row; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][col][i];
		b = irint ((constant[last]) ? factor[last] : stack[last][col][i]);
		(void) GMT_student_t_a (a, b, &stack[prev][col][i]);
	}
}

void table_UPPER (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double high;

	if (constant[last]) {	/* Trivial case */
		for (i = 0; i < n_row; i++) stack[last][col][i] = factor[last];
		return;
	}

	for (i = 0, high = -DBL_MAX; i < n_row; i++) {
		if (GMT_is_fnan (stack[last][col][i])) continue;
		if (stack[last][col][i] > high) high = stack[last][col][i];
	}
	for (i = 0; i < n_row; i++) if (!GMT_is_fnan (stack[last][col][i])) stack[last][col][i] = high;
}

void table_XOR (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double a = 0.0, b = 0.0;

	prev = last - 1;
	if (constant[prev]) a = factor[prev];
	if (constant[last]) b = factor[last];
	for (i = 0; i < n_row; i++) {
		if (!constant[prev]) a = stack[prev][col][i];
		if (!constant[last]) b = stack[last][col][i];
		stack[prev][col][i] = (GMT_is_dnan (a)) ? b : a;
	}
}

void table_Y0 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand = 0 for Y0!\n", GMT_program);
	if (constant[last]) a = y0 (fabs (factor[last]));
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : y0 (fabs (stack[last][col][i]));
}

void table_Y1 (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (gmtdefs.verbose && constant[last] && factor[last] == 0.0) fprintf (stderr, "%s: Warning, operand = 0 for Y1!\n", GMT_program);
	if (constant[last]) a = y1 (fabs (factor[last]));
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : y1 (fabs (stack[last][col][i]));
}

void table_YN (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev, order = 0;
	BOOLEAN simple = FALSE;
	double b = 0.0;

	prev = last - 1;
	if (gmtdefs.verbose && constant[last] && factor[last] < 0.0) fprintf (stderr, "%s: Warning, order < 0 for YN!\n", GMT_program);
	if (gmtdefs.verbose && constant[last] && fabs (rint(factor[last]) - factor[last]) > GMT_SMALL) fprintf (stderr, "%s: Warning, order not an integer for YN!\n", GMT_program);
	if (gmtdefs.verbose && constant[prev] && factor[prev] == 0.0) fprintf (stderr, "%s: Warning, argument = 0 for YN!\n", GMT_program);
	if (constant[last]) order = irint (fabs (factor[last]));
	if (constant[last] && constant[prev]) {
		b = yn (order, fabs (factor[prev]));
		simple = TRUE;
	}
	for (i = 0; i < n_row; i++) {
		if (simple)
			stack[prev][col][i] = b;
		else {
			if (!constant[last]) order = irint (fabs (stack[last][col][i]));
			stack[prev][col][i] = yn (order, fabs (stack[prev][col][i]));
		}
	}
}

void table_ZCRIT (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_zcrit (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_zcrit (stack[last][col][i]);
}

void table_ZDIST (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i;
	double a = 0.0;

	if (constant[last]) a = GMT_zdist (factor[last]);
	for (i = 0; i < n_row; i++) stack[last][col][i] = (constant[last]) ? a : GMT_zdist (stack[last][col][i]);
}

void table_ROOTS (struct GMTMATH_INFO *info, double **stack[], BOOLEAN *constant, double *factor, int last, int col, int n_row)
{
	int i, prev;
	double *roots;

	/* Treats the chosen column (at there is only one) as f(t) and solves for t that makes f(t) == 0.
	 * For now we only solve using a linear spline but in the future this should depend on the users
	 * choice of INTERPOLANT. */

	if (info->roots_found) return;	/* Already been here */
	if (!constant[last]) {
		fprintf (stderr, "%s: Argument to operator ROOTS must be a constant: the column number. Reset to 0\n", GMT_program);
		info->r_col = 0;
	}
	else
		info->r_col = irint (factor[last]);
	if (info->r_col < 0 || info->r_col >= info->header.n_col) {
		fprintf (stderr, "%s: Argument to operator ROOTS must be a column number 0 < col < %d. Reset to 0\n", GMT_program, info->header.n_col);
		info->r_col = 0;
	}
	roots = (double *) GMT_memory (VNULL, (size_t)(n_row), sizeof (double), GMT_program);
	info->n_roots = 0;
	prev = last - 1;
	if (stack[prev][info->r_col][0] == 0.0) roots[info->n_roots++] = info->t_coordinates[0]; 
	for (i = 1; i < n_row; i++) {
		if (stack[prev][info->r_col][i] == 0.0) {
			roots[info->n_roots++] = info->t_coordinates[i];
			continue;
		}

		if ((stack[prev][info->r_col][i-1] * stack[prev][info->r_col][i]) < 0.0) {	/* Crossing 0 */
			roots[info->n_roots] = info->t_coordinates[i-1] - stack[prev][info->r_col][i-1] * (info->t_coordinates[i] - info->t_coordinates[i-1]) / (stack[prev][info->r_col][i] - stack[prev][info->r_col][i-1]);
			info->n_roots++;
		}
	}
	for (i = 0; i < info->n_roots; i++) stack[prev][info->r_col][i] = roots[i];
	GMT_free ((void *)roots);
	info->roots_found = TRUE;
}

/* ---------------------- end operator functions --------------------- */

void solve_LSQFIT (struct GMTMATH_INFO *info, double **stack[], int last, int n_col, int n_row, int skip[], char *file)
{
	/* Consider the current table the augmented matrix [A | b], making up the linear system Ax = b.
	 * We will set up the normal equations, solve for x, and output the solution before quitting.
	 * This function is special since it operates across columns and returns n_col scalars.
	 * We try to solve this positive definite & symmetric matrix with Cholsky methods; if that fails
	 * we do a full SVD decomposition and set small eigenvalues to zero, yielding an approximate solution.
	 */
	 
	int i, j, k, k0, i2, j2, rhs, n, ier;
	double cond, *N, *B, *d, *x, *b, *z, *v, *lambda;
	FILE *fp;

	for (i = n = 0; i < n_col; i++) if (!skip[i]) n++;	/* Need to find how many active columns we have */
	if (n < 2) {
		fprintf (stderr, "%s: Error, LSQSOL requires at least 2 active columns!\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	rhs = n_col - 1;
	while (skip[rhs] && rhs > 0) rhs--; 	/* Get last active col number as the rhs vector b */
	n--;					/* Account for b, the rhs vector, to get row & col dimensions of normal matrix N */

	N = (double *) GMT_memory (VNULL, (size_t)(n*n), sizeof (double), GMT_program);
	B = (double *) GMT_memory (VNULL, (size_t)n_row, sizeof (double), GMT_program);

	/* Do the row & col dot products, skipping inactive columns as we go along */
	for (j = j2 = 0; j < n; j2++) {	/* j2 is table column, j is row in N matrix */
		if (skip[j2]) continue;
		for (i = i2 = 0; i < n; i2++) {	/* i2 is table column, i is column in N matrix */
			if (skip[i2]) continue;
			k0 = j * n + i;
			N[k0] = 0.0;
			for (k = 0; k < n_row; k++) N[k0] += stack[last][j][k] * stack[last][i][k];
			i++;
		}
		B[j] = 0.0;
		for (k = 0; k < n_row; k++) B[j] += stack[last][j][k] * stack[last][rhs][k];
		j++;
	}

	d = (double *) GMT_memory (VNULL, (size_t)n, sizeof(double), GMT_program);
	x = (double *) GMT_memory (VNULL, (size_t)n, sizeof(double), GMT_program);
	if ( (ier = GMT_chol_dcmp (N, d, &cond, n, n) ) != 0) {	/* Decomposition failed, use SVD method */
		int nrots;
		GMT_chol_recover (N, d, n, n, ier, TRUE);		/* Restore to former matrix N */
		/* Solve instead using GMT_jacobi */
		lambda = (double *) GMT_memory (VNULL, (size_t)n, sizeof(double), GMT_program);
		b = (double *) GMT_memory (VNULL, (size_t)n, sizeof(double), GMT_program);
		z = (double *) GMT_memory (VNULL, (size_t)n, sizeof(double), GMT_program);
		v = (double *) GMT_memory (VNULL, (size_t)n*n, sizeof(double), GMT_program);

		if (GMT_jacobi (N, &n, &n, lambda, v, b, z, &nrots)) {
			fprintf (stderr, "%s: Eigenvalue routine failed to converge in 50 sweeps.\n", GMT_program);
			fprintf (stderr, "%s: The reported L2 positions might be garbage.\n", GMT_program);
		}
		/* Solution x = v * lambda^-1 * v' * B */

		/* First do d = V' * B, so x = v * lambda^-1 * d */
		for (j = 0; j < n; j++) for (k = 0, d[j] = 0.0; k < n; k++) d[j] += v[k*n+j] * B[k];
		/* Then do d = lambda^-1 * d by setting small lambda's to zero */
		for (j = k = 0; j < n; j++) {
			if (lambda[j] < 1.0e7) {
				d[j] = 0.0;
				k++;
			}
			else
				d[j] /= lambda[j];
		}
		if (k) fprintf (stderr,"%s: %d eigenvalues < 1.0e-7 set to zero to yield a stable solution\n", GMT_program, k);

		/* Finally do x = v * d */
		for (j = 0; j < n; j++) for (k = 0; k < n; k++) x[j] += v[j*n+k] * d[k];

		GMT_free ((void *)b);
		GMT_free ((void *)z);
		GMT_free ((void *)v);
	}
	else {	/* Decomposition worked, now solve system */
		GMT_chol_solv (N, x, B, n, n);
	}

	if (!file)
		fp = GMT_stdout;
	else if ((fp = GMT_fopen (file, GMT_io.w_mode)) == NULL) {
		fprintf (stderr, "%s: Error creating file %s\n", GMT_program, file);
		exit (EXIT_FAILURE);
	}

#ifdef SET_IO_MODE
	GMT_setmode (1);
#endif
	GMT_output (fp, n, x);
	if (fp != GMT_stdout) GMT_fclose (fp);

	GMT_free ((void *)x);
	GMT_free ((void *)d);
	GMT_free ((void *)N);
	GMT_free ((void *)B);
}

/* ---------------------- start convenience functions --------------------- */

int decode_argument (char *txt, double *value, struct GMT_HASH *H) {
	int number, minus, plus, exp, dec, n_digits = 0, expect, i;
	char *s, copy[GMT_LONG_TEXT];
	int get_operator (char *choice, struct GMT_HASH *H);

	/* Check if argument is operator */

	if ((i = get_operator (txt, H)) >= GMTMATH_ARG_IS_OPERATOR) return (i);

	/* Next look for symbols with special meaning */

	if (!(strcmp (txt, "STDIN"))) return GMTMATH_ARG_IS_FILE;	/* read from stdin */
	if (!(strcmp (txt, "PI") && strcmp (txt, "pi"))) return GMTMATH_ARG_IS_PI;
	if (!(strcmp (txt, "E") && strcmp (txt, "e"))) return GMTMATH_ARG_IS_E;
	if (!(strcmp (txt, "T") && strcmp (txt, "t"))) return GMTMATH_ARG_IS_T_MATRIX;
	if (!(strcmp (txt, "Tn") && strcmp (txt, "tn"))) return GMTMATH_ARG_IS_t_MATRIX;

	strcpy (copy, txt);
	
	expect = (strchr (copy, 'T')) ? GMT_IS_ABSTIME : GMT_IS_UNKNOWN;
	if (GMT_scanf (copy, expect, value) != GMT_IS_NAN) return GMTMATH_ARG_IS_NUMBER;

	/* Here we must check if argument is a numerical value */

	s = txt;
	if (*s == '-' || *s == '+') s++;	/* Skip leading sign */

	minus = plus = exp = dec = 0;
	number = TRUE;

	while (number && *s) {
		if (isdigit ((int)*s))
			n_digits++;
		else {
			switch (*s) {
				case '-':
					minus++;
					break;
				case '+':
					plus++;
					break;
				case 'E':
				case 'e':
					exp++;
					break;
				case '.':
					dec++;
					break;
				default:
					number = FALSE;
					break;
			}
		}
		if (minus > 1 || exp > 1 || dec > 1) number = FALSE;
		s++;
	}
	if (txt[0] == '-' && !number) {	/* Probably a bad commandline option */
		fprintf (stderr, "%s: ERROR: Option %s not recognized\n", GMT_program, txt);
		exit (EXIT_FAILURE);
	}
	if (number && n_digits > 0) return GMTMATH_ARG_IS_NUMBER;

	/* Determine if argument is file */

	if (!GMT_access (txt, R_OK)) return GMTMATH_ARG_IS_FILE;

	fprintf (stderr, "%s: GMT SYNTAX ERROR: %s is not a number, operator or file name\n", GMT_program, txt);
	exit (EXIT_FAILURE);
}

int get_operator (char *choice, struct GMT_HASH *H)
{
	int op;

	/* Returns -1 if not a registered operator */

	op = GMT_hash_lookup (choice, H, GMT_HASH_SIZE, GMT_HASH_SIZE);

	if (op < 0 && strlen (choice) == 1) {	/* Check for old-style operators */

		switch (choice[0]) {
			case '+':
				op = ADD;
				break;
			case '-':
				op = SUB;
				break;
			case 'x':
				op = MUL;
				break;
			case '/':
				op = DIV;
				break;
			case '^':
				op = RAISE;
				break;
		}
	}

	return (op);
}

void *New_Gmtmath_Ctrl () {	/* Allocate and initialize a new control structure */
	struct GMTMATH_CTRL *C;
	
	C = (struct GMTMATH_CTRL *) GMT_memory (VNULL, 1, sizeof (struct GMTMATH_CTRL), "New_Gmtmath_Ctrl");
	
	/* Initialize values whose defaults are not 0/FALSE/NULL */
	
	C->C.cols = (int *) GMT_memory (VNULL, BUFSIZ, sizeof (int), GMT_program);
	C->F.cols = (int *) GMT_memory (VNULL, BUFSIZ, sizeof (int), GMT_program);
	C->N.ncol = 2;
		
	return ((void *)C);
}

void Free_Gmtmath_Ctrl (struct GMTMATH_CTRL *C) {	/* Deallocate control structure */
	if (C->A.file) GMT_free ((void *)C->A.file);	
	GMT_free ((void *)C->C.cols);
	GMT_free ((void *)C->F.cols);
	if (C->T.file) GMT_free ((void *)C->T.file);	
	GMT_free ((void *)C);	
}

void gmtmath_init (PFV ops[], int n_args[], int n_out[])
{

	/* Operator function		# of operands  		# of outputs */

	ops[0] = table_ABS;		n_args[0] = 1;		n_out[0] = 1;
	ops[1] = table_ACOS;		n_args[1] = 1;		n_out[1] = 1;
	ops[2] = table_ACOSH;		n_args[2] = 1;		n_out[2] = 1;
	ops[3] = table_ADD;		n_args[3] = 2;		n_out[3] = 1;
	ops[4] = table_AND;		n_args[4] = 2;		n_out[4] = 1;
	ops[5] = table_ASIN;		n_args[5] = 1;		n_out[5] = 1;
	ops[6] = table_ASINH;		n_args[6] = 1;		n_out[6] = 1;
	ops[7] = table_ATAN;		n_args[7] = 1;		n_out[7] = 1;
	ops[8] = table_ATAN2;		n_args[8] = 2;		n_out[8] = 1;
	ops[9] = table_ATANH;		n_args[9] = 1;		n_out[9] = 1;
	ops[10] = table_BEI;		n_args[10] = 1;		n_out[10] = 1;
	ops[11] = table_BER;		n_args[11] = 1;		n_out[11] = 1;
	ops[12] = table_CEIL;		n_args[12] = 1;		n_out[12] = 1;
	ops[13] = table_CHICRIT;		n_args[13] = 2;		n_out[13] = 1;
	ops[14] = table_CHIDIST;		n_args[14] = 2;		n_out[14] = 1;
	ops[15] = table_CORRCOEFF;		n_args[15] = 2;		n_out[15] = 1;
	ops[16] = table_COL;		n_args[16] = 1;		n_out[16] = 1;
	ops[17] = table_COS;		n_args[17] = 1;		n_out[17] = 1;
	ops[18] = table_COSD;		n_args[18] = 1;		n_out[18] = 1;
	ops[19] = table_COSH;		n_args[19] = 1;		n_out[19] = 1;
	ops[20] = table_CPOISS;		n_args[20] = 2;		n_out[20] = 1;
	ops[21] = table_D2DT2;		n_args[21] = 1;		n_out[21] = 1;
	ops[22] = table_D2R;		n_args[22] = 1;		n_out[22] = 1;
	ops[23] = table_DILOG;		n_args[23] = 1;		n_out[23] = 1;
	ops[24] = table_DIV;		n_args[24] = 2;		n_out[24] = 1;
	ops[25] = table_DDT;		n_args[25] = 1;		n_out[25] = 1;
	ops[26] = table_DUP;		n_args[26] = 1;		n_out[26] = 2;
	ops[27] = table_ERF;		n_args[27] = 1;		n_out[27] = 1;
	ops[28] = table_ERFC;		n_args[28] = 1;		n_out[28] = 1;
	ops[29] = table_ERFINV;		n_args[29] = 1;		n_out[29] = 1;
	ops[30] = table_EQ;		n_args[30] = 2;		n_out[30] = 1;
	ops[31] = table_EXCH;		n_args[31] = 2;		n_out[31] = 2;
	ops[32] = table_EXP;		n_args[32] = 1;		n_out[32] = 1;
	ops[33] = table_FCRIT;		n_args[33] = 3;		n_out[33] = 1;
	ops[34] = table_FDIST;		n_args[34] = 3;		n_out[34] = 1;
	ops[35] = table_FLIPUD;		n_args[35] = 1;		n_out[35] = 1;
	ops[36] = table_FLOOR;		n_args[36] = 1;		n_out[36] = 1;
	ops[37] = table_FMOD;		n_args[37] = 2;		n_out[37] = 1;
	ops[38] = table_GE;		n_args[38] = 2;		n_out[38] = 1;
	ops[39] = table_GT;		n_args[39] = 2;		n_out[39] = 1;
	ops[40] = table_HYPOT;		n_args[40] = 2;		n_out[40] = 1;
	ops[41] = table_I0;		n_args[41] = 1;		n_out[41] = 1;
	ops[42] = table_I1;		n_args[42] = 1;		n_out[42] = 1;
	ops[43] = table_IN;		n_args[43] = 2;		n_out[43] = 1;
	ops[44] = table_INT;		n_args[44] = 1;		n_out[44] = 1;
	ops[45] = table_INV;		n_args[45] = 1;		n_out[45] = 1;
	ops[46] = table_ISNAN;		n_args[46] = 1;		n_out[46] = 1;
	ops[47] = table_J0;		n_args[47] = 1;		n_out[47] = 1;
	ops[48] = table_J1;		n_args[48] = 1;		n_out[48] = 1;
	ops[49] = table_JN;		n_args[49] = 2;		n_out[49] = 1;
	ops[50] = table_K0;		n_args[50] = 1;		n_out[50] = 1;
	ops[51] = table_K1;		n_args[51] = 1;		n_out[51] = 1;
	ops[52] = table_KN;		n_args[52] = 2;		n_out[52] = 1;
	ops[53] = table_KEI;		n_args[53] = 1;		n_out[53] = 1;
	ops[54] = table_KER;		n_args[54] = 1;		n_out[54] = 1;
	ops[55] = table_LE;		n_args[55] = 2;		n_out[55] = 1;
	ops[56] = table_LMSSCL;		n_args[56] = 1;		n_out[56] = 1;
	ops[57] = table_LOG;		n_args[57] = 1;		n_out[57] = 1;
	ops[58] = table_LOG10;		n_args[58] = 1;		n_out[58] = 1;
	ops[59] = table_LOG1P;		n_args[59] = 1;		n_out[59] = 1;
	ops[60] = table_LOG2;		n_args[60] = 1;		n_out[60] = 1;
	ops[61] = table_LOWER;		n_args[61] = 1;		n_out[61] = 1;
	ops[62] = table_LRAND;		n_args[62] = 2;		n_out[62] = 1;
	ops[63] = table_LSQFIT;		n_args[63] = 1;		n_out[63] = 0;
	ops[64] = table_LT;		n_args[64] = 2;		n_out[64] = 1;
	ops[65] = table_MAD;		n_args[65] = 1;		n_out[65] = 1;
	ops[66] = table_MAX;		n_args[66] = 2;		n_out[66] = 1;
	ops[67] = table_MEAN;		n_args[67] = 1;		n_out[67] = 1;
	ops[68] = table_MED;		n_args[68] = 1;		n_out[68] = 1;
	ops[69] = table_MIN;		n_args[69] = 2;		n_out[69] = 1;
	ops[70] = table_MODE;		n_args[70] = 1;		n_out[70] = 1;
	ops[71] = table_MUL;		n_args[71] = 2;		n_out[71] = 1;
	ops[72] = table_NAN;		n_args[72] = 2;		n_out[72] = 1;
	ops[73] = table_NEG;		n_args[73] = 1;		n_out[73] = 1;
	ops[74] = table_NEQ;		n_args[74] = 2;		n_out[74] = 1;
	ops[75] = table_NRAND;		n_args[75] = 2;		n_out[75] = 1;
	ops[76] = table_OR;		n_args[76] = 2;		n_out[76] = 1;
	ops[77] = table_PLM;		n_args[77] = 3;		n_out[77] = 1;
	ops[78] = table_POP;		n_args[78] = 1;		n_out[78] = 0;
	ops[79] = table_POW;		n_args[79] = 2;		n_out[79] = 1;
	ops[80] = table_R2;		n_args[80] = 2;		n_out[80] = 1;
	ops[81] = table_R2D;		n_args[81] = 1;		n_out[81] = 1;
	ops[82] = table_RAND;		n_args[82] = 2;		n_out[82] = 1;
	ops[83] = table_RINT;		n_args[83] = 1;		n_out[83] = 1;
	ops[84] = table_ROOTS;		n_args[84] = 2;		n_out[84] = 1;
	ops[85] = table_ROTT;		n_args[85] = 2;		n_out[85] = 1;
	ops[86] = table_SIGN;		n_args[86] = 1;		n_out[86] = 1;
	ops[87] = table_SIN;		n_args[87] = 1;		n_out[87] = 1;
	ops[88] = table_SINC;		n_args[88] = 1;		n_out[88] = 1;
	ops[89] = table_SIND;		n_args[89] = 1;		n_out[89] = 1;
	ops[90] = table_SINH;		n_args[90] = 1;		n_out[90] = 1;
	ops[91] = table_SQRT;		n_args[91] = 1;		n_out[91] = 1;
	ops[92] = table_STD;		n_args[92] = 1;		n_out[92] = 1;
	ops[93] = table_STEP;		n_args[93] = 1;		n_out[93] = 1;
	ops[94] = table_STEPT;		n_args[94] = 1;		n_out[94] = 1;
	ops[95] = table_SUB;		n_args[95] = 2;		n_out[95] = 1;
	ops[96] = table_SUM;		n_args[96] = 1;		n_out[96] = 1;
	ops[97] = table_TAN;		n_args[97] = 1;		n_out[97] = 1;
	ops[98] = table_TAND;		n_args[98] = 1;		n_out[98] = 1;
	ops[99] = table_TANH;		n_args[99] = 1;		n_out[99] = 1;
	ops[100] = table_TN;		n_args[100] = 2;		n_out[100] = 1;
	ops[101] = table_TCRIT;		n_args[101] = 2;		n_out[101] = 1;
	ops[102] = table_TDIST;		n_args[102] = 2;		n_out[102] = 1;
	ops[103] = table_UPPER;		n_args[103] = 1;		n_out[103] = 1;
	ops[104] = table_XOR;		n_args[104] = 2;		n_out[104] = 1;
	ops[105] = table_Y0;		n_args[105] = 1;		n_out[105] = 1;
	ops[106] = table_Y1;		n_args[106] = 1;		n_out[106] = 1;
	ops[107] = table_YN;		n_args[107] = 2;		n_out[107] = 1;
	ops[108] = table_ZCRIT;		n_args[108] = 1;		n_out[108] = 1;
	ops[109] = table_ZDIST;		n_args[109] = 1;		n_out[109] = 1;
}
