#include "stdafx.h"
#include "WinCanvas.h"
#include "ChildFrm.h"
#include "ContextFree.h"
#include "WinSystem.h"
#include <gdiplus.h>
#include "CGdiPlusBitmap.h"

using namespace Gdiplus;

bool WinCanvas::GdiPlusInitialized = false;
Bitmap* WinCanvas::AppBitmap = 0;
ColorPalette* WinCanvas::GrayPalette = 0;

extern CContextFreeApp theApp;

void WinCanvas::start(bool clear, const agg::rgba& bk, int actualWidth, int actualHeight)
{
    mFrame->mOwnCanvas.Lock();

    mChanged = true;    // tell window container to update backing store
    
    aggCanvas::start(clear, bk, actualWidth, actualHeight);
}

void WinCanvas::end()
{
    if (!mFrame->mOwnCanvas.Unlock()) DoError();

    // This message tells the RenderView to update the window and the status icon
    mFrame->PostMessage(WM_USER_STATUS_UPDATE, NULL, NULL);

    aggCanvas::end();
}

void WinCanvas::DoError() 
{
#ifdef _DEBUG
    LPVOID lpMsgBuf;
    FormatMessage( 
        FORMAT_MESSAGE_ALLOCATE_BUFFER | 
        FORMAT_MESSAGE_FROM_SYSTEM | 
        FORMAT_MESSAGE_IGNORE_INSERTS,
        NULL,
        GetLastError(),
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
        (LPTSTR) &lpMsgBuf,
        0,
        NULL 
        );
    // Process any inserts in lpMsgBuf.
    // ...
    // Display the string. Go through WinSystem because we don't know what thread
    // we are in and mFrame->OnMessageUpdate can only be called by the UI thread.
    mFrame->mSystem.message("%s", (char*)lpMsgBuf);
    // Free the buffer.
    LocalFree( lpMsgBuf );
#endif // _DEBUG
}

WinCanvas::WinCanvas(CChildFrame* f, aggCanvas::PixelFormat pixfmt, 
                     int width, int height, agg::rgba bkgrnd)
:   aggCanvas(pixfmt),
    mPixelFormat(pixfmt),
    mChanged(true),
    mDone(false),
    mBackground(),
    mFrame(f)
{
    // Do global initialization of GDI+ in a lazy fashion. Doing this in
    // CWinApp::InitInstance caused failures
    WinCanvas::StartUp();

    mWidth = width;
    mHeight = height;
    mStride = mWidth * aggCanvas::BytesPerPixel[mPixelFormat];

    // pad row stride to DWORD boundary
    mStride += ((-mStride) & 3);

    mBM = (char*)(new DWORD[mStride * mHeight / 4]);
    aggCanvas::attach((void*)mBM, mWidth, mHeight, mStride);

    // clear the bitmap
    aggCanvas::start(true, bkgrnd, width, height);
    aggCanvas::end();

    // provide the opaque background color for the view
    agg::rgba8 back8 = bkgrnd;
    mBackground = Color(back8.a, back8.r, back8.g, back8.b);
    mBackgroundOpaque = Color(255, back8.r, back8.g, back8.b);
}

Bitmap* WinCanvas::MakeBitmap(bool cropped)
{
    if (!GrayPalette) {
        GrayPalette = (ColorPalette*)malloc(sizeof(ColorPalette) + 256*sizeof(ARGB));
        GrayPalette->Count = 256;
        GrayPalette->Flags = PaletteFlagsGrayScale;
        for (int i = 0; i < 256; i++)
            GrayPalette->Entries[i] = Color::MakeARGB(255, (BYTE)i, (BYTE)i, (BYTE)i);
    }

    Bitmap* bm;
    BYTE* data = (BYTE*)mBM;
    int width = mWidth;
    int height = mHeight;
    if (cropped) {
        width = cropWidth();
        height = cropHeight();
        data += mStride * cropY() + 
                aggCanvas::BytesPerPixel[mPixelFormat] * cropX();
    }

    switch (mPixelFormat) {
        case aggCanvas::Gray8_Blend:
            bm = new Bitmap(width, height, mStride, PixelFormat8bppIndexed, data);
            bm->SetPalette(GrayPalette);
            break;
        case aggCanvas::RGB8_Blend:
            bm = new Bitmap(width, height, mStride, PixelFormat24bppRGB, data);
            break;
        case aggCanvas::RGBA8_Blend:
            bm = new Bitmap(width, height, mStride, PixelFormat32bppPARGB, data);
            break;
        default:
            bm = 0;
            break;
    }

    return bm;
}

WinCanvas::~WinCanvas()
{
    delete[] mBM;
}

ULONG_PTR WinCanvas::GdiPToken;
GdiplusStartupInput WinCanvas::GdiPStartInput;
GdiplusStartupOutput WinCanvas::GdiPStartOutput;

void WinCanvas::StartUp()
{
    if (!WinCanvas::GdiPlusInitialized) {
        GdiplusStartup(&GdiPToken, &GdiPStartInput, &GdiPStartOutput);
        CGdiPlusBitmapResource* bm = new CGdiPlusBitmapResource();
        bm->Load((LPCTSTR)IDR_APP_PNG, RT_RCDATA);
        WinCanvas::AppBitmap = bm->m_pBitmap;
        WinCanvas::GdiPlusInitialized = true;
    }
}

void WinCanvas::Shutdown()
{
    GdiplusShutdown(GdiPToken);
    if (GrayPalette)
        free((void*)GrayPalette);
}