// HueSatCtrl.cpp : implementation file
//

#include "stdafx.h"
#include "HueSatCtrl.h"
#include <math.h>
#include "resource.h"

using namespace Gdiplus;

#define RAD2DEG(x)  ((180.0 * (x))/ 3.14159265358979323846)
#define DEG2RAD(x)	(((x) * 3.14159265358979323846)/180.0)


static double Distance(const Point& pt1, const Point& pt2);
static double AngleFromPoint(const Point& pt, const Point& center);
static Gdiplus::Point PtFromAngle(double angle, double sat, const Point& center);
static Color ToColor(agg::rgba c);

// CHueSatCtrl

IMPLEMENT_DYNAMIC(CHueSatCtrl, CWnd)

BOOL CHueSatCtrl::RegisterWindowClass()
{
    WNDCLASS  wndcls;
    HINSTANCE hInst = AfxGetInstanceHandle();

    // Already registered?
    if(!(::GetClassInfo(hInst, C_CHUESATCTRL_CLASSNAME, &wndcls))) { 
        wndcls.style            = CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW;
        wndcls.lpfnWndProc      = ::DefWindowProc;
        wndcls.cbClsExtra       = wndcls.cbWndExtra = 0;
        wndcls.hInstance        = hInst;
        wndcls.hIcon            = NULL;
        wndcls.hCursor          = AfxGetApp()->LoadCursor(IDC_ARROW);
        wndcls.hbrBackground    = (HBRUSH) (COLOR_3DFACE + 1);
        wndcls.lpszMenuName     = NULL;
        wndcls.lpszClassName    = C_CHUESATCTRL_CLASSNAME;

        if(!AfxRegisterClass(&wndcls)) {
            AfxThrowResourceException();
            return FALSE;
        }
    }

    return TRUE;
}
CHueSatCtrl::CHueSatCtrl() 
:   mInitialized(false),
    mHue(0.0),
    mSat(0.0),
    mDrawMarker(true),
    mIgnoreMouseMove(false),
    mBlackPen(Color(0,0,0)),
    mHSbitmap(0),
    mFocusPen(0),
    mNoFocusPen(0)
{
    RegisterWindowClass();
}

CHueSatCtrl::~CHueSatCtrl()
{
    delete mHSbitmap;
    delete mFocusPen;
    delete mNoFocusPen;
}


BEGIN_MESSAGE_MAP(CHueSatCtrl, CWnd)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_PAINT()
    ON_WM_SETFOCUS()
    ON_WM_KILLFOCUS()
    ON_WM_KEYDOWN()
    ON_WM_GETDLGCODE()
END_MESSAGE_MAP()

void CHueSatCtrl::CalcRect()
{
    Point pt;
	pt = PtFromAngle(mHue, mSat * mRadius, mHScenter);
	mCurrentHSrect = Rect(pt.X - RectSize, pt.Y - RectSize, 
                         2 * RectSize, 2 * RectSize);	
}

void CHueSatCtrl::SetHueSat(double hue, double sat)
{
    if (GetCapture() == this ||
        (hue == mHue && sat == mSat)) return;

    if (sat < 0.0) sat = 0.0;
    if (sat > 1.0) sat = 1.0;
    double dummy;
    hue = hue / 360.0;
    hue = modf(hue, &dummy);
    hue *= 360.0;
    mHue = hue;
    mSat = sat;

    RedrawPoint(TRUE);
    CalcRect();
    RedrawPoint(FALSE);
}

void CHueSatCtrl::GetHueSat(double& hue, double& sat)
{
    hue = mHue;
    sat = mSat;
}

void CHueSatCtrl::RedrawPoint(BOOL erase)
{
    CRect cr(mCurrentHSrect.GetLeft(), mCurrentHSrect.GetTop(), 
        mCurrentHSrect.GetRight(), mCurrentHSrect.GetBottom());
    cr.InflateRect(1, 1);
    InvalidateRect(&cr, erase);
}


// CHueSatCtrl message handlers

void CHueSatCtrl::Init()
{
    CRect r;
    GetClientRect(&r);

    int size = r.Height() < r.Width() ? r.Height() : r.Width();
    if ((size & 1) == 0) size--;
    mFocusRect.X = r.left;
    mFocusRect.Y = r.top;
    mFocusRect.Width = size;
    mFocusRect.Height = size;
    mHSrect = mFocusRect;
    mFocusRect.Width--;
    mFocusRect.Height--;
    mHSrect.Inflate(-1, -1);
    mHSrect.GetLocation(&mHScenter);
    mHScenter = mHScenter + Point(mHSrect.Width / 2, mHSrect.Height / 2);

    CalcRect();

    mHSbitmap = new Bitmap(mHSrect.Width, mHSrect.Height, PixelFormat24bppRGB);
    DWORD backcolor = ::GetSysColor(COLOR_3DFACE);
    Color back((BYTE)(backcolor & 255), 
        (BYTE)((backcolor >> 8) & 255), 
        (BYTE)((backcolor >> 16) & 255));
    Color c;
    Point center(mHSrect.Width / 2, mHSrect.Height / 2);
    mRadius = (double)(center.X - RectSize);
    for (int y = 0; y < mHSrect.Height; y++)
        for (int x = 0; x < mHSrect.Width; x++) {
            Point p(x, y);
            double sat = Distance(p, center) / mRadius;
            double hue = AngleFromPoint(p, center);
            if (sat > 1.01) {
                mHSbitmap->SetPixel(x, y, back);
            } else {
                if (sat > 1.0) sat = 1.0;
                HSBColor hsb(hue, sat, 1.0, 1.0);
                agg::rgba rgbcolor;
                hsb.getRGBA(rgbcolor);
                c = ToColor(rgbcolor);
                mHSbitmap->SetPixel(x, y, c);
            }
        }

    mFocusPen = new Pen(Color());
    mFocusPen->SetDashStyle(DashStyleDot);
    mNoFocusPen = new Pen(back);

    mMarkerCursor = AfxGetApp()->LoadCursor(IDC_MARKER);

    mNotify.code = HS_CHANGE;
    mNotify.hwndFrom = GetSafeHwnd();
    mNotify.idFrom = GetDlgCtrlID();
    mInitialized = true;
}

void CHueSatCtrl::OnPaint()
{
    if (GetUpdateRect(NULL, TRUE)) {
        if (!mInitialized) Init();
        CPaintDC dc(this);
        Graphics g(dc);
        g.DrawImage(mHSbitmap, mHSrect);
        if (mDrawMarker)
            g.DrawRectangle(&mBlackPen, mCurrentHSrect);
        bool hasFocus = GetFocus() == this;
        g.DrawRectangle(hasFocus ? mFocusPen : mNoFocusPen, mFocusRect);
    }
}

void CHueSatCtrl::OnSetFocus(CWnd*)
{
    DrawFocus(mFocusPen);
}

void CHueSatCtrl::OnKillFocus(CWnd*)
{
    DrawFocus(mNoFocusPen);
}

void CHueSatCtrl::DrawFocus(Gdiplus::Pen* p)
{
    if (!mInitialized) return;
    Graphics g(GetSafeHwnd());
    g.DrawRectangle(p, mFocusRect);
}

void CHueSatCtrl::OnLButtonDown(UINT nFlags, CPoint point)
{
    Point pt(point.x, point.y);
	
	if (Distance(pt, mHScenter) > (mRadius + 5)) return;

    SetFocus();

    mDrawMarker = false;
    RedrawPoint(TRUE);

    mPrevCursor = ::SetCursor(mMarkerCursor);

	SetCapture();
	TrackPoint(point);

	CWnd::OnLButtonDown(nFlags, point);
}

void CHueSatCtrl::OnLButtonUp(UINT nFlags, CPoint point)
{
	if (GetCapture() == this) {
		ReleaseCapture();
        CalcRect();
        mDrawMarker = true;
        RedrawPoint(FALSE);
        ::SetCursor(mPrevCursor);
    }
	CWnd::OnLButtonUp(nFlags, point);
}

void CHueSatCtrl::OnMouseMove(UINT nFlags, CPoint point)
{
	if (GetCapture() == this) {
		TrackPoint(point);
	}
	CWnd::OnMouseMove(nFlags, point);
}

void CHueSatCtrl::TrackPoint(CPoint point)
{
    if (mIgnoreMouseMove) {
        mIgnoreMouseMove = false;
        return;
    }

    Point pt(point.x, point.y);
	
	mHue = AngleFromPoint(pt, mHScenter);
	if (mHue < 0.0) {
		mHue += 360.0;
	}
	mSat = Distance(pt, mHScenter) / mRadius;
    if (mSat > 1.0) {
        mIgnoreMouseMove = true;
        mSat = 1.0;
        pt = PtFromAngle(mHue, 1.01 * mRadius, mHScenter);
        CPoint p(pt.X, pt.Y);
        ClientToScreen(&p);
        ::SetCursorPos(p.x, p.y);
    }
    GetParent()->SendNotifyMessage(WM_NOTIFY, mNotify.idFrom, (LPARAM)(&mNotify));
}

void CHueSatCtrl::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags)
{
    double hue = mHue;
    double sat = mSat;
    if (nChar == VK_UP) sat += (double)nRepCnt / 64.0;
    if (nChar == VK_DOWN) sat -= (double)nRepCnt / 64.0;
    if (nChar == VK_LEFT) hue += (double)nRepCnt;
    if (nChar == VK_RIGHT) hue -= (double)nRepCnt;
    if (sat != mSat || hue != mHue) {
        SetHueSat(hue, sat);
        GetParent()->SendNotifyMessage(WM_NOTIFY, mNotify.idFrom, (LPARAM)(&mNotify));
    }
}

UINT CHueSatCtrl::OnGetDlgCode()
{
    return DLGC_WANTARROWS;
}

static double Distance(const Point& pt1, const Point& pt2)
{
	int y = (pt1.Y - pt2.Y);
	y *= y;

	int x = (pt1.X - pt2.X);
	x *= x;
	
	return sqrt((double)x + (double)y);
}

static double AngleFromPoint(const Point& pt, const Point& center)
{
	double y = center.Y - pt.Y;
	double x = pt.X - center.X;
	if(x == 0 && y == 0) {
		return 0;
    } else {
		return RAD2DEG(atan2(y, x));
	}
}

static Point PtFromAngle(double angle, double sat, const Point& center)
{
	angle = DEG2RAD(angle);

    double x = sat * cos(angle);
	double y = sat * sin(angle);

    Point pt((int)x, (int)y);
	pt.Y *= -1;
	pt = pt + center;
	return pt;
}

Color ToColor(agg::rgba c)
{
    agg::rgba8 c8(c);
    return Color(c8.a, c8.r, c8.g, c8.b);
}

