// ColorPickerDlg.cpp : implementation file
//

#include "stdafx.h"
#include "ColorPickerDlg.h"
#include "HSBColor.h"
#include <gdiplus.h>
#include <agg_color_rgba.h>
#include "resource.h"
#include <math.h>
#include "HueSatCtrl.h"
#include "ColorSliderCtrl.h"
#include "ColorDropperCtrl.h"
#include "ColorWell.h"

using namespace Gdiplus;

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

Color ToColor(const agg::rgba& c);

/////////////////////////////////////////////////////////////////////////////
// CColorPickerDlg dialog


CColorPickerDlg::CColorPickerDlg(CWnd* pParent /*=NULL*/)
	: CDialog(CColorPickerDlg::IDD, pParent),
      mSettingEditValue(false),
      mSelectedColor(FinishSelected)
{
	//{{AFX_DATA_INIT(CColorPickerDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT

    mFinishColor = agg::rgba(0.5, 0.5, 0.5, 1.0);
    mStartColor = mFinishColor;
    mFinishHSBcolor = HSBColor(mFinishColor);
    mStartHSBcolor = HSBColor(mStartColor);    
    
    mSelectedHSBcolor = &mFinishHSBcolor;
    mSelectedRGBAcolor = &mFinishColor;
    
	mInitOver = false;
}

CColorPickerDlg::~CColorPickerDlg()
{
}

COLORREF CColorPickerDlg::GetColor() {
    return ToColor(mFinishColor).ToCOLORREF();
};


void CColorPickerDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CColorPickerDlg)
		// NOTE: the ClassWizard will add DDX and DDV calls here
    DDX_Control(pDX, IDC_HUESAT, mHueSat);
    DDX_Control(pDX, IDC_BRIGHT, mBright);
    DDX_Control(pDX, IDC_REDS, mRed);
    DDX_Control(pDX, IDC_GREENS, mGreen);
    DDX_Control(pDX, IDC_BLUES, mBlue);
    DDX_Control(pDX, IDC_DROPPER, mDropper);
    DDX_Control(pDX, IDC_START_COLOR, mStartWell);
    DDX_Control(pDX, IDC_FINISH_COLOR, mFinishWell);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CColorPickerDlg, CDialog)
	//{{AFX_MSG_MAP(CColorPickerDlg)
    ON_NOTIFY(CHueSatCtrl::HS_CHANGE, IDC_HUESAT, OnHSBChange)
    ON_NOTIFY(CColorSliderCtrl::SLIDER_CHANGE, IDC_BRIGHT, OnHSBChange)
    ON_NOTIFY(CColorSliderCtrl::SLIDER_CHANGE, IDC_REDS, OnRGBChange)
    ON_NOTIFY(CColorSliderCtrl::SLIDER_CHANGE, IDC_GREENS, OnRGBChange)
    ON_NOTIFY(CColorSliderCtrl::SLIDER_CHANGE, IDC_BLUES, OnRGBChange)
    ON_NOTIFY(CColorDropperCtrl::DROPPER_CHANGE, IDC_DROPPER, OnDropper)
	ON_EN_CHANGE(IDC_EDIT_HUE, OnChangeEditFinish)
	ON_EN_CHANGE(IDC_EDIT_SAT, OnChangeEditFinish)
	ON_EN_CHANGE(IDC_EDIT_VAL, OnChangeEditFinish)
	ON_EN_CHANGE(IDC_EDIT_HUE_OLD, OnChangeEditStart)
	ON_EN_CHANGE(IDC_EDIT_SAT_OLD, OnChangeEditStart)
	ON_EN_CHANGE(IDC_EDIT_VAL_OLD, OnChangeEditStart)
	ON_EN_CHANGE(IDC_EDIT_HUE_DELTA, OnChangeEditDelta)
	ON_EN_CHANGE(IDC_EDIT_SAT_DELTA, OnChangeEditDelta)
	ON_EN_CHANGE(IDC_EDIT_VAL_DELTA, OnChangeEditDelta)
	ON_EN_CHANGE(IDC_RED, OnRGBEditChange)
	ON_EN_CHANGE(IDC_GREEN, OnRGBEditChange)
	ON_EN_CHANGE(IDC_BLUE, OnRGBEditChange)
	ON_COMMAND(IDC_START_COLOR, OnSelectStart)
	ON_COMMAND(IDC_FINISH_COLOR, OnSelectFinish)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CColorPickerDlg message handlers

void CColorPickerDlg::OnHSBChange(NMHDR* , LRESULT* )
{
    HSBColor hsb(0, 0, 0, 1);
    agg::rgba rgb;

    mHueSat.GetHueSat(hsb.h, hsb.s);
    hsb.b = (double)mBright.GetPos() / 255.0;

	hsb.getRGBA(rgb);
    DispatchColor(hsb, rgb, 0);
}

void CColorPickerDlg::OnDropper(NMHDR* , LRESULT* )
{
    Color c;
    mDropper.GetColor(c);
    mSelectedRGBAcolor->r = (float)c.GetR() / 255.0F;
    mSelectedRGBAcolor->g = (float)c.GetG() / 255.0F;
    mSelectedRGBAcolor->b = (float)c.GetB() / 255.0F;
    *mSelectedHSBcolor = HSBColor(*mSelectedRGBAcolor);
    DispatchColor(*mSelectedHSBcolor, *mSelectedRGBAcolor, 0);
}


BOOL CColorPickerDlg::OnInitDialog() 
{
	CDialog::OnInitDialog();    

	SetEditVals();
    CheckDlgButton(mSelectedColor == FinishSelected ? 
        IDC_FINISH_COLOR : IDC_START_COLOR, TRUE );
    mRed.SetColor(Color(255,0,0), Color());
    mGreen.SetColor(Color(0,255,0), Color());
    mBlue.SetColor(Color(0,0,255), Color());
    mStartWell.SetSelected(mSelectedColor == StartSelected);
    mFinishWell.SetSelected(mSelectedColor == FinishSelected);
    DispatchColor(*mSelectedHSBcolor, *mSelectedRGBAcolor, exc_hsb_edit);

	mInitOver = true;
	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
}

void CColorPickerDlg::DispatchColor(HSBColor& hsb, agg::rgba& rgb, int exclude)
{
    agg::rgba8 rgb8(rgb);

    *mSelectedHSBcolor = hsb;
    *mSelectedRGBAcolor = rgb;
    mFinishWell.SetColor(ToColor(mFinishColor));
    mStartWell.SetColor(ToColor(mStartColor));

    if ((exclude & exc_huesat) == 0) {
        mHueSat.SetHueSat(hsb.h, hsb.s);
        HSBColor brightHSB(hsb.h, hsb.s, 1.0, 1.0);
        agg::rgba brightRGB;
        brightHSB.getRGBA(brightRGB);

        mBright.SetColor(ToColor(brightRGB), Color());
    }
    if ((exclude & exc_bright) == 0) {
        mBright.SetPos((int)(hsb.b * 255.0));
    }
    if ((exclude & exc_rgb_slider) == 0) {
        mRed.SetPos(rgb8.r);
        mGreen.SetPos(rgb8.g);
        mBlue.SetPos(rgb8.b);
    }
    if ((exclude & exc_rgb_edit) == 0) {
        mSettingEditValue = true;
        SetDlgItemInt(IDC_RED, rgb8.r, FALSE);
        SetDlgItemInt(IDC_GREEN, rgb8.g, FALSE);
        SetDlgItemInt(IDC_BLUE, rgb8.b, FALSE);
        mSettingEditValue = false;
    }
    if ((exclude & exc_hsb_edit) == 0) {
        if (mSelectedColor == FinishSelected) {
            SetEditVals(mFinishHSBcolor, IDC_EDIT_HUE);
        } else {
            SetEditVals(mStartHSBcolor, IDC_EDIT_HUE_OLD);
        }
        SetDeltaHSB();
    }
}



void CColorPickerDlg::SetEditVals(HSBColor& hsb, UINT HSBctrl)
{
    // tell EditChange routines to ignore these changes since they are caused by
    // code and not by the user editing the HSB value edit controls
    mSettingEditValue = true;

    TCHAR buf[64];
    _stprintf(buf, _T("%.4f"), hsb.h);
	SetDlgItemText(HSBctrl + 0, buf);
    _stprintf(buf, _T("%.4f"), hsb.s);
	SetDlgItemText(HSBctrl + 1, buf);
    _stprintf(buf, _T("%.4f"), hsb.b);
	SetDlgItemText(HSBctrl + 2, buf);
    _stprintf(buf, _T("hue %.4f sat %.4f b %.4f"), hsb.h, hsb.s, hsb.b);
	SetDlgItemText(HSBctrl + 3, buf);

    // Done changing the edit controls, any subsequent changes are due to the user
    // directly modifying the control (i.e. typing or pasting something)
    mSettingEditValue = false;
}

void CColorPickerDlg::SetEditVals()
{
    SetEditVals(mFinishHSBcolor, IDC_EDIT_HUE);
    SetEditVals(mStartHSBcolor, IDC_EDIT_HUE_OLD);

    SetDeltaHSB();
}

void CColorPickerDlg::SetDeltaHSB()
{
    mDeltaHSBcolor = mFinishHSBcolor.getAdjustmentFrom(mStartHSBcolor);

    SetEditVals(mDeltaHSBcolor, IDC_EDIT_HUE_DELTA);
}

void CColorPickerDlg::SetFinishHSB()
{
    mFinishHSBcolor = mStartHSBcolor;
    mFinishHSBcolor.adjustWith(mDeltaHSBcolor);

    SetEditVals(mFinishHSBcolor, IDC_EDIT_HUE);
}

static bool isNumeric(TCHAR* buf, bool canBeNeg)
{
    bool dotFound = false;
    TCHAR cbuf[2];
    cbuf[1] = _T('\0');

    // scan for anything other than white space or number characters
    if (_tcsspn(buf, _T(" \t-1234567890.")) != _tcsclen(buf)) return false;

    // skip leading spaces & tabs
    buf += _tcsspn(buf, _T(" \t"));

    // skip '-' if canBeNeg
    if (*buf == _T('-') && canBeNeg) buf++;

    // scan for digits, one decimal point and stop when done or whitespace
    while (*buf) {
        cbuf[0] = *buf++;
        if (cbuf[0] == _T('.')) {
            if (dotFound) return false;
            dotFound = true;
            continue;
        }
        if (_tcscspn(cbuf, _T(" \t")) == 0 && 
            _tcsspn(buf, _T(" \t")) == _tcsclen(buf)) return true;
        if (_tcscspn(cbuf, _T("0123456789")) == 1) return false;
    }

    return true;
}

void CColorPickerDlg::changedHSBedit(UINT HSBctrl,  
    HSBColor& hsb, agg::rgba& c, bool isDelta /*= false*/) 
{
    bool fixed = false;
    double h, s, b;
    TCHAR buf[64];
    LPTSTR eptr;

    if (GetDlgItemText(HSBctrl, buf, 64) >= 64 || !isNumeric(buf, isDelta)) {
        fixed = true;
        h = 0.0;
    } else {
    	h = _tcstod(buf, &eptr);
    }

    if (GetDlgItemText(HSBctrl + 1, buf, 64) >= 64 || !isNumeric(buf, isDelta)) {
        fixed = true;
        s = 0.0;
    } else {
    	s = _tcstod(buf, &eptr);
    }

    if (GetDlgItemText(HSBctrl + 2, buf, 64) >= 64 || !isNumeric(buf, isDelta)) {
        fixed = true;
        b = 0.0;
    } else {
    	b = _tcstod(buf, &eptr);
    }

    double min = isDelta ? -360.0 : 0.0;
    double dummy;
    if (h < min || h > 360.0) {
        fixed = true;
        h = h / 360.0;
        h = modf(h, &dummy);
        h *= 360.0;
    }
    min = isDelta ? -1.0 : 0.0;
    if (s < min || s > 1.0) {
        fixed = true;
        s = (s < min) ? min : 1.0;
    }
    if (b < min || b > 1.0) {
        fixed = true;
        b = (b < min) ? min : 1.0;
    }

    if (fixed) {
        HSBColor hsb2(h, s, b, 1.0);
        SetEditVals(hsb2, HSBctrl);
        MessageBeep(MB_OK);
    }

    if (mInitOver && (fabs(h - hsb.h) > 0.00001 ||
                      fabs(s - hsb.s) > 0.00001 ||
                      fabs(b - hsb.b) > 0.00001)) {
		hsb.h = h; hsb.s = s; hsb.b = b;
        hsb.getRGBA(c);

        if (!fixed) {
            _stprintf(buf, _T("hue %.4f sat %.4f b %.4f"), hsb.h, hsb.s, hsb.b);
	        SetDlgItemText(HSBctrl + 3, buf);
        }

        if (isDelta) {
            SetFinishHSB();
    		mFinishHSBcolor.getRGBA(mFinishColor);
            if (mSelectedColor == FinishSelected)
                DispatchColor(mFinishHSBcolor, mFinishColor, exc_hsb_edit);
        } else {
            SetDeltaHSB();
            if (&hsb == mSelectedHSBcolor)
                DispatchColor(hsb, c, exc_hsb_edit);
        }
	}
}

// Process user manipulation of the 9 HSB edit controls to make sure that the 
// delta and finished colors are matched and the HSB picker matches the controls.
// If the change to the edit control was caused by CColorPickerDlg itself then
// do nothing. Whatever code changes these edit controls is responsible for
// keeping CColorPickerDlg consistent.

void CColorPickerDlg::OnChangeEditFinish() 
{
    if (!mSettingEditValue) 
        changedHSBedit(IDC_EDIT_HUE, mFinishHSBcolor, mFinishColor);
}

void CColorPickerDlg::OnChangeEditStart() 
{
    if (!mSettingEditValue) 
        changedHSBedit(IDC_EDIT_HUE_OLD, mStartHSBcolor, mStartColor);
}

void CColorPickerDlg::OnChangeEditDelta() 
{
    if (!mSettingEditValue) 
        changedHSBedit(IDC_EDIT_HUE_DELTA, mDeltaHSBcolor, mFinishColor, true);
}


void CColorPickerDlg::OnRGBEditChange() 
{
    if (!mSettingEditValue) {
        int r, g, b;
        BOOL err;
        bool fixed = false;
        r = GetDlgItemInt(IDC_RED, &err, FALSE);
        fixed = fixed || !err;
        g = GetDlgItemInt(IDC_GREEN, &err, FALSE);
        fixed = fixed || !err;
        b = GetDlgItemInt(IDC_BLUE, &err, FALSE);
        fixed = fixed || !err;

        if (r < 0 || r > 255) {
            r = (r < 0) ? 0 : 255;
            fixed = true;
        }

        if (g < 0 || g > 255) {
            g = (g < 0) ? 0 : 255;
            fixed = true;
        }

        if (b < 0 || b > 255) {
            b = (b < 0) ? 0 : 255;
            fixed = true;
        }

        UpdateRGB(r, g, b, fixed ? 0 : exc_rgb_edit);
    }
}

void CColorPickerDlg::OnRGBChange(NMHDR* , LRESULT* ) 
{
    int r, g, b;
    r = mRed.GetPos();
    g = mGreen.GetPos();
    b = mBlue.GetPos();
    UpdateRGB(r, g, b, exc_rgb_slider);
}

void CColorPickerDlg::UpdateRGB(int r, int g, int b, int exclude)
{
    agg::rgba rgb((double)r / 255.0, (double)g / 255.0, (double)b / 255.0);
    HSBColor hsb(rgb);
    DispatchColor(hsb, rgb, exclude);
}

void CColorPickerDlg::OnSelectStart() 
{
    if (mSelectedColor != StartSelected) {
        CheckDlgButton(IDC_START_COLOR, TRUE );
        mStartWell.SetSelected(true);
        mFinishWell.SetSelected(false);
        mSelectedColor = StartSelected;
        mSelectedHSBcolor = &mStartHSBcolor;
        mSelectedRGBAcolor = &mStartColor;
        DispatchColor(*mSelectedHSBcolor, *mSelectedRGBAcolor, exc_hsb_edit);
    }
}

void CColorPickerDlg::OnSelectFinish() 
{
    if (mSelectedColor != FinishSelected) {
        CheckDlgButton(IDC_FINISH_COLOR, TRUE );
        mStartWell.SetSelected(false);
        mFinishWell.SetSelected(true);
        mSelectedColor = FinishSelected;
        mSelectedHSBcolor = &mFinishHSBcolor;
        mSelectedRGBAcolor = &mFinishColor;
        DispatchColor(*mSelectedHSBcolor, *mSelectedRGBAcolor, exc_hsb_edit);
    }
}

Color ToColor(const agg::rgba& c)
{
    agg::rgba8 c8(c);
    return Color(c8.a, c8.r, c8.g, c8.b);
}

