#include "pngCanvas.h"
#include "png.h"
#include <stdlib.h>
#include <math.h>

using namespace std;

pngCanvas::pngCanvas(const char* outfilename, int width, int height, 
                     aggCanvas::PixelFormat pixfmt, bool crop, int frameCount)
    : aggCanvas(pixfmt), mData(0), mOutputFileName(outfilename),  
      mFrameCount(frameCount), mCurrentFrame(0), 
      mPixelFormat(pixfmt), mCrop(crop)
{
	mWidth = width;
	mHeight = height;
    cout << width << "w x " << height << "h pixel image." << endl;
    cout << "Generating..." << endl;
}

pngCanvas::~pngCanvas()
{
    delete[] mData;
}

void pngCanvas::start(bool clear, const agg::rgba& bk, int width, int height)
{
	if (!mFrameCount)
        cout << endl << "Rendering..." << endl;

    if (mCrop) {
        mWidth = width;
        mHeight = height;
    }

    mStride = mWidth * aggCanvas::BytesPerPixel[mPixelFormat];
    mData = new unsigned char[mStride * mHeight];
    attach(mData, mWidth, mHeight, mStride);

    aggCanvas::start(clear, bk, width, height);
}

void pngCanvas::end()
{
	aggCanvas::end();
	
	if (mFrameCount) {
		char frameNum[64];
		int numlength = (int)log10((double)mFrameCount) + 1;
		sprintf(frameNum, "_%0*d", numlength, mCurrentFrame);
		string frameName = mOutputFileName;
		int framePoint = frameName.find_last_of("/.");
		if (framePoint == string::npos || frameName[framePoint] == '/')
		    framePoint = frameName.length();
		frameName.insert(framePoint, frameNum);
		output(frameName.c_str(), mCurrentFrame++);
	} else {
		output(mOutputFileName);
	}
}

namespace {
    void
    pngWriteError(png_structp png_ptr, png_const_charp message)
    {
        cerr << message << endl;
        throw false;
    }

    void
    pngWriteWarning(png_structp png_ptr, png_const_charp message)
    {
        cerr << message << endl;
    }
}


void pngCanvas::output(const char* outfilename, int frame)
{
	if (frame == -1 ) {
        cout << endl << "Writing "
             << mWidth << "w x " << mHeight << "h pixel image..." << endl;
	} 

    FILE *out = 0;
    png_structp png_ptr = 0;
    png_infop info_ptr = 0;

    try {
        png_ptr = png_create_write_struct(
            PNG_LIBPNG_VER_STRING,
            0, pngWriteError, pngWriteWarning);
        if (!png_ptr) throw "couldn't create png write struct";

        png_infop info_ptr = png_create_info_struct(png_ptr);
        if (!info_ptr) throw "couldn't create png info struct";

        out = fopen(outfilename, "wb");
        if (!out) {
            cerr << "Couldn't open " << outfilename << "\n";
            throw false;
        }

        png_init_io(png_ptr, out);
        
        int pngFormat;
        switch (mPixelFormat) {
            case aggCanvas::RGBA8_Blend:
                pngFormat = PNG_COLOR_TYPE_RGB_ALPHA;
                break;
            case aggCanvas::RGB8_Blend:
                pngFormat = PNG_COLOR_TYPE_RGB;
                break;
            case aggCanvas::Gray8_Blend:
                pngFormat = PNG_COLOR_TYPE_GRAY;
                break;
            default:
                throw "Unknown pixel format";
        }

        png_set_IHDR(png_ptr, info_ptr,
            mWidth, mHeight, 8, pngFormat,
            PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT,
            PNG_FILTER_TYPE_DEFAULT);

        png_text comments[1];
        memset(comments, 0, sizeof(comments));
        comments[0].compression = PNG_TEXT_COMPRESSION_NONE;
        comments[0].key = "Software";
        comments[0].text = "Context Free";
        comments[0].text_length = strlen(comments[0].text);
        png_set_text(png_ptr, info_ptr,
            comments, sizeof(comments)/sizeof(comments[0]));

        png_write_info(png_ptr, info_ptr);

        png_bytep rowPtr = mData;
        for (int r = 0; r < mHeight; ++r) {
            png_write_row(png_ptr, rowPtr);
            rowPtr += mStride;
        }

        png_write_end(png_ptr, 0);

        if (fclose(out) != 0) throw "File I/O error!?!?!";
        out = 0;
    }
    catch (const char* msg) {
        cerr << "***" << msg << endl;
    }
    catch (bool) { }

    if (out)        fclose(out), out = 0;
    if (png_ptr)    png_destroy_write_struct(&png_ptr, &info_ptr);
}


