#include "builder.h"

#include "agg_trans_affine.h"
#include "cfdgimpl.h"

#ifdef WIN32
#include "rand48.h"
#endif

unsigned Builder::ModSeedIndex;

Builder::~Builder()
{
    delete m_CFDG;
    delete m_input;
}


void
Builder::buildDefaultTransform()
{
    agg::trans_affine_translation tr(mXLocation, mYLocation);
    agg::trans_affine_scaling sc(mSizeX, mSizeY);
    agg::trans_affine_skewing sk(mSkewX * MY_PI / 180.0, mSkewY * MY_PI / 180.0);
    agg::trans_affine_rotation rot(mOrientation * MY_PI / 180.0);

    mCurrentModification.m_transform.reset();

    if (mReflected) {
        agg::trans_affine_reflection ref(mReflection * MY_PI / 180.0);
        mCurrentModification.m_transform *= ref;
    }

    mCurrentModification.m_transform *= sk;
    mCurrentModification.m_transform *= sc;
    mCurrentModification.m_transform *= rot;
    mCurrentModification.m_transform *= tr;

    mCurrentModification.m_Z = mZLocation;
    mCurrentModification.m_SizeZ = mSizeZ;
}

const char*
Builder::buildTileTransform()
{
    if (mSizeX > 0 && mSizeY > 0) {
        double o_x = 0.0;
        double o_y = 0.0;
        double u_x = 1.0;
        double u_y = 0.0;
        double v_x = 0.0;
        double v_y = 1.0;
        
        mCurrentModification.m_transform.transform(&o_x, &o_y);
        mCurrentModification.m_transform.transform(&u_x, &u_y);
        mCurrentModification.m_transform.transform(&v_x, &v_y);
        
        if (fabs(u_y - o_y) >= 0.0001 && fabs(v_x - o_x) >= 0.0001) 
            return "Tile must be aligned with the X or Y axis.";
        if ((u_x - o_x) < 0.0 || (v_y - o_y) < 0.0)
            return "Tile must be in the positive X/Y quadrant.";
        
        m_CFDG->setTiled(mCurrentModification.m_transform, u_x - o_x, v_y - o_y);
    }
    
    return 0;
}

const char*
Builder::buildSizeTransform()
{
    if (mCurrentModification.m_transform.shx != 0.0 || 
        mCurrentModification.m_transform.shy != 0.0)
        return "Only size transforms may be used in the size directive.";
    
    m_CFDG->setSized(mCurrentModification.m_transform);
    return 0;
}

void 
Builder::Orientation(double a)
{
    agg::trans_affine_rotation rot(a * MY_PI / 180.0);
    mCurrentModification.m_transform.premultiply(rot);

    mOrientation = a;
    addEntropy("Orientation");
}

void 
Builder::Reflection(double a)
{
    agg::trans_affine_reflection ref(a * MY_PI / 180.0);
    mCurrentModification.m_transform.premultiply(ref);
    
    mReflection = a;
    mReflected = true;
    addEntropy("Reflection");
}

void 
Builder::Size(double sx, double sy, double sz)
{
    agg::trans_affine_scaling sc(sx, sy);
    mCurrentModification.m_transform.premultiply(sc);

    mCurrentModification.m_SizeZ *= sz;

    mSizeX = sx; mSizeY = sy; mSizeZ = sz;
    addEntropy("Size");
}

void 
Builder::Skew(double sx, double sy)
{
    agg::trans_affine_skewing sk(sx * MY_PI / 180.0, sy * MY_PI / 180.0);
    mCurrentModification.m_transform.premultiply(sk);

    mSkewX = sx; mSkewY = sy;
    addEntropy("Skew");
}

void 
Builder::XLocation(double x)
{
    agg::trans_affine_translation tr(x, 0.0);
    mCurrentModification.m_transform.premultiply(tr);

    mXLocation = x;
    addEntropy("XLocation");
}

void 
Builder::YLocation(double y)
{
    agg::trans_affine_translation tr(0.0, y);
    mCurrentModification.m_transform.premultiply(tr);

    mYLocation = y;
    addEntropy("YLocation");
}

void 
Builder::ZLocation(double z)
{
    mCurrentModification.m_Z += mCurrentModification.m_SizeZ * z;

    mZLocation = z;
    addEntropy("ZLocation");
}

void 
Builder::clearModification()
{
    // Restore builder current shape info to initial state for next replacement
    mCurrentModification = Modification();
    mOrientation = mXLocation = mYLocation = mZLocation = 0.0;
    mReflected = false;
    mSizeX = mSizeY = mSizeZ = 1.0;
    mSkewX = mSkewY = 0.0;
    ModSeedIndex = 0;
}

void
Builder::addEntropy(const char* txt)
// Randomizes the rand48 seed using text from the cfdg file
{
    unsigned len = strlen(txt);
	for (unsigned i = 0; i < len; i++) {
		((char*)(mCurrentModification.mRand48Seed))[ModSeedIndex] ^= txt[i];
        nrand48(mCurrentModification.mRand48Seed);
        ModSeedIndex = (ModSeedIndex + 1) % 6;
    }
 }


