/*
 *  SVGCanvas.cpp
 *  Context Free
 *
 *  Created by John Horigan on Jan. 20, 2006.
 *  Copyright 2006 John Horigan. All rights reserved.
 *
 */

#include "SVGCanvas.h"
#include <assert.h>
#include <iomanip>
#include <math.h>
#include <iostream>
#include <sstream>
#include "primShape.h"

using namespace std;

void SVGCanvas::start(bool , const agg::rgba& , int width, int height)
{
    //assert(clear);

    if (mCropped) {
        mWidth = width;
        mHeight = height;
    }

    agg::trans_affine_translation off((mWidth - width) / 2.0, (mHeight - height) / 2.0);
    mOffset = off;

    mOutput << "<?xml version=\"1.0\" standalone=\"no\"?>" << endl;
    mOutput << "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\" " << endl;
    mOutput << "\"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">" << endl;
    mOutput << "<svg width=\"" << mWidth << "mm\" height=\"" << mHeight << "mm\"" << endl;
    mOutput << "     xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\">" << endl;
    mOutput << "  <g transform=\"matrix(3.779528 0 0 -3.779528 0 " 
            << mHeight * 96 / 25.4 << ")\">" << endl;

    if (mDescription) {
        mOutput << "    <desc>" << endl;
        mOutput.write(mDescription, mLength);
        mOutput << endl << "    </desc>" << endl;
    }
}

void SVGCanvas::end() {
    mOutput << "  </g>" << endl;
    mOutput << "</svg>" << endl;

    mOutput.close();
}

void SVGCanvas::complete(RGBA8 c, agg::trans_affine tr, int padding)
{
    static const char* pad16 = "                ";

    int rgb = c.r * 65536 + c.g * 256 + c.b;

    mOutput << (pad16 + (16 - padding));
    mOutput << "fill=\"#" << hex << setw(6) << setfill('0') << rgb << "\"";
    
    if (c.a < 255) mOutput <<  " fill-opacity=\"" << c.opacity() << "\"";
        
    mOutput << endl;
    mOutput << (pad16 + (16 - padding)) << setprecision(8) << setw(10);
    mOutput << "transform=\"matrix(" << tr.sx << " " << tr.shy << " " 
        << tr.shx << " " << tr.sy << " " << tr.tx << " " << tr.ty << ")\"/>" << endl; 
}

void SVGCanvas::circle(RGBA8 c, agg::trans_affine tr)
{
    tr *= mOffset;
    mOutput << "    <circle r=\"0.5\"" << endl;
    complete(c, tr, 12);
}

void SVGCanvas::square(RGBA8 c, agg::trans_affine tr)
{
    tr *= mOffset;
    mOutput << "    <rect x=\"-0.5\" y=\"-0.5\" width=\"1\" height=\"1\" " << endl;
    complete(c, tr, 10);
}

void SVGCanvas::triangle(RGBA8 c, agg::trans_affine tr)
{
    static string triString = "";

    if (triString.length() == 0) {
        ostringstream out;
        out << "    <polygon points=\"";
        double x, y;
        for (int i = 0; primShape::triangle.vertex(&x, &y, i); i++) {
            if (i) out << ' ';
            out << x << ',' << y;
        }
        out << '"';
        triString = out.str();
    }


    tr *= mOffset;
    mOutput << triString << endl;
    complete(c, tr, 13);
}

SVGCanvas::SVGCanvas(const char* opath, int width, int height, bool crop, const char* desc, int length)
:   Canvas(width, height),
    mCropped(crop),
    mOutput(opath),
    mDescription(desc),
    mLength(length)
{
    if (mLength == -1 && mDescription)
        mLength = strlen(mDescription);
}



