/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * $Id: modulecall.h,v 1.3 2003/02/16 15:32:34 bad Exp $
 * **********************************************************/


/*
 * modulecall.h
 *
 *        Monitor <-->  Module (kernel driver) interface
 */

#ifndef _MODULECALL_H
#define _MODULECALL_H

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_MONITOR
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMKERNEL
#include "includeCheck.h"

#include "x86.h"
#include "iocontrols.h"

/*
 *----------------------------------------------------------------------
 *
 * ModuleCallType --
 *
 *      Enumaration of support calls done by the module
 *
 *----------------------------------------------------------------------
 */

typedef enum ModuleCallType {
   MODULECALL_NONE = 100,            /* */
   
   /*
    * All I/O interrupts go under MODULECALL_INTR, special-cased
    * in task switching code
    */
   MODULECALL_INTR,                  
   MODULECALL_LOCKPAGE,                  
   MODULECALL_UNLOCKPAGE,                  

   MODULECALL_ISMPNLOCKED, 
   /*
    * calls to user mode
    */
   MODULECALL_USERRETURN,            /* return code for USER RPC */
   MODULECALL_LAST                   /* Number of entries. Must be the last one */
} ModuleCallType;

#define MODULECALL_USER_START 300
#define MODULECALL_USER_END   399

#define MODULECALL_CROSS_PAGE_LEN    1
#define MODULECALL_CROSS_PAGE_START  4

/* we don't want it in the final version */
/*#define USE_BLUE_SCREEN */

/*
 *----------------------------------------------------------------------
 *
 * VMCrossPage --
 *
 *      data structure shared between the monitor and the module
 *      that is used for crossing between the two.
 *      Accessible as vm->cross (kernel module) and CROSS_PAGE
 *      (monitor)
 *
 *      Exactly one page long
 *
 *----------------------------------------------------------------------
 */

/*
 * We arrived at 1008 as follows. The first 1024 bytes of the cross
 * page contains the version number (4 bytes) and then the task switch
 * code aligned at a 16 byte boundary. This alignment allows the
 * decoders to get a good start. We allocate 1024 bytes for all this
 * although the task switch code is only about 192 bytes large so
 * that the context information (data) that follows this code fragment
 * is on another 1024 byte chunk of memory. This separation makes the
 * trace cache of the P4 happy. If it any closer, the trace cache
 * gets flushed each time we write to the context info data area. 
 */
#define CONTEXT_SWITCH_CODE_SIZE 1008

#define MAX_PASSTHROUGH_PORT 1024
#define NUM_PASSTHROUGH_WORDS (MAX_PASSTHROUGH_PORT / 32)

#ifdef __GNUC__
 struct ContextInfo {
#else
 typedef  struct ContextInfo {
#endif
    DTRWords    gdtr;
    DTRWords    ldtr;
    DTRWords    idtr;
    uint16 tr;
    uint8 jump[6];
    Task task;   
#ifdef __GNUC__
 } __attribute__ ((packed));

typedef struct ContextInfo ContextInfo;
#else
 } ContextInfo;
#endif

#ifdef __GNUC__
 struct VMCrossPage {
#else
 typedef struct VMCrossPage {
#endif
   /*
    * Version checking. Should remain at offset 0
    */ 
   uint32 version; 
  

   /*
    * Changes in contextSwitchCode location need to
    * be reflected in addrlayout.h:COMINGBACK_OFFSET
    * (ASSERT fail otherwise)
    */
   uint32 _pad[3];
   uint8 contextSwitchCode[CONTEXT_SWITCH_CODE_SIZE];

   /*
    * iobitMap must immediately follow monContext.task
    * (ASSERT fail otherwise)
    */
   ContextInfo hostContext;
   ContextInfo monContext;
   uint32      iobitMap[NUM_PASSTHROUGH_WORDS];

   struct VMCrossPage * hostVA;
   PTE    hostSharedPTE;

   uint32 hostContextVA;


   uint32 switchPDOffset;     /* index into the page directory of switchPDE */
   PDE  switchPDE;            /* PDE that points to switch page table */


   LA monitorLinearAddr;      /* base location of monitor in linear space */
   LA crosspageLinearAddr;    /* location of crosspage page in both linear spaces */

   /*
    * ModuleCall parameters
    */
   ModuleCallType moduleCallType;
   uint32 args[4];
   uint32 retval;

   VA userSharedAreaAddr;

   /* host irq relocation values */
   int irqRelocateOffset[2]; 

#if defined(WINNT_DDK) && defined(VMX86_DEBUG)
   void *vmPtr;  /* Monitor to VmDriver structure, useful for debugging */
#endif

#ifdef __GNUC__
 } __attribute__ ((packed));

typedef struct VMCrossPage VMCrossPage;
#else
 } VMCrossPage;
#endif

#define MODULECALL_CROSSPAGE_SIZE     0x5d0
#define MODULECALL_CROSSPAGE_LAOFFSET 0x5a8


#define CROSSPAGE_VERSION (0x1776 + VERSION_MAGIC)

#ifndef VMX86_SERVER
#define MONITOR_LINEAR_START            (CROSS_PAGE->monitorLinearAddr)
#ifdef MONITOR
#define CROSS_PAGE  ((VMCrossPage *)(MONITOR_BASE_VADDR + VPN_2_VA(CROSS_PAGE_START)))
#endif
#else
#ifdef VMX86_8MB_MONITOR
#define MONITOR_LINEAR_START		0xff800000
#else
#define MONITOR_LINEAR_START		0xffc00000
#endif
#endif

#define NULLPAGE_LINEAR_START           (MONITOR_LINEAR_START + PAGE_SIZE*NULLPAGE_START)
   

struct passthrough_iorange {
  unsigned short ioBase;	/* Base of range to pass through */
  unsigned short numPorts;	/* Length of range */
};

#endif
