/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * **********************************************************/
#ifdef VMX86_DEVEL
char rcsId_vmx86[] = "$Id: vmx86.c,v 1.6 2003/02/16 17:42:22 bad Exp $";
#else
#define FILECODE "F(301)"
#endif

/*
 * vmx86.c --
 *
 *     Platform independent routines for creating/destroying/running
 *     virtual machine monitors.
 *
 */

#include "hostif.h"     /* Must come before any linux header files */

#ifdef linux
# include "driver-config.h" /* Versioned symbols from linux/string.h */
# include <linux/string.h> /* memset() in the kernel */
# include <linux/sched.h> /* jiffies from the kernel */
#elif __FreeBSD__
# include <sys/param.h>
# include <sys/systm.h>
# include <string.h>
#elif defined(__NetBSD__)
# include <sys/param.h>
# include <sys/systm.h>
# include <sys/lock.h>
# if __NetBSD_Version__ > 105009900
#  include <uvm/uvm_extern.h>
# else
#  include <vm/vm.h>
# endif
#elif defined(WINNT_DDK)
# include <string.h>
#else
# error "Unknown platform"
#endif

#include "x86.h"
#include "vm_types.h"
#include "vm_assert.h"
#include "modulecall.h"
#include "vmx86.h"
#include "task.h"
#include "initblock.h"
#include "vm_asm.h"
#include "vtrace.h"
#include "memmgmt.h"
#ifdef SUPPORT_PASSTHROUGH
#include "passthrough.h"
#endif
#include "wslimits.h"

/*
 * Keep track of the virtual machines that have been
 * created using the following structures.
 *
 */

static VMDriver *vmDriverList = NULL;

static Bool lockedPageLimitMonitor;
static int32 lockedPageLimit;
static int32 numLockedPages;
static int32 vmCount;

static void FreeAllResources(VMDriver *vm);

static uint32 unlockedPages;
static uint32 dirtyUnlockedPages;


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_CreateVM --
 *
 *      Allocate and initialize a driver structure for a virtual machine.
 *
 * Results:
 *      VMDriver structure or NULL on error.
 *
 * Side effects:
 *       Memory allocated.
 *
 *----------------------------------------------------------------------
 */

VMDriver *
Vmx86_CreateVM(void *uniqHandle,    // Unique id for VM to be created
	       void *processId,     // Process creating VM.
	       WslimitsInfo *wslimitsInfoIn)  // Information about Resource limits
{
   VMDriver *vm;
   VMDriver **vmp;
   
   /*
    * For NT:
    * The code within a critical region guarded by an spin lock must
    * neither be pageable nor make any references to pageable data.
    */

   vm = (VMDriver *) HostIF_AllocKernelMem(sizeof *vm, TRUE);
   if (vm == NULL) {
      return NULL;
   }
   memset(vm, 0, sizeof *vm);
#ifdef linux
   vm->logFD = -1;
#endif

   HostIF_GlobalVMLock(0);

   if (vmCount >= MAX_VMS) {
      HostIF_GlobalVMUnLock(0);
      HostIF_FreeKernelMem(vm);
      return NULL;
   }
   vmCount++;

   vm->count++;
   for (vmp = &vmDriverList; *vmp != NULL; vmp = &(*vmp)->nextDriver) {
   }
   *vmp = vm;

   ASSERT(sizeof (unsigned) == sizeof vm);
   vm->id = (unsigned) vm >> 1;
   vm->uniqHandle = uniqHandle;
   vm->processID = processId;

   memcpy(&vm->wslimitsInfo, (char *) wslimitsInfoIn, sizeof(WslimitsInfo));

   HostIF_GlobalVMUnLock(0);

   if (HostIF_Init(vm)) {
      Vmx86_ReleaseVM(vm);
      return NULL;
   }

   return vm;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_BindVM  --
 *
 *      Bind to an existing VM.
 *
 * Results:
 *      VMDriver structure or NULL on error.
 *
 * Side effects:
 *	Increment VM reference count.
 *
 *----------------------------------------------------------------------
 */

VMDriver *
Vmx86_BindVM(int id)
{
   VMDriver *vm;

   HostIF_GlobalVMLock(14);
   for (vm = vmDriverList; vm != NULL; vm = vm->nextDriver) {
      if (vm->id == id) {
	 vm->count++;
	 break;
      }
   }
   HostIF_GlobalVMUnLock(14);
   return vm;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_ReleaseVM  --
 *
 *      Release a VM (either created here or from a bind).
 *
 * Results:
 *      zero if successful
 *
 * Side effects:
 *	Decrement VM reference count.
 *      Release resources (those that are left) when count reaches 0.
 *
 *----------------------------------------------------------------------
 */
int
Vmx86_ReleaseVM(VMDriver *vm)
{
   VMDriver **vmp;

   HostIF_GlobalVMLock(1);

   /*
    * Do reference counting first
    */

   if (--vm->count > 0) {
      HostIF_GlobalVMUnLock(1);
      return 0;
   }

   ASSERT(vm->count == 0);

   /*
    *   Deletion of the virtual machine from list
    */

   for (vmp = &vmDriverList; *vmp != vm; vmp = &(*vmp)->nextDriver) {
      ASSERT(*vmp != NULL);
   }
   *vmp = vm->nextDriver;

   vmCount--;

   HostIF_GlobalVMUnLock(1);

   Log("Vmx86_ReleaseVM: unlocked pages: %d, unlocked dirty pages: %d\n",
       unlockedPages, dirtyUnlockedPages);

   FreeAllResources(vm);

   return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_InitVM --
 *
 *      Initializaiton of the VM.  Expects all initial arguments
 *      to be part of the InitBlock structure.
 *
 * Results:
 *     non-zero on error, zero on success;
 *
 * Side effects:
 *      many
 *
 *----------------------------------------------------------------------
 */

int
Vmx86_InitVM(VMDriver *vm,
             InitBlock *initParams) // Initial params from the VM */
{
   int retval;

   if (initParams->magicNumber != INIT_BLOCK_MAGIC) {
     Warning("Bad magic number for init block 0x%x\n", initParams->magicNumber);
     return 1;
   }
   vm->logFD = initParams->logFD;

   HostIF_InitFP(vm);

   /*
    * Initialize the driver's part of the cross-over page used to
    * talk to the monitor
    */

   retval = Task_InitCrosspage(vm,initParams);
   if (retval) {
      Warning("Task crosspage init died with retval=%d\n", retval);
      goto error;
   }

   unlockedPages = dirtyUnlockedPages = 0;

   Vmx86_SetStartTime(&initParams->st);
   return 0;

 error:
   FreeAllResources(vm);
   return 1;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_LateInitVM --
 *
 *      Do late initialization of the driver.
 *	This should be called after Vmx86_CreateVM and
 *	after all the user-level device initialization.
 *
 * Results: 
 *	non-zero on error, zero on success;
 *
 * Side effects:
 *      VTrace_Init is called
 *
 *----------------------------------------------------------------------
 */

int
Vmx86_LateInitVM(VMDriver *vm)
{
   VTrace_Init(vm);
   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_RunVM  --
 *
 *      Return to the monitor.
 *      Main interaction between the module and the monitor.
 *      Leaves the loop only to go back to user mode.
 *      Every interation of the loop switches back to the monitor
 *
 * Results:
 *      Returns the MODULECALL that forced it to leave the loop.
 *      back to the IOCTL handler of the module device driver
 *
 * Side effects:
 *      Dispatches the messages, everything can change
 *
 *----------------------------------------------------------------------
 */
#ifdef __FreeBSD__
#include <sys/param.h>
#include <sys/systm.h>
#include <vm/vm.h>
#include <vm/vm_param.h>
#include <sys/lock.h>
#include <vm/pmap.h>
#endif 
int 
Vmx86_RunVM(VMDriver *vm)
{
   ModuleCallType op;
   int retval;

   ASSERT(vm && vm->crosspage);


   retval = MODULECALL_USERRETURN;

   while(1) {

      vm->crosspage->retval = retval;

      /*
       * Task_Switch changes the world to the monitor.
       * The monitor is waiting in the BackToHost routine
       */
      Task_Switch(vm);

      op = vm->crosspage->moduleCallType;

      retval = MODULECALL_USER_START;

      if (op >  MODULECALL_USER_START&& op < MODULECALL_USER_END) {
         return op;
      }


      switch (op) {
      case MODULECALL_NONE :
         break;


      case MODULECALL_INTR : {
	 /*
	  * Note that as of SMP support, irq is the actual Interrupt number
	  * rather than the IRQ, b/c the IO APIC redirects IRQs every which
	  * way. --Jeremy.
	  */
#ifdef notdef
	 if (vm->crosspage->args[1] == USERCALL_NONE) {
	    /* nothing to be done if MODULECALL_USERRETURN */
	    retval = MODULECALL_USERRETURN;
	    break;
	 }
#endif
	 return vm->crosspage->args[1];
      }
      case MODULECALL_ISMPNLOCKED : {
         MPN mpn = vm->crosspage->args[0];
         retval = HostIF_IsMPNLocked(vm, mpn);
         break;
      }

      case MODULECALL_LOCKPAGE: {
         char dummy;
         void *addr = (void *)vm->crosspage->args[0];
#ifdef linux
	 HostIF_LockKernel();
#endif
         retval = Vmx86_LockPage(vm, addr, FALSE);
         if (retval != PAGE_LOCK_FAILED && retval != PAGE_LOCK_LIMIT_EXCEEDED) {
            if (HostIF_CopyFromUser(&dummy, addr, 1) == 0) {
               HostIF_CopyToUser(addr, &dummy, 1);
            } else {
               (void) Vmx86_UnlockPage(vm, addr, FALSE);
               retval = 0;
            }
         }
#ifdef linux
	 HostIF_UnLockKernel();
#endif
         break;
      }

      case MODULECALL_UNLOCKPAGE: {
         char dummy;
         void *addr = (void *)vm->crosspage->args[0];
         int dirty = vm->crosspage->args[1];
#ifdef linux
	 HostIF_LockKernel();
#endif

         retval = FALSE;
         unlockedPages++;
         if (dirty) {
            dirtyUnlockedPages++;
            if (HostIF_CopyFromUser(&dummy, addr, 1) == 0) {
               if (HostIF_CopyToUser(addr, &dummy, 1) == 0) {
                  if (!Vmx86_UnlockPage(vm, addr, FALSE)) {
                     retval = TRUE;
                  }
               }
            }
         } else {
            if (!Vmx86_UnlockPage(vm, addr, FALSE)) {
               retval = TRUE;
            }
         }
#ifdef linux
	 HostIF_UnLockKernel();
#endif
         break;
      }

      default:
         Panic("ModuleCallLoop %d not supported \n",op);
      }
   }

   ASSERT(op >= MODULECALL_USER_START && op < MODULECALL_USER_END);
   return op;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_SetStartTime --
 *
 *      Initial the data structures that track the starting time of the
 *      virtual machine.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      none
 *
 *
 *----------------------------------------------------------------------
 */

void
Vmx86_SetStartTime(VmTimeStart *st)	// OUT: return value
{
   uint32 flags;
#ifdef linux
   uint32 dummy;
#endif

   SAVE_FLAGS(flags);
   CLEAR_INTERRUPTS();

#ifdef linux
   Div643264((uint64)jiffies * 1000000, HZ, &st->time, &dummy);
#else
   st->time = HostIF_ReadTime();
#endif
   st->count = GET_TSC();

   RESTORE_FLAGS(flags);
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetMHzEstimate
 *
 *      Return an estimate the of the processor's MHZ rating, based on
 *      the ratio of the cycle counter and gettimeofday
 *
 * Results:
 *
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

uint32
Vmx86_GetMHzEstimate(VmTimeStart *st)	// IN: start time
{

   VmTimeType vDiff, rDiff;
   uint32 flags;
#ifdef linux
   uint32 dummy;
#endif

   /*
    * Compute the cycle rate based on the change in the
    * cycle counter and real time clock since we open the
    * device.
    */

   SAVE_FLAGS(flags);
   CLEAR_INTERRUPTS();
   vDiff = GET_TSC() - st->count;
#ifdef linux
   Div643264((uint64)jiffies * 1000000, HZ, &rDiff, &dummy);
   rDiff -= st->time;
#else
   rDiff = HostIF_ReadTime() - st->time;
#endif
   RESTORE_FLAGS(flags);

   if (rDiff == 0) {
      return 0;  /* Failure */
   }
   /* Need to do 32bit divide since 64bit one doesn't seem to
    * work in a linux device driver. Scale the numbers back into
    * 32bit numbers.
    */
   while (vDiff > (0xffffffff/10)) {
      vDiff >>= 1;
      rDiff >>= 1;
   }
   return ((10*(uint32)vDiff)/(uint32)rDiff + 5)/10;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_CurrentVM --
 *
 *      Return the VMDriver structure associated with
 *      the handle
 *
 *
 * Results:
 *      VMDriver * structure.NULL if not available
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
VMDriver *
Vmx86_CurrentVM(void *uniqHandle)
{
  VMDriver *vm;

  HostIF_GlobalVMLock(2);

  vm = vmDriverList;

  while (vm) {
     if (vm->uniqHandle == uniqHandle) {
        break;
     }
     vm = vm->nextDriver;
  }

  HostIF_GlobalVMUnLock(2);

  return vm;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetVMForProcess  --
 *
 *      Return the VMDriver structure associated with
 *      a process
 *
 *
 * Results:
 *      VMDriver * structure.NULL if not available
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

VMDriver *
Vmx86_GetVMforProcess(void *processId)
{
  VMDriver *vm;

  HostIF_GlobalVMLock(3);

  vm = vmDriverList;

  while (vm) {
    if (vm->processID == processId) {
       break;
    }
    vm = vm->nextDriver;
  }

  HostIF_GlobalVMUnLock(3);

  return vm;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetNumVMs  --
 *
 *      Return the number of VMs.
 *
 * Results:
 *      The number of VMs.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
int32
Vmx86_GetNumVMs()
{
   return vmCount;
}

int32
Vmx86_GetTotalMemUsage()
{
   VMDriver *vm;
   int totalmem = 0;

   HostIF_GlobalVMLock(15);
   vm = vmDriverList;

   while (vm) {
      totalmem += vm->wslimitsInfo.memsize;
      vm = vm->nextDriver;
   }
   
   HostIF_GlobalVMUnLock(15);
   return totalmem;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_SetLockedPagesLimit --
 *
 *      Set the hard limit for the number of locked pages.
 *      The limit can be set only when one VM is running or if it not set.
 *
 * Results:
 *      Returns TRUE on success and FALSE on failure to set the limit
 *
 * Side effects:
 *      Hard limit changed.
 *
 *----------------------------------------------------------------------
 */
Bool
Vmx86_SetLockedPagesLimit(int32 limit)
{
   Bool retval = FALSE;

   HostIF_GlobalVMLock(4);
   if (!lockedPageLimitMonitor && 
		 (lockedPageLimit == 0 || (lockedPageLimit != 0 && vmCount == 1))) {
      lockedPageLimit = limit;
      retval = TRUE;
   }
   HostIF_GlobalVMUnLock(4);

   return retval;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_SetLockedPageLimitMonitorStatus --
 *
 *      Set the status of the hard limit monitor - 1 is on, 0 is off.
 *
 * Results:
 *      Returns TRUE on success and FALSE on failure to set the limit
 *
 * Side effects:
 *      Hard limit changed.
 *
 *----------------------------------------------------------------------
 */
void
Vmx86_SetLockedPageLimitMonitorStatus(Bool status)
{
   lockedPageLimitMonitor = status;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_ChangeLockedPagesLimit --
 *
 *      Change the hard limit for the number of locked pages.
 *      The limit can be reduced at any time.
 *
 * Results:
 *      Returns new limit.
 *
 * Side effects:
 *      Hard limit changed.
 *
 *----------------------------------------------------------------------
 */
int32
Vmx86_ChangeLockedPagesLimit(int32 limit)
{
   HostIF_GlobalVMLock(11);
   lockedPageLimit = limit;
   HostIF_GlobalVMUnLock(11);

   return lockedPageLimit;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetLockedPagesLimit --
 *
 *      Set the hard limit for the number of locked pages.
 *
 * Results:
 *      None
 *
 * Side effects:
 *      Hard limit changed.
 *
 *----------------------------------------------------------------------
 */
int32
Vmx86_GetLockedPagesLimit()
{
   return lockedPageLimit;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_LockPage --
 *
 *      Lock a page.
 *
 * Results:
 *      None
 *
 * Side effects:
 *      Number of global and per-VM locked pages increased.
 *
 *----------------------------------------------------------------------
 */
int
Vmx86_LockPage(VMDriver *vm, void * addr, Bool userPage)
{
   if (lockedPageLimit != 0 && numLockedPages >= lockedPageLimit) {
      return PAGE_LOCK_LIMIT_EXCEEDED;
   } else {
      int retval = (int)HostIF_LockPage(vm,addr);
      if (retval != 0) {
         HostIF_GlobalVMLock(5);

         numLockedPages++;

         HostIF_GlobalVMUnLock(5);

         vm->memMgmtInfo.currentLocked++;

         if (userPage) {
            vm->stats.userLockedPages++;
            if (vm->stats.userLockedPages > vm->stats.maxUserLockedPages) {
               vm->stats.maxUserLockedPages = vm->stats.userLockedPages;
            }
         } else {
            vm->stats.monitorLockedPages++;
            if (vm->stats.monitorLockedPages > vm->stats.maxMonitorLockedPages) {
               vm->stats.maxMonitorLockedPages = vm->stats.monitorLockedPages;
            }
         }
         return retval;
      } else  {
         return PAGE_LOCK_FAILED;
      }
   }
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_UnlockPage --
 *
 *      Unlock a page.
 *
 * Results:
 *      None
 *
 * Side effects:
 *      Number of global and per-VM locked pages decreased.
 *
 *----------------------------------------------------------------------
 */
int
Vmx86_UnlockPage(VMDriver *vm, void * addr, Bool userPage)
{
   int retVal = (int)HostIF_UnlockPage(vm, addr);
   if (retVal == 0) {
      HostIF_GlobalVMLock(6);

      numLockedPages--;

      HostIF_GlobalVMUnLock(6);

      vm->memMgmtInfo.currentLocked--;

      if (userPage) {
         vm->stats.userLockedPages--;
      } else {
         vm->stats.monitorLockedPages--;
      }
   }

   return retVal;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetMemInfo --
 *
 *      Return the info about all VMs.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      VMGetMemInfoArgs is filled in.
 *
 *----------------------------------------------------------------------
 */
Bool
Vmx86_GetMemInfo(VMDriver *curVM,
                 int32 curVMOnly,
                 VMGetMemInfoArgs *outArgs,
                 int outArgsLength)
{
   VMDriver *vm;
   int i;
   int outSize;
   int wantedVMs;

   HostIF_GlobalVMLock(7);

   if (curVMOnly) {
      wantedVMs = 1;
   } else {
      wantedVMs = vmCount;
   }

   outSize = VM_GET_MEM_INFO_SIZE(wantedVMs);
   if (outSize > outArgsLength) {
      HostIF_GlobalVMUnLock(7);
      return FALSE;
   }

   outArgs->numVMs = wantedVMs;
   outArgs->numLockedPages = numLockedPages;
   outArgs->maxLockedPages = lockedPageLimit;
   outArgs->callerIndex = -1;

   if (curVM != NULL) {
      if (wantedVMs == 1) {
         outArgs->memMgmtInfo[0] = curVM->memMgmtInfo;
         outArgs->callerIndex = 0;
      } else {
         vm = vmDriverList;
         i = 0;
         outArgs->callerIndex = -1;
         while (vm != NULL && i < vmCount) {
	    if (vm == curVM) {
	       outArgs->callerIndex = i;
	    }
	    outArgs->memMgmtInfo[i] = vm->memMgmtInfo;
	    i++;

	    vm = vm->nextDriver;
         }
      }
   }

   HostIF_GlobalVMUnLock(7);

   return TRUE;
}


/*
 *-----------------------------------------------------------------------------
 *
 * Vmx86_GetMemInfoCopy --
 *
 *    Return the information about all VMs by copying the data out to user
 *    memory.
 *
 *    On input, outArgs->numVMs indicates how much space has been allocated for
 *    the information. On output, it indicates how much space has been
 *    filled --hpreg
 *
 * Results:
 *    TRUE on success
 *    FALSE on failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

Bool
Vmx86_GetMemInfoCopy(VMDriver *curVM,           // IN
                     VMGetMemInfoArgs *outArgs) // IN/OUT
{
   /*
    * XXX This static variable is not used in a SMP-safe way at first glance.
    *     A deeper look shows that it is OK, because the code is only used on
    *     Linux in the ioctl path, which is protected by the Big Kernel
    *     Lock --hpreg
    */
   static union {
      char f[VM_GET_MEM_INFO_SIZE(MAX_VMS)];
      VMGetMemInfoArgs s;
   } buf;

   if (   curVM == NULL
       || HostIF_CopyFromUser(buf.f, outArgs, VM_GET_MEM_INFO_SIZE(1))) {
      return FALSE;
   }

   HostIF_GlobalVMLock(8);

   /* Now that we have the lock, we can read vmCount --hpreg */
   if (buf.s.numVMs < vmCount) {
      HostIF_GlobalVMUnLock(8);

      return FALSE;
   }

   buf.s.numLockedPages = numLockedPages;
   buf.s.maxLockedPages = lockedPageLimit;
   if (buf.s.numVMs == 1) {
      buf.s.memMgmtInfo[0] = curVM->memMgmtInfo;

      HostIF_GlobalVMUnLock(8);
   } else {
      VMDriver *vm;
      int vmIndex;

      for (vm = vmDriverList, vmIndex = 0;
           vm;
           vm = vm->nextDriver, vmIndex++) {
         ASSERT(vmIndex < vmCount);
         if (vm == curVM) {
            buf.s.callerIndex = vmIndex;
         }
         buf.s.memMgmtInfo[vmIndex] = vm->memMgmtInfo;
      }
      ASSERT(vmIndex == vmCount);

      HostIF_GlobalVMUnLock(8);

      buf.s.numVMs = vmIndex;
   }

   if (HostIF_CopyToUser(outArgs, buf.f, VM_GET_MEM_INFO_SIZE(buf.s.numVMs))) {
      return FALSE;
   }

   return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_SetMemInfo --
 *
 *      Set the memory management information about this VM.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      vm->memMgmtInfo is filled in.
 *
 *----------------------------------------------------------------------
 */
void
Vmx86_SetMemInfo(VMDriver *curVM,
                 VMSetMemInfoArgs *inArgs)
{
   HostIF_GlobalVMLock(9);

   if (inArgs->info.pageFaultRate >= 0) {
      curVM->memMgmtInfo.pageFaultRate = inArgs->info.pageFaultRate;
   }
   if (inArgs->info.lockFailRate >= 0) {
      curVM->memMgmtInfo.lockFailRate = inArgs->info.lockFailRate;
   }
   if (inArgs->info.maxLocked >= 0) {
      curVM->memMgmtInfo.maxLocked = inArgs->info.maxLocked;
   }
   if (inArgs->info.highWaterMark >= 0) {
      curVM->memMgmtInfo.highWaterMark = inArgs->info.highWaterMark;
   }
   if (inArgs->info.waitingForMemoryTime >= 0) {
      curVM->memMgmtInfo.waitingForMemoryTime = inArgs->info.waitingForMemoryTime;
   }
   if (inArgs->info.poweringOn >= 0) {
      curVM->memMgmtInfo.poweringOn = inArgs->info.poweringOn;
   }
   if (inArgs->info.hasFocus >= 0) {
      curVM->memMgmtInfo.hasFocus = inArgs->info.hasFocus;
   }

   HostIF_GlobalVMUnLock(9);
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_PAEEnabled --
 *
 *      Is PAE enabled?
 *
 * Results:
 *      TRUE if PAE enabled.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
Bool
Vmx86_PAEEnabled()
{
   uint32 cr4;

   GET_CR4(cr4);
   return (cr4 & CR4_PAE) != 0;
}

/*
 *----------------------------------------------------------------------
 *
 * FreeAllResources
 *
 *     Free all the resources allocated for a vm.
 *
 *
 * Results:
 *      None
 *
 * Side effects:
 *      Memory freed.
 *
 *----------------------------------------------------------------------
 */

static void
FreeAllResources(VMDriver *vm)
{
   if (vm) {
#ifdef USE_PERFCOUNTERS
      /*
       * Can't log in PerfCtr_Release() when we're holding the lock
       * and (in any case) the VM has be dequeued.
       */
      PerfCtr_Release(vm, FALSE);
#endif

#ifdef SUPPORT_PASSTHROUGH
      Passthrough_Release(vm);
#endif

      HostIF_FreeAllResources(vm);

      HostIF_GlobalVMLock(10);
      numLockedPages -= vm->memMgmtInfo.currentLocked;
      HostIF_GlobalVMUnLock(10);

      HostIF_FreeKernelMem(vm);
   }
   VTrace_FreeAllResources();
}
