#ifndef __PGTBL_H__
#   define __PGTBL_H__


#include "compat_pgtable.h"
#include "compat_spinlock.h"


/*
 *-----------------------------------------------------------------------------
 *
 * PgtblPte2Page --
 *
 *    Returns the page structure associated to a Page Table Entry.
 *
 *    This function is not allowed to schedule() because it can be called while
 *    holding a spinlock --hpreg
 *
 * Results:
 *    The page structure if the page table entry points to a physical page
 *    mem_map if the page table entry does not point to a physical page
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static INLINE struct page *
PgtblPte2Page(pte_t *pte) // IN
{
   if (pte_present(*pte) == 0) {
      return mem_map;
   }

   return compat_pte_page(*pte);
}


/*
 *-----------------------------------------------------------------------------
 *
 * PgtblVa2PageLocked --
 *
 *    Walks through the hardware page tables to try to find the page structure
 *    associated to a virtual address.
 *
 *    The mm->page_table_lock must be held, so this function is not allowed to
 *    schedule() --hpreg
 *
 * Results:
 *    Same as PgtblPte2Page()
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static INLINE struct page *
PgtblVa2PageLocked(struct mm_struct *mm, // IN: Mm structure of a process
                   VA addr)              // IN: Address in the virtual address
                                         //     space of that process
{
   /* Level 1 entry --hpreg */
   pgd_t *pgd;
   /*
    * Level 2 entry (folded into the Level 1 entry by the compiler if the host
    * is not in PAE mode) --hpreg
    */
   pmd_t *pmd;
   /* Level 3 entry --hpreg */
   pte_t *pte;
   struct page *page;

   pgd = pgd_offset(mm, addr);
   if (pgd_present(*pgd) == 0) {
      return 0;
   }

   pmd = pmd_offset(pgd, addr);
   if (pmd_present(*pmd) == 0) {
      return 0;
   }

   pte = pte_offset_map(pmd, addr);
   page = PgtblPte2Page(pte);
   pte_unmap(pte);

   return page;
}


/*
 *-----------------------------------------------------------------------------
 *
 * PgtblVa2Page --
 *
 *    Walks through the hardware page tables of the current process to try to
 *    find the page structure associated to a virtual address.
 *
 * Results:
 *    Same as PgtblVa2PageLocked()
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static INLINE struct page *
PgtblVa2Page(VA addr) // IN
{
   struct mm_struct *mm;
   struct page *page;

   mm = current->mm;
   if (compat_get_page_table_lock(mm)) {
      spin_lock(compat_get_page_table_lock(mm));
   }
   page = PgtblVa2PageLocked(mm, addr);
   if (compat_get_page_table_lock(mm)) {
      spin_unlock(compat_get_page_table_lock(mm));
   }

   return page;
}


#endif /* __PGTBL_H__ */
