/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * $Id: vm_asm.h,v 1.2 2003/02/16 01:29:51 bad Exp $
 * **********************************************************/


/*
 * vm_asm.h
 *
 * asm macros
 */



#ifndef VM_ASM_H
#define VM_ASM_H

#include "x86.h"


#ifdef __GNUC__
/* Checked against the Intel manual and GCC --hpreg */
#define __GCC_SET_DT(dt, constraint, var) do { \
   __asm__(                                    \
      "l" #dt "dt %0"                          \
      :                                        \
      : #constraint (var)                   \
   );                                          \
} while (0)

#define SET_IDT(_idt) __GCC_SET_DT(i, m, _idt)
#define SET_GDT(_gdt) __GCC_SET_DT(g, m, _gdt)
#define SET_LDT(_ldt) __GCC_SET_DT(l, rm, _ldt)


/* Checked against the Intel manual and GCC --hpreg

   volatile because the *dt can change without the compiler knowing it
   (when we use l*dt). */
#define __GCC_GET_DT(dt, constraint, var) do { \
   __asm__ __volatile__(                       \
      "s" #dt "dt %0"                          \
      : #constraint (var)                  \
   );                                          \
} while (0)

#define GET_IDT(_idt) __GCC_GET_DT(i, =m, _idt)
#define GET_GDT(_gdt) __GCC_GET_DT(g, =m, _gdt)
#define GET_LDT(_ldt) __GCC_GET_DT(l, =rm, _ldt)


#define SET_TR(_tr)      __asm__("ltr   %0" : : "m" (_tr))
#define GET_TR(_tr)      __asm__("str   %0" : : "m" (_tr))


/* Checked against the Intel manual and GCC --hpreg */
#define __GCC_SET(reg, var) do { \
   __asm__(                      \
      "movl %0, %%" #reg         \
      :                          \
      : "r" (var)                \
      : "cc"                     \
   );                            \
} while (0)

#define SET_CR0(var) __GCC_SET(cr0, var)
#define SET_CR2(var) __GCC_SET(cr2, var)
#define SET_CR3(var) __GCC_SET(cr3, var)
#define SET_CR4(var) __GCC_SET(cr4, var)

#define SET_DR0(var) __GCC_SET(dr0, var)
#define SET_DR1(var) __GCC_SET(dr1, var)
#define SET_DR2(var) __GCC_SET(dr2, var)
#define SET_DR3(var) __GCC_SET(dr3, var)
#define SET_DR6(var) __GCC_SET(dr6, var)
#define SET_DR7(var) __GCC_SET(dr7, var)


/* Checked against the Intel manual and GCC --hpreg

   volatile because CRs and DRs can change without the compiler knowing it
   (when there is a page fault, when a breakpoint occurs, and moreover it seems
   there is no way to teach gcc that smsw clobbers cr0 for example). */
#define __GCC_GET(reg, var) do { \
   __asm__ __volatile__(         \
      "movl %%" #reg ", %0"      \
      : "=r" (var)               \
      :                          \
      : "cc"                     \
   );                            \
} while (0)

#define GET_CR0(var) __GCC_GET(cr0, var)
#define GET_CR2(var) __GCC_GET(cr2, var)
#define GET_CR3(var) __GCC_GET(cr3, var)
#define GET_CR4(var) __GCC_GET(cr4, var)

#define GET_DR0(var) __GCC_GET(dr0, var)
#define GET_DR1(var) __GCC_GET(dr1, var)
#define GET_DR2(var) __GCC_GET(dr2, var)
#define GET_DR3(var) __GCC_GET(dr3, var)
#define GET_DR6(var) __GCC_GET(dr6, var)
#define GET_DR7(var) __GCC_GET(dr7, var)


#define CLTS()           __asm__("clts" ::);

#define FNCLEX()         __asm__("fnclex" ::);

#define TLB_INVALIDATE_PAGE(_addr) { \
     __asm__ __volatile__("invlpg %0": :"m" (*(char *) _addr)); \
}

#define TLB_INVALIDATE_PAGE_OFF_FS(_addr) { \
     __asm__ __volatile__("fs; invlpg %0": :"m" (*(char *) _addr)); \
}




#define ENABLE_INTERRUPTS() __asm__ __volatile__ ("sti": : :"memory")
#define CLEAR_INTERRUPTS()  __asm__ __volatile__ ("cli": : :"memory")
#define RAISE_INTERRUPT(_x)  __asm__ __volatile__("int %0" :: "g" (_x))
#define RETURN_FROM_INT()   __asm__ __volatile__("iret" :: ) 
   

#define FARCALLCONCAT(_a,_b,_c,_d) _a ## #_b ## _c ## #_d
#define FARCALL_IMMEDIATE(SEG,OFF) { \
   __asm__ __volatile__( FARCALLCONCAT("lcall $", SEG, ",$",OFF)  ::);  \
}

#define SAVE_FLAGS(x) { \
     __asm__ __volatile__("pushfl ; popl %0":"=g" (x): /* no input */ :"memory");\
}
   
#define RESTORE_FLAGS(x) { \
     __asm__ __volatile__("pushl %0 ; popfl": /* no output */ :"g" (x):"memory");\
}


static INLINE void SET_DS(unsigned long val)
{
   __asm__ __volatile__("mov %w0,%%ds": /* no output */ :"r" (val));
}

static INLINE void SET_ES(unsigned long val)
{
   __asm__ __volatile__("mov %w0,%%es": /* no output */ :"r" (val));
}

static INLINE void SET_FS(unsigned long val)
{
   __asm__ __volatile__("mov %w0,%%fs": /* no output */ :"r" (val));
}

static INLINE void SET_GS(unsigned long val)
{
   __asm__ __volatile__("mov %w0,%%gs": /* no output */ :"r" (val));
}

static INLINE Selector GET_DS(void)
{
	volatile Selector _v;
	__asm__("mov %%ds,%w0":"=r" (_v):"0" (0));
	return _v;
}

static INLINE Selector GET_FS(void)
{
	volatile Selector _v;
	__asm__("mov %%fs,%w0":"=r" (_v):"0" (0));
	return _v;
}

static INLINE Selector GET_GS(void)
{
	volatile Selector _v;
	__asm__("mov %%gs,%w0":"=r" (_v):"0" (0));
	return _v;
}


/* Checked against the Intel manual and GCC --hpreg

   volatile because the content of CS can change without the compiler knowing
   it (when we use call gates). */
static INLINE Selector GET_CS(void) {
   Selector s;

   __asm__ __volatile__(
      "movw %%cs, %0"
      : "=rm" (s)
   );

   return s;
}


#define GET_WORD_FROM_FS(_fsAddr) ({ uint32 _res; __asm__ ("movl %%fs:%1,%0" : "=r" (_res) : "m" (*(unsigned*)(_fsAddr))); _res; })
#define GET_SHORT_FROM_FS(_fsAddr)({ uint16 _res;  __asm__ ("movw %%fs:%1,%0" : "=r" (_res) : "m" (*(unsigned short*)(_fsAddr))); _res; })
#define GET_BYTE_FROM_FS(_fsAddr) ({ uint8 _res; __asm__ ("movb %%fs:%1,%0" : "=r" (_res) : "m" (*(char*)(_fsAddr))); _res; })
#define SET_WORD_FS(_fsAddr,_val)	__asm__ ("movl %1,%%fs:%0" ::  "m" (*(unsigned*)(_fsAddr)) ,"r" (_val));
#define SET_SHORT_FS(_fsAddr,_val)	__asm__ ("movw %1,%%fs:%0" ::  "m" (*(unsigned short*)(_fsAddr)) ,"r" (_val));
#define SET_BYTE_FS(_fsAddr,_val)	__asm__ ("movb %1,%%fs:%0" ::  "m" (*(unsigned*)(_fsAddr)) ,"r" (_val));

static INLINE void
MEMCOPY_TO_FS(VA to, 
              char * from, 
              unsigned long n)
{
   unsigned long i =0;
   while (i+4  <=n) {
      uint32 x = *(uint32*) (from + i);
      SET_WORD_FS(to+i,x);
      i +=4;
   }
   while (i<n) {
      uint8 x = from[i];
      SET_BYTE_FS(to+i,x);
      i++;
   }
}



static INLINE void
MEMCOPY_FROM_FS(char * to, 
                VA from, 
                unsigned long n)
{
   unsigned long i =0;
   while (i+4  <=n) {
      *(uint32*) (to+i) = GET_WORD_FROM_FS(from+i);
      i +=4;
   }
   while (i<n) {
      *(uint8*)(to+i)= GET_BYTE_FROM_FS(from+i);
      i++;
   }
}

#elif _MSC_VER  /* !__GNUC__ */

#define SET_IDT(_idt)    { DTR __dt = (_idt); __asm lidt  __dt }
#define SET_GDT(_gdt)    __asm lgdt  _gdt
#define SET_TR(_tr)      __asm ltr  _tr
#define SET_LDT(_tr)     __asm lldt  _tr

#define GET_IDT(_idt)    __asm sidt _idt
#define GET_GDT(_gdt)    __asm sgdt _gdt
#define GET_TR(_tr)      __asm str  _tr
#define GET_LDT(_tr)     __asm sldt  _tr

#define GET_CR0(_reg)    __asm mov eax, cr0 __asm mov _reg, eax
#define SET_CR0(_reg)    __asm mov eax, _reg __asm mov cr0, eax
#define GET_CR2(_reg)    __asm mov eax, cr2 __asm mov _reg, eax
#define SET_CR2(_reg)    __asm mov eax, _reg __asm mov cr2, eax
#define GET_CR3(_reg)    __asm mov eax, cr3 __asm mov _reg, eax
#define SET_CR3(_reg)    __asm mov eax, _reg __asm mov cr3, eax
/*
 * MSC doesn't seem to like CR4 in __asm statements. We emit
 * the opcode for MOV EAX,CR4 = 0xf020e0 and MOV CR4,EAX = 0xf022e0
 */
#define GET_CR4(_reg) { \
 __asm _emit 0x0f __asm _emit 0x20 __asm _emit 0xe0 \
 __asm mov _reg, eax \
}
#define SET_CR4(_reg)    { \
  __asm mov eax, _reg \
  __asm _emit 0x0f __asm _emit 0x22 __asm _emit 0xe0 \
}


#define GET_DR0(_reg)    __asm mov eax,dr0 __asm mov _reg, eax
#define SET_DR0(_reg)    __asm mov eax, _reg __asm mov dr0, eax
#define GET_DR1(_reg)    __asm mov eax,dr1 __asm mov _reg, eax
#define SET_DR1(_reg)    __asm mov eax, _reg __asm mov dr1, eax
#define GET_DR2(_reg)    __asm mov eax,dr2 __asm mov _reg, eax
#define SET_DR2(_reg)    __asm mov eax, _reg __asm mov dr2, eax
#define GET_DR3(_reg)    __asm mov eax,dr3 __asm mov _reg, eax
#define SET_DR3(_reg)    __asm mov eax, _reg __asm mov dr3, eax
#define GET_DR6(_reg)    __asm mov eax,dr6 __asm mov _reg, eax
#define SET_DR6(_reg)    __asm mov eax, _reg __asm mov dr6, eax
#define GET_DR7(_reg)    __asm mov eax,dr7 __asm mov _reg, eax
#define SET_DR7(_reg)    __asm mov eax, _reg __asm mov dr7, eax


#define CLTS()           __asm clts

#define FNCLEX()         __asm fnclex

#define TLB_INVALIDATE_PAGE(_addr) {  \
	 void *_a = (_addr); \
     __asm mov eax, _a __asm invlpg [eax] \
}

#define TLB_INVALIDATE_PAGE_OFF_FS(_addr) { \
	uint32 __a = (uint32) (_addr); \
	__asm mov eax, __a _asm invlpg fs:[eax] \
}


#define ENABLE_INTERRUPTS() { __asm sti }
#define CLEAR_INTERRUPTS()  { __asm cli }

#define RAISE_INTERRUPT(_x)  {__asm int _x }
#define RETURN_FROM_INT()   {__asm iretd }
   

#define FARCALL_IMMEDIATE(SEG,OFF) { \
     FarPtr32 _addr;  \
	_addr.selector = SEG; _addr.offset = OFF; \
	__asm call FWORD PTR _addr \
}

#define SAVE_FLAGS(x) { \
     __asm pushfd __asm pop eax __asm mov x, eax \
}
   
#define RESTORE_FLAGS(x) { \
     __asm push x __asm popfd\
}



static INLINE void SET_DS(Selector val)
{
   __asm mov ax, val
   __asm mov ds, ax
}

static INLINE void SET_ES(Selector val)
{
   __asm mov ax, val 
   __asm mov es, ax
}

static INLINE void SET_FS(Selector val)
{
   __asm mov ax, val 
   __asm mov fs, ax
}

static INLINE void SET_GS(Selector val)
{
   __asm mov ax, val 
   __asm mov gs, ax
}

static INLINE Selector GET_FS(void)
{
	Selector _v;
	__asm mov _v,fs
	return _v;
}

static INLINE Selector GET_GS(void)
{
	Selector _v;
	__asm mov _v,gs
	return _v;
}


static INLINE Selector GET_CS(void)
{
	Selector _v;
	__asm mov _v,cs
	return _v;
}

#pragma warning( disable : 4035)

static INLINE uint32  GET_WORD_FROM_FS(uint32 *_addr) { 
	__asm mov eax, _addr 
    __asm mov eax, fs:[eax] 
}

static INLINE uint16  GET_SHORT_FROM_FS(uint16 *_addr) { 
	__asm mov eax, _addr 
    __asm mov ax, fs:[eax] 
}

static INLINE uint8  GET_BYTE_FROM_FS(uint8 *_addr) { 
	__asm mov eax, _addr 
     __asm mov al, fs:[eax] 
}

#pragma warning (default: 4035)

static INLINE void  SET_WORD_FS(uint32 *_addr, uint32 _val) {
    __asm mov eax, _addr 
    __asm mov ebx, _val
    __asm mov fs:[eax], ebx
}

static INLINE void  SET_SHORT_FS(uint32 *_addr, uint16 _val) {
    __asm mov eax, _addr 
    __asm mov bx, _val
    __asm mov fs:[eax], bx
}

static INLINE void  SET_BYTE_FS(uint32 *_addr, uint8 _val) {
    __asm mov eax, _addr 
    __asm mov bl, _val
    __asm mov fs:[eax], bl
}



static INLINE void
MEMCOPY_TO_FS(VA to, 
              char * from, 
              unsigned long n)
{
   unsigned long i =0;
   while (i+4  <=n) {
      uint32 x = *(uint32*) (from + i);
	  uint32 _faddr = (uint32) (to+i);
	  __asm mov eax, _faddr
	  __asm mov ebx, x
	  __asm mov fs:[eax], ebx
      i +=4;
   }
   while (i<n) {
      uint8 x = from[i];
	  uint32 _faddr = (uint32) (to+i);
	  __asm mov eax, _faddr
	  __asm mov bl, x
	  __asm mov fs:[eax], bl
      i++;
   }
}



static INLINE void
MEMCOPY_FROM_FS(char * to, 
                VA from, 
                unsigned long n)
{
   unsigned long i =0;
   while (i+4  <=n) {
      uint32 x;
	  uint32 _faddr = (uint32)(from+i);
	  __asm mov eax, _faddr
	  __asm mov ebx, fs:[eax]
	  __asm mov x,ebx
      *(uint32*)(to+i)=x;
      i +=4;
   }
   while (i<n) {
      uint8 x;
	  uint32 _faddr = (uint32) (from+i);
	  __asm mov eax, _faddr;
      __asm mov bl, fs:[eax]
	  __asm mov x, bl
      *(uint8*)(to+i)=x;
      i++;
   }
}

#endif /* !__GNUC__ && !_MSC_VER */


#ifdef __GNUC__
static INLINE int CURRENT_CPL(void)
{
   return SELECTOR_RPL(GET_CS());
}
#elif _MSC_VER
static INLINE int CURRENT_CPL(void) {
   volatile Selector _v;
   __asm mov ax, cs _asm mov _v, ax
   return SELECTOR_RPL(_v);
}
#else
#error
#endif


#ifdef __GNUC__
/* Checked against the Intel manual and GCC --hpreg

   volatile because the tsc always changes without the compiler knowing it. */
static INLINE uint64 GET_TSC(void)
{
   uint64 tim;

   __asm__ __volatile__(
      "rdtsc"
      : "=A" (tim)
   );

   return tim;
}
#elif _MSC_VER
#pragma warning( disable : 4035)
static INLINE uint64 GET_TSC(void)
{
   __asm _emit 0x0f __asm _emit 0x31
}
#pragma warning (default: 4035)
#else
#error
#endif

#ifdef __GNUC__
/* Checked against the Intel manual and GCC --hpreg

   volatile because the msr can change without the compiler knowing it
   (when we use wrmsr). */
static INLINE uint64 __GET_MSR(int cx)
{
   uint64 msr;

   __asm__ __volatile__(
      "rdmsr"
      : "=A" (msr)
      : "c" (cx)
   );

   return msr;
}
#elif _MSC_VER
#pragma warning( disable : 4035)
static INLINE uint64 __GET_MSR(int input)
{
   __asm push ecx
   __asm mov  ecx, input
   __asm _emit 0x0f __asm _emit 0x32
   __asm pop ecx
}   
#pragma warning (default: 4035)
#else
#error
#endif

/*
 * from linux: usr/include/asm/io.h
 */
#ifdef __GNUC__
#ifndef __SLOW_DOWN_IO
#ifdef SLOW_IO_BY_JUMPING
#define __SLOW_DOWN_IO __asm__ __volatile__("jmp 1f\n1:\tjmp 1f\n1:")
#else
#define __SLOW_DOWN_IO __asm__ __volatile__("outb %al,$0x80")
#endif
#endif
#elif _MSC_VER
#ifdef SLOW_IO_BY_JUMPING
#define __SLOW_DOWN_IO __asm jmp SHORT $+2 __asm  jmp SHORT $+2
#else
#define __SLOW_DOWN_IO __asm out 80h,al
#endif
#else
#error
#endif

#ifdef REALLY_SLOW_IO
#define SLOW_DOWN_IO { __SLOW_DOWN_IO; __SLOW_DOWN_IO; __SLOW_DOWN_IO; __SLOW_DOWN_IO; }
#else
#define SLOW_DOWN_IO __SLOW_DOWN_IO
#endif

#ifdef __GNUC__

/* Checked against the Intel manual and GCC --hpreg

   volatile because reading from port can modify the state of the underlying
   hardware.

   Note: The undocumented %z construct doesn't work (internal compiler error)
         with gcc-2.95.1 */
#define __GCC_IN(s, type, name) \
static INLINE type              \
name(uint16 port)               \
{                               \
   type val;                    \
                                \
   __asm__ __volatile__(        \
      "in" #s " %w1, %0"        \
      : "=a" (val)              \
      : "Nd" (port)             \
   );                           \
                                \
   return val;                  \
}

__GCC_IN(b, uint8, INB)
__GCC_IN(w, uint16, INW)
__GCC_IN(l, uint32, IN32)


/* Checked against the Intel manual and GCC --hpreg

   Note: The undocumented %z construct doesn't work (internal compiler error)
         with gcc-2.95.1 */
#define __GCC_OUT(s, s2, port, val) do { \
   __asm__(                              \
      "out" #s " %" #s2 "1, %w0"         \
      :                                  \
      : "Nd" (port), "a" (val)           \
   );                                    \
} while (0)

#define OUTB(port, val) __GCC_OUT(b, b, port, val)
#define OUTW(port, val) __GCC_OUT(w, w, port, val)
#define OUT32(port, val) __GCC_OUT(l, , port, val)


#define GET_CURRENT_EIP(_eip) __asm__ __volatile("call 0\n\tpopl %0" : "=r" (_eip): );


/* Checked against the Intel manual and GCC --hpreg

   Could perhaps be __const__ but we will never use it in loops anyway */
static INLINE void
__GET_CPUID(int ax, int *regs)
{
   __asm__(
      "cpuid"
      : "=a" (regs[0]), "=b" (regs[1]), "=c" (regs[2]), "=d" (regs[3])
      : "a" (ax)
   );
}


#elif _MSC_VER

static INLINE  uint8
INB(uint16 port) 
{ 
   uint8 val;
   __asm mov dx,port
   __asm in al,dx
   __asm mov val, al
   return val;
}
static INLINE void
OUTB(uint16 port, uint8 value)
{
  __asm mov dx, port
  __asm mov al, value
  __asm out dx,al
}

static INLINE  uint16
INW(uint16 port) 
{ 
   uint16 val;
   __asm mov dx,port
   __asm in ax,dx
   __asm mov val, ax
   return val;
}
static INLINE void 
OUTW(uint16 port, uint16 value)
{
  __asm mov dx, port
  __asm mov ax, value
  __asm out dx,ax
}

static INLINE  uint32
IN32(uint16 port) 
{ 
   uint32 val;
   __asm mov dx,port
   __asm in eax,dx
   __asm mov val, eax
   return val;
}
static INLINE void 
OUT32(uint16 port, uint32 value)
{
  __asm mov dx, port
  __asm mov eax, value
  __asm out dx,eax
}


#ifdef NEAR
#undef NEAR
#endif

#define GET_CURRENT_EIP(_eip) { \
   __asm call NEAR PTR $+5 \
   __asm pop eax \
   __asm mov _eip, eax \
}

static INLINE void
__GET_CPUID(int input, int *regs)
{
   __asm push esi
   __asm push ebx
   __asm push ecx
   __asm push edx

   __asm mov  eax, input
   __asm mov  esi, regs
   __asm _emit 0x0f __asm _emit 0xa2
   __asm mov 0x0[esi], eax
   __asm mov 0x4[esi], ebx
   __asm mov 0x8[esi], ecx
   __asm mov 0xC[esi], edx

   __asm pop edx
   __asm pop ecx
   __asm pop ebx
   __asm pop esi

}


#else
#error 
#endif


#define GET_CPUID(_ax,_bx,_cx,_dx) { \
   uint32 regs[4];                   \
   __GET_CPUID(_ax,regs);             \
   _ax = regs[0];                    \
   _bx = regs[1];                    \
   _cx = regs[2];                    \
   _dx = regs[3];                    \
}

#define START_TRACING() { \
   uint32 flags;          \
   SAVE_FLAGS(flags);     \
   flags |= EFLAGS_TF;    \
   RESTORE_FLAGS(flags);  \
}

#define STOP_TRACING() { \
   uint32 flags;          \
   SAVE_FLAGS(flags);     \
   flags &= ~EFLAGS_TF;    \
   RESTORE_FLAGS(flags);  \
}  

                   

#endif /* VM_ASM_H */
