/***********************************************************************
This file provides an implementation of the BSD functions sigblock()
and sigsetmask(), which newer systems no longer support, even though
they have SIGTSTP, the signal that ccmd code tests for being defined
before calling sigblock().

In the old days, the set of signals fit in an int mask, which is what
sigblock() and sigsetmask() handle.  On newer systems, there are more
signals than can fit in an int, so the sigset_t type was introduced to
hold an array of (possibly unsigned) int values.

The two functions defined in this file affect only the first word of
the sigset_t; higher (newer) signals are simply not supported.

For example, on Sun Solaris 10 in <sys/iso/signal_iso.h>, we find

#define	SIGWAITING 32	// reserved signal no longer used by threading code
#define	SIGLWP	33	// reserved signal no longer used by threading code
#define	SIGFREEZE 34	// special signal used by CPR
#define	SIGTHAW 35	// special signal used by CPR
#define	SIGCANCEL 36	// reserved signal for thread cancellation
#define	SIGLOST	37	// resource lost (eg, record-lock lost)
#define	SIGXRES	38	// resource control exceeded
#define	SIGJVM1	39	// reserved signal for Java Virtual Machine
#define	SIGJVM2	40	// reserved signal for Java Virtual Machine

// insert new signals here, and move _SIGRTM* appropriately
#define	_SIGRTMIN 41	// first (highest-priority) realtime signal
#define	_SIGRTMAX 48	// last (lowest-priority) realtime signal

None of these additional signals are likely to be found in older
programs, like ccmd.


[28-May-2005]
***********************************************************************/

#include <sys/types.h>
#include <signal.h>

#if defined(NEED_SIGBLOCK_SIGSETMASK)

extern int sigemptyset(sigset_t *set);
extern int sigprocmask(int how, const sigset_t *set, sigset_t *oset);

int
sigblock(int mask)
{
    sigset_t oldmask;
    sigset_t newmask;

    sigemptyset(&newmask);
    *(int*)&newmask = mask;	/* set first word of multiword mask */
    sigprocmask(SIG_BLOCK, &newmask, &oldmask);

    return (*(int*)&oldmask);	/* return first word of old multiword mask */
}

int
sigsetmask(int mask)
{
    sigset_t oldmask;
    sigset_t newmask;

    sigemptyset(&newmask);
    *(int*)&newmask = mask;	/* set first word of multiword mask */
    sigprocmask(SIG_SETMASK, &newmask, &oldmask);

    return (*(int*)&oldmask);	/* return first word of old multiword mask */
}

#endif
