/*	$NetBSD: $ */

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include "opt_interrupt.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#include <sys/kernel.h>

#include <uvm/uvm_extern.h>

#include <machine/pio.h>

#include <dev/ofw/openfirm.h>

#include <machine/autoconf.h>
#include <arch/powerpc/pic/picvar.h>

#include <arch/powerpc/mpc5200/mpc5200reg.h>
#include <arch/powerpc/mpc5200/mpc5200var.h>

static void mpc5200_enable_irq(struct pic_ops *, int, int);
static void mpc5200_disable_irq(struct pic_ops *, int);
static int  mpc5200_get_irq(struct pic_ops *, int);
static void mpc5200_ack_irq(struct pic_ops *, int);
static void mpc5200_establish_irq(struct pic_ops *, int, int, int);

struct mpc5200_ops {
	struct pic_ops pic;
	uint32_t enable_mask_critical;
	uint32_t disable_mask_main;
	uint32_t disable_mask_peripheral;
	uint32_t pending_events_critical;
	uint32_t pending_events_main;
	uint32_t pending_events_peripheral;
};

#define INT_MASK_REG_PERIPHERAL		((uint32_t)pic->pic_cookie + 0x00)
#define INT_PRI_REG_PERIPHERAL1		((uint32_t)pic->pic_cookie + 0x04)
#define INT_PRI_REG_PERIPHERAL2		((uint32_t)pic->pic_cookie + 0x08)
#define INT_PRI_REG_PERIPHERAL3		((uint32_t)pic->pic_cookie + 0x0C)
#define INT_ENABLE_REG			((uint32_t)pic->pic_cookie + 0x10)
#define INT_MASK_REG_MAIN		((uint32_t)pic->pic_cookie + 0x14)
#define INT_PRI_REG_MAIN1		((uint32_t)pic->pic_cookie + 0x18)
#define INT_PRI_REG_MAIN2		((uint32_t)pic->pic_cookie + 0x1Cf)
#define INT_STATE_REG_ENCODED		((uint32_t)pic->pic_cookie + 0x24)
#define INT_STATE_REG_CRITICAL		((uint32_t)pic->pic_cookie + 0x28)
#define INT_STATE_REG_MAIN		((uint32_t)pic->pic_cookie + 0x2c)
#define INT_STATE_REG_PERIPHERAL	((uint32_t)pic->pic_cookie + 0x30)

#define INT_ENC_CRITICAL_FLAG		0x00000400
#define INT_ENC_CRITICAL_MASK		0x00000300
#define INT_ENC_MAIN_FLAG		0x00200000
#define INT_ENC_MAIN_MASK		0x001F0000
#define INT_ENC_PERIPHERAL_FLAG		0x20000000
#define INT_ENC_PERIPHERAL_MASK		0x1F000000

#define INT_LEVEL_MASK_CRITICAL		(0x0F000000)
#define INT_LEVEL_MASK_MAIN		(0x0001FFFF)
#define INT_LEVEL_MASK_PERIPHERAL	(0x00FFFFF9)

struct pic_ops *
setup_mpc5200pic(void *addr)
{
	struct mpc5200_ops *mpc5200_pic;
	struct pic_ops *pic;

	mpc5200_pic = malloc(sizeof(struct mpc5200_ops), M_DEVBUF, M_NOWAIT);
	KASSERT(mpc5200_pic != NULL);
	pic = &mpc5200_pic->pic;

	pic->pic_numintrs = 96;
	pic->pic_cookie = addr;
	pic->pic_enable_irq = mpc5200_enable_irq;
	pic->pic_reenable_irq = mpc5200_enable_irq;
	pic->pic_disable_irq = mpc5200_disable_irq;
	pic->pic_get_irq = mpc5200_get_irq;
	pic->pic_ack_irq = mpc5200_ack_irq;
	pic->pic_establish_irq = mpc5200_establish_irq;
	pic->pic_finish_setup = NULL;
	strcpy(pic->pic_name, "mpc5200");

	mpc5200_pic->pending_events_critical = 0;
	mpc5200_pic->enable_mask_critical = 0x00001001;	/* MEE & CEb */
	mpc5200_pic->pending_events_main = 0;
	mpc5200_pic->disable_mask_main = 0x0001efff; /* LO_int */
	mpc5200_pic->pending_events_peripheral = 0;
	mpc5200_pic->disable_mask_peripheral = 0xffffff00;
	pic_add(pic);

	out32(INT_MASK_REG_MAIN, mpc5200_pic->disable_mask_main);
	out32(INT_MASK_REG_PERIPHERAL, mpc5200_pic->disable_mask_peripheral);

	out32(INT_ENABLE_REG, 0x0fc01001);

	out32(INT_PRI_REG_PERIPHERAL1, 0x0);
	out32(INT_PRI_REG_PERIPHERAL2, 0x0);
	out32(INT_PRI_REG_PERIPHERAL3, 0x0);
	out32(INT_PRI_REG_MAIN1, 0x0);
	out32(INT_PRI_REG_MAIN2, 0x0);

	return pic;
}

static void
mpc5200_enable_irq(struct pic_ops *pic, int irq, int type)
{
	struct mpc5200_ops *mpc5200 = (struct mpc5200_ops *)pic;
	uint32_t mask;

	if (irq & 0x20) {
		if ((irq & 0x1f) > 16) return;
		mask = 1 << (16 - (irq & 0x1f));
		mpc5200->disable_mask_main &= ~mask;
		out32(INT_MASK_REG_MAIN,
			mpc5200->disable_mask_main);
	} else if (irq & 0x40) {
		if ((irq & 0x1f) > 23) return;
		mask = 1 << (31 - (irq & 0x1f));
		mpc5200->disable_mask_peripheral &= ~mask;
		out32(INT_MASK_REG_PERIPHERAL,
		      mpc5200->disable_mask_peripheral);
	} else {
		if ((irq & 0x1f) > 3) return;
		mask = (1 << (11 - irq)) | (type << (22 - (irq * 2)));
		mpc5200->enable_mask_critical |= mask;
		out32(INT_ENABLE_REG, mpc5200->enable_mask_critical);
	}
}

static void
mpc5200_disable_irq(struct pic_ops *pic, int irq)
{
	struct mpc5200_ops *mpc5200 = (struct mpc5200_ops *)pic;
	uint32_t mask;

	if (irq == 0x24) return; /* LO_int */

	if (irq & 0x20) {
		if ((irq & 0x1f) > 16) return;
		mask = 1 << (16 - (irq & 0x1f));
		mpc5200->disable_mask_main |= mask;
		out32(INT_MASK_REG_MAIN, mpc5200->disable_mask_main);
	} else if (irq & 0x40) {
		if ((irq & 0x1f) > 23) return;
		mask = 1 << (31 - (irq & 0x1f));
		mpc5200->disable_mask_peripheral |= mask;
		out32(INT_MASK_REG_PERIPHERAL,
		      mpc5200->disable_mask_peripheral);
	} else {
		if ((irq & 0x1f) > 3) return;
		mask = 1 << (11 - (irq & 0x1f));
		mpc5200->enable_mask_critical &= ~mask;
		out32(INT_ENABLE_REG, mpc5200->enable_mask_critical);
	}
}

static int
mpc5200_get_irq(struct pic_ops *pic, int mode)
{
	int enc, irq;

	enc = in32(INT_STATE_REG_ENCODED);
	if (enc & INT_ENC_CRITICAL_FLAG) {
		irq = (enc & INT_ENC_CRITICAL_MASK) >> 8;
		if (irq == 2) goto peripheral;
		return irq;
	} else if (enc & INT_ENC_MAIN_FLAG) {
		irq = (enc & INT_ENC_MAIN_MASK) >> 16;
		if (irq == 4) goto peripheral;
		return irq + 32;
	} else if (enc & INT_ENC_PERIPHERAL_FLAG) {
peripheral:
		irq = (enc & INT_ENC_PERIPHERAL_MASK) >> 24;
		return irq + 64;
	}

	/* we should never get here */
	return 255;
}

static void
mpc5200_ack_irq(struct pic_ops *pic, int irq)
{
}

static void
mpc5200_establish_irq(struct pic_ops *pic, int irq, int type, int maxlevel)
{
	aprint_normal("mpc5200_establish_irq: irq %d type %d pri %d\n",
		      irq, type, maxlevel);

}
