/*	$NetBSD: $	*/

/*
 * Copyright (c) 2007 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgement:
 *        This product includes software developed by the NetBSD
 *        Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND
 * CONTRIBUTORS ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (C) 1998 Scott Reynolds
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 1995 Allen Briggs.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Allen Briggs.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * This driver implements a very generic and simple framebuffer for mac68k
 * cards attached to the obio bus.  It probably fails to deal with colors
 * and other stuff in most of the cases, but it does the necessary work to
 * provide a minimum, full-featured wscons terminal.
 *
 * Ideally we'd have an independent driver for each card (as, e.g., the
 * valkyriefb) that properly deals with the hardware (even if it is also
 * backed by genfb).  But as we are not there yet, we provide this simple
 * driver.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/ioctl.h>
#include <sys/kernel.h>
#include <sys/systm.h>

#include <uvm/uvm_extern.h>

#include <machine/bus.h>
#include <machine/cpu.h>
#include <machine/viareg.h>
#include <machine/video.h>

#include <mac68k/obio/obiovar.h>
#include <dev/wsfb/genfbvar.h>

#include "opt_wsfb.h"
#include "opt_genfb.h"

/* --------------------------------------------------------------------- */

/*
 * Generic definitions for genfb attached at obio.
 */

/* XXX Has to die when intvid is removed. */
bool genfb_at_obio = false;

struct colormap_softc {
	bus_space_tag_t sc_tag;
	bus_space_handle_t sc_handle;
	struct genfb_colormap_callback sc_cb;

	/* Used by dafb. */
	int sc_rgb_cache[256 * 3];
	size_t sc_last;
};

struct obio_genfb_softc {
	struct genfb_softc sc_gen;
	bus_addr_t sc_basepa;
	bus_addr_t sc_fbofs;
	bus_space_handle_t sc_handle;
	bus_space_handle_t sc_regh;
	bus_space_tag_t sc_tag;

	struct colormap_softc sc_cmap;
};

static int	obio_genfb_match(device_t, cfdata_t, void *);
static void	obio_genfb_attach(device_t, device_t, void *);
static int	obio_genfb_ioctl(void *, void *, unsigned long, void *, int,
		    struct lwp *);
static paddr_t	obio_genfb_mmap(void *, void *, off_t, int);

CFATTACH_DECL_NEW(genfb_obio, sizeof(struct obio_genfb_softc),
    obio_genfb_match, obio_genfb_attach, NULL, NULL);

/* --------------------------------------------------------------------- */

/*
 * Prototypes for card-specific stuff.
 */

static bool	civic_match(struct obio_attach_args *);
static size_t	civic_attach(struct obio_genfb_softc *,
		    struct obio_attach_args *);

static bool	dafb_match(struct obio_attach_args *);
static size_t	dafb_attach(struct obio_genfb_softc *,
		    struct obio_attach_args *);
static void	dafb_putpalreg(void *, int, int, int, int);

static bool	rbv_match(struct obio_attach_args *);
static size_t	rbv_attach(struct obio_genfb_softc *,
		    struct obio_attach_args *);

static bool	valkyrie_match(struct obio_attach_args *);
static size_t	valkyrie_attach(struct obio_genfb_softc *,
		    struct obio_attach_args *);
static void	valkyrie_putpalreg(void *, int, int, int, int);

/* --------------------------------------------------------------------- */

/*
 * Card-independent code.
 */

static int
obio_genfb_match(device_t parent, cfdata_t match, void *aux)
{
	struct obio_attach_args *oa = (struct obio_attach_args *)aux;
	bool found;

	switch (current_mac_model->class) {
	case MACH_CLASSQ2:
		if (current_mac_model->machineid != MACH_MACLC575) {
			found = valkyrie_match(oa);
		} else {
			/*
			 * Note: the only system in this class that does
			 * not have the Valkyrie chip -- at least, that we
			 * know of -- is the Performa/LC 57x series.
			 * This system has a version of the DAFB controller
			 * instead.
			 *
			 * If this assumption proves false, we'll have to
			 * be more intelligent here.
			 */
			found = dafb_match(oa);
		}
		break;

	case MACH_CLASSQ:
		/*
		 * Assume DAFB for all of these, unless we can't
		 * access the memory.
		 */
		found = dafb_match(oa);
		break;

	case MACH_CLASSAV:
		found = civic_match(oa);
		break;

	case MACH_CLASSIIci:
	case MACH_CLASSIIsi:
		found = rbv_match(oa);
		break;

	default:
		found = (mac68k_video.mv_len != 0);
		break;
	}

	if (found)
		genfb_at_obio = true;

	return found ? 1 : 0;
}

static void
obio_genfb_attach(device_t parent, device_t self, void *aux)
{
	struct obio_attach_args *oa = (struct obio_attach_args *)aux;
	struct obio_genfb_softc *sc = device_private(self);
	struct colormap_softc *cmap = &sc->sc_cmap;
	struct genfb_ops ops;
	size_t length;

	cmap->sc_cb.gcc_cookie = NULL;
	cmap->sc_cb.gcc_set_mapreg = NULL;

	sc->sc_gen.sc_dev = self;

	switch (current_mac_model->class) {
	case MACH_CLASSQ2:
		if (current_mac_model->machineid != MACH_MACLC575)
			length = valkyrie_attach(sc, oa);
		else
			length = dafb_attach(sc, oa);
		break;

	case MACH_CLASSQ:
		length = dafb_attach(sc, oa);
		break;

	case MACH_CLASSAV:
		length = civic_attach(sc, oa);
		break;

	case MACH_CLASSIIci:
	case MACH_CLASSIIsi:
		length = rbv_attach(sc, oa);
		break;

	default:
		sc->sc_basepa = m68k_trunc_page(mac68k_video.mv_phys);
		sc->sc_fbofs = m68k_page_offset(mac68k_video.mv_phys);
		length = mac68k_video.mv_len + sc->sc_fbofs;

		aprint_normal(": On-board video\n");
		break;
	}

	if (length == 0)
		return;

	if (bus_space_map(sc->sc_tag, sc->sc_basepa, length, 0,
	    &sc->sc_handle)) {
		aprint_normal_dev(sc->sc_gen.sc_dev,
				"failed to map video RAM\n");
		return;
	}

	if (sc->sc_basepa <= mac68k_video.mv_phys &&
	    mac68k_video.mv_phys < (sc->sc_basepa + length)) {
		/* XXX Hack */
		mac68k_video.mv_kvaddr = sc->sc_handle.base + sc->sc_fbofs;
	}

	/* XXX This has to go in favour of the code below... but uh-huh!
	 * If I remove it the machine hangs! */
	{
	prop_dictionary_t dict;
	dict = device_properties(sc->sc_gen.sc_dev);
	prop_dictionary_set_uint32(dict, "width", mac68k_video.mv_width);
	prop_dictionary_set_uint32(dict, "height", mac68k_video.mv_height);
	prop_dictionary_set_uint32(dict, "depth", mac68k_video.mv_depth);
	prop_dictionary_set_uint32(dict, "address", mac68k_video.mv_kvaddr);
	prop_dictionary_set_uint32(dict, "linebytes", mac68k_video.mv_stride);
	}

	genfb_init(&sc->sc_gen);

	sc->sc_gen.sc_fbaddr = (void *)mac68k_video.mv_kvaddr;
	sc->sc_gen.sc_fboffset = mac68k_video.mv_phys;
	sc->sc_gen.sc_width = mac68k_video.mv_width;
	sc->sc_gen.sc_height = mac68k_video.mv_height;
	sc->sc_gen.sc_depth = mac68k_video.mv_depth;
	sc->sc_gen.sc_stride = mac68k_video.mv_stride;
	sc->sc_gen.sc_fbsize = mac68k_video.mv_stride * sc->sc_gen.sc_height;

	if (cmap->sc_cb.gcc_cookie != NULL) {
		KASSERT(cmap->sc_cb.gcc_set_mapreg != NULL);
		sc->sc_gen.sc_cmcb = &cmap->sc_cb;
	} else
		KASSERT(cmap->sc_cb.gcc_set_mapreg == NULL);

	ops.genfb_ioctl = obio_genfb_ioctl;
	ops.genfb_mmap = obio_genfb_mmap;

	genfb_attach(&sc->sc_gen, &ops);
}

static int
obio_genfb_ioctl(void *v, void *vs, unsigned long cmd, void *data, int flag,
    struct lwp *l)
{

	switch (cmd) {
	case WSDISPLAYIO_GTYPE:
		*(unsigned int *)data = WSDISPLAY_TYPE_UNKNOWN; /* XXX */
		return 0;
	}

	return EPASSTHROUGH;
}

static paddr_t
obio_genfb_mmap(void *v, void *vs, off_t offset, int prot)
{
	struct obio_genfb_softc *sc = v;
	paddr_t pa;

	aprint_normal_dev(sc->sc_gen.sc_dev, "mmap %08x limit %08x\n",
			(uint32_t)offset, (uint32_t)sc->sc_gen.sc_fbsize);
	if ((offset >= 0) && (offset < sc->sc_gen.sc_fbsize)) {
		(void) pmap_extract(pmap_kernel(), (vaddr_t)sc->sc_handle.base,
		    &pa);
		return m68k_btop(pa + offset);
	}
	return -1;
}

/* --------------------------------------------------------------------- */

/*
 * CIVIC-specific definitions and routines.
 */

#define	CIVIC_BASE		0x50100000
#define CIVIC_CONTROL_BASE	0x50036000

static bool
civic_match(struct obio_attach_args *oa)
{
	bus_space_handle_t bsh;

	if (bus_space_map(oa->oa_tag, CIVIC_CONTROL_BASE, 0x1000, 0, &bsh))
		return 0;

	/* Disable interrupts */
	bus_space_write_1(oa->oa_tag, bsh, 0x120, 0);

	bus_space_unmap(oa->oa_tag, bsh, 0x1000);

	return 1;
}

static size_t
civic_attach(struct obio_genfb_softc *sc, struct obio_attach_args *oa)
{
	size_t length;

	sc->sc_basepa = CIVIC_BASE;
	length = 0x00200000;		/* 2MB */
	if (mac68k_video.mv_phys >= sc->sc_basepa &&
	    mac68k_video.mv_phys < (sc->sc_basepa + length)) {
		sc->sc_fbofs = mac68k_video.mv_phys - sc->sc_basepa;
	} else {
		sc->sc_basepa = m68k_trunc_page(mac68k_video.mv_phys);
		sc->sc_fbofs = m68k_page_offset(mac68k_video.mv_phys);
		length = mac68k_video.mv_len + sc->sc_fbofs;
	}

	aprint_normal(": CIVIC video subsystem\n");

	return length;
}

/* --------------------------------------------------------------------- */

/*
 * DAFB-specific definitions and routines.
 */

#define	DAFB_BASE		0xf9000000
#define DAFB_CMAP_BASE		0xf9800200
#define DAFB_CMAP_RESET		0
#define DAFB_CMAP_RGB		19
#define DAFB_CMAP_SIZE		20
#define DAFB_CONTROL_BASE	0xf9800000

static bool
dafb_match(struct obio_attach_args *oa)
{
	bus_space_handle_t bsh;

	if (bus_space_map(oa->oa_tag, DAFB_CONTROL_BASE, 0x20, 0, &bsh))
		return false;

	if (mac68k_bus_space_probe(oa->oa_tag, bsh, 0x1c, 4) == 0) {
		bus_space_unmap(oa->oa_tag, bsh, 0x20);
		return false;
	}

	bus_space_unmap(oa->oa_tag, bsh, 0x20);

	if (bus_space_map(oa->oa_tag, DAFB_CONTROL_BASE + 0x100, 0x20,
	    0, &bsh))
		return false;

	if (mac68k_bus_space_probe(oa->oa_tag, bsh, 0x04, 4) == 0) {
		bus_space_unmap(oa->oa_tag, bsh, 0x20);
		return false;
	}

	/* Disable interrupts */
	bus_space_write_4(oa->oa_tag, bsh, 0x04, 0);

	/* Clear any interrupts */
	bus_space_write_4(oa->oa_tag, bsh, 0x0C, 0);
	bus_space_write_4(oa->oa_tag, bsh, 0x10, 0);
	bus_space_write_4(oa->oa_tag, bsh, 0x14, 0);

	bus_space_unmap(oa->oa_tag, bsh, 0x20);

	return true;
}

static size_t
dafb_attach(struct obio_genfb_softc *sc, struct obio_attach_args *oa)
{
	uint32_t vbase1, vbase2;
	size_t length;
	struct colormap_softc *cmap = &sc->sc_cmap;

	sc->sc_tag = oa->oa_tag;
	if (bus_space_map(sc->sc_tag, DAFB_CONTROL_BASE, 0x20, 0,
	    &sc->sc_regh)) {
		aprint_normal(": failed to map DAFB register space\n");
		return 0;
	}

	sc->sc_basepa = DAFB_BASE;
	length = 0x00100000;		/* 1MB */

	/* Compute the current frame buffer offset */
	vbase1 = bus_space_read_4(sc->sc_tag, sc->sc_regh, 0x0) & 0xfff;
#if 1
	/*
	 * XXX The following exists because the DAFB v7 in these
	 * systems doesn't return reasonable values to use for fbofs.
	 * Ken'ichi Ishizaka gets credit for this hack.  (sar 19990426)
	 * (Does this get us the correct result for _all_ DAFB-
	 * equipped systems and monitor combinations?  It seems
	 * possible, if not likely...)
	 */
	switch (current_mac_model->machineid) {
	case MACH_MACLC475:
	case MACH_MACLC475_33:
	case MACH_MACLC575:
	case MACH_MACQ605:
	case MACH_MACQ605_33:
		vbase1 &= 0x3f;
		break;
	}
#endif
	vbase2 = bus_space_read_4(sc->sc_tag, sc->sc_regh, 0x4) & 0xf;
	sc->sc_fbofs = (vbase1 << 9) | (vbase2 << 5);

	cmap->sc_tag = oa->oa_tag;
	if (bus_space_map(cmap->sc_tag, DAFB_CMAP_BASE,
	    DAFB_CMAP_SIZE, 0, &cmap->sc_handle) == 0) {
		cmap->sc_cb.gcc_cookie = cmap;
		cmap->sc_cb.gcc_set_mapreg = dafb_putpalreg;
		cmap->sc_last = -1;
		memset(cmap->sc_rgb_cache, 0, 256 * 3 * sizeof(int));
	}

	aprint_normal(": DAFB video subsystem, monitor sense %x\n",
	    bus_space_read_4(sc->sc_tag, sc->sc_regh, 0x1c) & 0x7);

	bus_space_unmap(sc->sc_tag, sc->sc_regh, 0x20);

	return length;
}

static void
dafb_putpalreg(void *v, int idx, int r, int g, int b)
{
	struct colormap_softc *cmap = (struct colormap_softc *)v;
	bus_space_tag_t t = cmap->sc_tag;
	bus_space_handle_t h = cmap->sc_handle;

	if (idx != cmap->sc_last + 1) {
		size_t i;

		bus_space_write_4(t, h, DAFB_CMAP_RESET, 0);
		bus_space_barrier(t, h, 0, DAFB_CMAP_SIZE,
		    BUS_SPACE_BARRIER_WRITE);

		for (i = 0; i < idx; i++) {
			int *c = &cmap->sc_rgb_cache[i * 3];
			bus_space_write_1(t, h, DAFB_CMAP_RGB, *c);
			bus_space_barrier(t, h, 0, DAFB_CMAP_SIZE,
			    BUS_SPACE_BARRIER_WRITE);
			bus_space_write_1(t, h, DAFB_CMAP_RGB, *(c + 1));
			bus_space_barrier(t, h, 0, DAFB_CMAP_SIZE,
			    BUS_SPACE_BARRIER_WRITE);
			bus_space_write_1(t, h, DAFB_CMAP_RGB, *(c + 2));
			bus_space_barrier(t, h, 0, DAFB_CMAP_SIZE,
			    BUS_SPACE_BARRIER_WRITE);
		}
	}

	bus_space_write_1(t, h, DAFB_CMAP_RGB, r);
	bus_space_barrier(t, h, 0, DAFB_CMAP_SIZE, BUS_SPACE_BARRIER_WRITE);
	bus_space_write_1(t, h, DAFB_CMAP_RGB, g);
	bus_space_barrier(t, h, 0, DAFB_CMAP_SIZE, BUS_SPACE_BARRIER_WRITE);
	bus_space_write_1(t, h, DAFB_CMAP_RGB, b);

	cmap->sc_rgb_cache[idx * 3    ] = r;
	cmap->sc_rgb_cache[idx * 3 + 1] = g;
	cmap->sc_rgb_cache[idx * 3 + 2] = b;

	cmap->sc_last = idx;
}

/* --------------------------------------------------------------------- */

/*
 * RBV-specific definitions and routines.
 */

static bool
rbv_match(struct obio_attach_args *oa)
{

	if (mac68k_video.mv_len == 0 ||
	    (via2_reg(rMonitor) & RBVMonitorMask) == RBVMonIDNone)
		return false;

	return true;
}

static size_t
rbv_attach(struct obio_genfb_softc *sc, struct obio_attach_args *oa)
{
	size_t length;

	sc->sc_basepa = m68k_trunc_page(mac68k_video.mv_phys);
	sc->sc_fbofs = m68k_page_offset(mac68k_video.mv_phys);
	length = mac68k_video.mv_len + sc->sc_fbofs;

	aprint_normal(": RBV video subsystem, ");
	switch (via2_reg(rMonitor) & RBVMonitorMask) {
	case RBVMonIDBWP:
		aprint_normal("15\" monochrome portrait");
		break;
	case RBVMonIDRGB12:
		aprint_normal("12\" color");
		break;
	case RBVMonIDRGB15:
		aprint_normal("15\" color");
		break;
	case RBVMonIDStd:
		aprint_normal("Macintosh II");
		break;
	default:
		aprint_normal("unrecognized");
		break;
	}
	aprint_normal(" display\n");

	return length;
}

/* --------------------------------------------------------------------- */

/*
 * Valkyrie-specific definitions and routines.
 */

#define	VALKYRIE_BASE		0xf9000000
#define VALKYRIE_CMAP_BASE	0x50f24000
#define VALKYRIE_CMAP_IDX	0
#define VALKYRIE_CMAP_RGB	4
#define VALKYRIE_CMAP_SIZE	5
#define VALKYRIE_CONTROL_BASE	0x50f2a000

static bool
valkyrie_match(struct obio_attach_args *oa)
{
	bus_space_handle_t bsh;

	if (bus_space_map(oa->oa_tag, VALKYRIE_CONTROL_BASE, 0x40, 0, &bsh))
		return false;

	/* Disable interrupts */
	bus_space_write_1(oa->oa_tag, bsh, 0x18, 0x1);

	bus_space_unmap(oa->oa_tag, bsh, 0x40);

	return true;
}

static size_t
valkyrie_attach(struct obio_genfb_softc *sc, struct obio_attach_args *oa)
{
	size_t length;
	struct colormap_softc *cmap = &sc->sc_cmap;

	sc->sc_basepa = VALKYRIE_BASE;
	length = 0x00100000;		/* 1MB */

	if (sc->sc_basepa <= mac68k_video.mv_phys &&
	    mac68k_video.mv_phys < (sc->sc_basepa + length)) {
		sc->sc_fbofs = mac68k_video.mv_phys - sc->sc_basepa;
	} else {
		sc->sc_basepa = m68k_trunc_page(mac68k_video.mv_phys);
		sc->sc_fbofs = m68k_page_offset(mac68k_video.mv_phys);
		length = mac68k_video.mv_len + sc->sc_fbofs;
	}

	cmap->sc_tag = oa->oa_tag;
	if (bus_space_map(cmap->sc_tag, VALKYRIE_CMAP_BASE,
	    VALKYRIE_CMAP_SIZE, 0, &cmap->sc_handle) == 0) {
		cmap->sc_cb.gcc_cookie = cmap;
		cmap->sc_cb.gcc_set_mapreg = valkyrie_putpalreg;
	}

	aprint_normal(": Valkyrie video subsystem\n");

	return length;
}

static void
valkyrie_putpalreg(void *v, int idx, int r, int g, int b)
{
	struct colormap_softc *cmap = (struct colormap_softc *)v;
	bus_space_tag_t t = cmap->sc_tag;
	bus_space_handle_t h = cmap->sc_handle;

	bus_space_write_1(t, h, VALKYRIE_CMAP_IDX, idx);
	bus_space_barrier(t, h, 0, VALKYRIE_CMAP_SIZE,
	    BUS_SPACE_BARRIER_WRITE);
	bus_space_write_1(t, h, VALKYRIE_CMAP_RGB, r);
	bus_space_write_1(t, h, VALKYRIE_CMAP_RGB, g);
	bus_space_write_1(t, h, VALKYRIE_CMAP_RGB, b);
}
