/*	$NetBSD: $ */

/*-
 * Copyright (c) 2006 Michael Lorenz
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include <sys/param.h>
#include <sys/types.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/pool.h>
#include <sys/queue.h>

#include <machine/cpu.h>
#include <machine/iopreg.h>

#include <dev/adb/adbvar.h>
#include "nadb.h"

#include "adbdebug.h"

static int adbiop_match(device_t, cfdata_t, void *);
static void adbiop_attach(device_t, device_t, void *);
static void adbiop_autopoll(void *, int);

struct adbiop_softc {
	device_t sc_dev;
	struct adb_bus_accessops sc_adbops;

	int sc_autopoll;
	/* IOP */
	void *sc_iop;
	/* ADB */
	void (*sc_adb_handler)(void *, int, uint8_t *);
	void *sc_adb_cookie;
};

CFATTACH_DECL_NEW(adbiop, sizeof(struct adbiop_softc),
    adbiop_match, adbiop_attach, NULL, NULL);

static void adbiop_adb_poll(void *);

static void adbiop_adb_handler(void *, struct iop_msg *);

static struct adbiop_softc *adbiop0 = NULL;

/* ADB bus attachment stuff */
static 	int adbiop_adb_send(void *, int, int, int, uint8_t *);
static	int adbiop_adb_set_handler(void *, void (*)(void *, int, uint8_t *), void *);

/*static int  adbiop_print(void *, const char *);*/

static int
adbiop_match(device_t parent, cfdata_t cf, void *aux)
{
	if (adbiop0 != NULL)
		return (0);

	switch (mac68k_machine.machineid) {
	case MACH_MACIIFX:		/* IIfx */
	case MACH_MACQ900:		/* Quadra 900 */
	case MACH_MACQ950:		/* Quadra 950 */
		
		return 1;
	}

	return 0;
}

static void
adbiop_attach(device_t parent, device_t self, void *aux)
{
	struct adbiop_softc *sc = device_private(self);

	sc->sc_dev = self;
	sc->sc_autopoll = 0;

	sc->sc_iop = device_private(parent);

	if (adbiop0 == NULL)
		adbiop0 = sc;

	aprint_normal("\n");
	aprint_normal_dev(sc->sc_dev, "initializing ADB\n");

	sc->sc_adbops.cookie = sc;
	sc->sc_adbops.send = adbiop_adb_send;
	sc->sc_adbops.poll = adbiop_adb_poll;
	sc->sc_adbops.autopoll = adbiop_autopoll;
	sc->sc_adbops.set_handler = adbiop_adb_set_handler;
	iop_register_listener(sc->sc_iop, IOP_CHAN_ADB,
			      adbiop_adb_handler, sc);
#if NNADB > 0
	config_found_ia(self, "adb_bus", &sc->sc_adbops, nadb_print);
#endif
}

static void
adbiop_adb_poll(void *cookie)
{
#if 0
	struct adbiop_softc *sc = cookie;

	adbiop_intr(sc);
#endif
	aprint_normal("adbiop_adb_poll\n");
}

static void
adbiop_adb_handler(void *iop, struct iop_msg *msg)
{
	struct adbiop_softc *sc;
	uint8_t cmd[32];
#if 1
	int s;


	s = splhigh();		/* can't be too careful - might be called */
				/* from a routine, NOT an interrupt */
#endif
	if (msg->status == IOP_MSGSTAT_RECEIVED) {
		sc = (struct adbiop_softc *) msg->user_data;

		if (sc->sc_adb_handler != NULL)
			sc->sc_adb_handler(sc->sc_adb_cookie, msg->msg[1] + 1,
					   &msg->msg[1]);

		if (msg->msg[0] & IOP_ADB_FL_TIMEOUT) {
			msg->msg[0] = IOP_ADB_FL_TIMEOUT | IOP_ADB_FL_AUTOPOLL;
			msg->msg[1] = 0;
			msg->msg[2] = 0;
		} else {
			msg->msg[0] = IOP_ADB_FL_AUTOPOLL;
		}
	
#if 1
		/*
		 * the IOP will turn off autopolling after each LISTEN so we
		 * need to re-enable it here whenever we receive an ACK for a
		 * LISTEN command
		 */
		if ((msg->msg[2] & 0x0c) == 0x08) {
			cmd[0] = IOP_ADB_FL_POLL_UPDATE | IOP_ADB_FL_AUTOPOLL;
			cmd[1] = 2;
			cmd[2] = 0;
			cmd[3] = (sc->sc_autopoll >> 8) & 0xff;
			cmd[4] = sc->sc_autopoll & 0xff;
			iop_send_msg(sc->sc_iop, IOP_CHAN_ADB, cmd, 5);
		}
#endif
	}
#if 1
	splx(s);
#endif
}

static void
adbiop_autopoll(void *cookie, int flag)
{
	struct adbiop_softc *sc = cookie;
	/* magical incantation to re-enable autopolling */
	uint8_t cmd[5];

	if (sc->sc_autopoll == flag)
		return;

	cmd[0] = IOP_ADB_FL_POLL_UPDATE | IOP_ADB_FL_AUTOPOLL;
	cmd[1] = 2;
	cmd[2] = 0;
	cmd[3] = (flag >> 8) & 0xff;
	cmd[4] = flag & 0xff;
	iop_send_msg(sc->sc_iop, IOP_CHAN_ADB, cmd, 5);
	sc->sc_autopoll = flag & 0xffff;
}

static int
adbiop_adb_send(void *cookie, int poll, int command, int len, uint8_t *data)
{
	struct adbiop_softc *sc = cookie;
	int i;
	uint8_t packet[32];

	/* construct an ADB command packet and send it */
	packet[0] = IOP_ADB_FL_EXPLICIT;
	packet[1] = len;
	packet[2] = command;
	for (i = 0; i < len; i++)
		packet[i + 3] = data[i];
	iop_send_msg(sc->sc_iop, IOP_CHAN_ADB, packet, len + 3);

	return 0;
}

static int
adbiop_adb_set_handler(void *cookie, void (*handler)(void *, int, uint8_t *),
    void *hcookie)
{
	struct adbiop_softc *sc = cookie;

	/* register a callback for incoming ADB messages */
	sc->sc_adb_handler = handler;
	sc->sc_adb_cookie = hcookie;
	return 0;
}

#if 0
static int
adbiop_print(void *aux, const char *what)
{

	return 0;
}
#endif
