/* $NetBSD$ */

/*-
 * Copyright (c) 2015 Nathanial Sloss <nathanialsloss@yahoo.com.au>
 * All rights reserved.
 *
 *		This software is dedicated to the memory of -
 *	   Baron James Anlezark (Barry) - 1 Jan 1949 - 13 May 2012.
 *
 *		Barry was a man who loved his music.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*-
 * Copyright (c) 2006 Itronix Inc.
 * All rights reserved.
 *
 * Written by Iain Hibbert for Itronix Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of Itronix Inc. may not be used to endorse
 *    or promote products derived from this software without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ITRONIX INC. ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ITRONIX INC. BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* This was based upon bthset.c */

#include <sys/cdefs.h>
__COPYRIGHT("@(#) Copyright (c) 2006 Itronix, Inc.  All rights reserved.");
__RCSID("$NetBSD");

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>

#include <assert.h>
#include <bluetooth.h>
#include <err.h>
#include <event.h>
#include <fcntl.h>
#include <sdp.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "avdtp_signal.h"
#include "sbc_encode.h"

#define A2DP_SINK	0x110b

struct l2cap_info {
	bdaddr_t laddr;
	bdaddr_t raddr;
};

__dead static void usage(void);

static int init_server(struct l2cap_info *);
static void client_query(void);
static void exit_func(void);

int sc;			/* avdtp streaming channel */
int hc;			/* avdtp control/command channel */
struct event_base *base;
static struct event interrupt_ev;		/* audio event */

struct l2cap_info	info;
static int 		verbose;	/* copy to stdout */
static int		channel_mode = MODE_STEREO;
static int		alloc_method = ALLOC_LOUDNESS;
static int		frequency = FREQ_44_1K;
static int		bitpool = 0;
static int		bands = BANDS_8;
static int		blocks = BLOCKS_16;
uint16_t mtu, l2cap_psm;
uint16_t userset_mtu = 0, service_class = A2DP_SINK;
static const char* service_type = "A2DP";
struct avdtp_sepInfo mySepInfo;

char *files2open[255];
char *filebuf;
FILE *currentAudioStream;
static int numfiles = 0;
static int startFileInd = 0;
int audfile;

#define log_err(x, ...)		if (verbose) { printf (x "\n",\
						__VA_ARGS__); }
#define log_info(x, ...)	if (verbose) { printf (x "\n",\
						__VA_ARGS__); }

int
main(int ac, char *av[])
{
	int	i, ch, freqnum, blocksnum, tmpbitpool;
	bdaddr_copy(&info.laddr, BDADDR_ANY);

	sc = hc = -1;
	verbose = 0;

	while ((ch = getopt(ac, av, "a:d:r:b:M:B:42jmsv")) != EOF) {
		switch (ch) {
		case 'a':
			if (!bt_aton(optarg, &info.raddr)) {
				struct hostent  *he = NULL;

				if ((he = bt_gethostbyname(optarg)) == NULL)
					usage();

				bdaddr_copy(&info.raddr, (bdaddr_t *)he->h_addr);
			}
			break;
		case 'd': /* local device address */
			if (!bt_devaddr(optarg, &info.laddr))
				usage();
			break;
		case 'B': /* Max bitpool value */
			bitpool = atoi(optarg);
			break;
		case 'M':
			userset_mtu = atoi(optarg);
			break;
		case 'r':
			freqnum = atoi(optarg);
			switch (freqnum) {
			case 16000:
				frequency = FREQ_16K;
				break;
			case 32000:
				frequency = FREQ_32K;
				break;
			case 44100:
				frequency = FREQ_44_1K;
				break;
			case 48000:
				frequency = FREQ_48K;
				break;
			default:
				usage();
			}
			break;
		case 'b':
			blocksnum = atoi(optarg);
			switch (blocksnum) {
			case 4:
				blocks = BLOCKS_4;
				break;
			case 8:
				blocks = BLOCKS_8;
				break;
			case 12:
				blocks = BLOCKS_12;
				break;
			case 16:
				blocks = BLOCKS_16;
				break;
			default:
				usage();
			}
			break;
		case '4':
			bands = BANDS_4;
			break;
		case '2':
			channel_mode = MODE_DUAL;
			break;
		case 'j':
			channel_mode = MODE_JOINT;
			break;
		case 'm':
			channel_mode = MODE_MONO;
			break;
		case 's':
			channel_mode = MODE_STEREO;
			break;
		case 'v':
			verbose = 1;
			break;
		default:
			usage();
		}
	}
	av += optind;
	ac -= optind;

	numfiles = ac;

	for (i = 0; i < numfiles; i++)
		files2open[i] = av[i];

	if (bdaddr_any(&info.raddr))
		usage();

	if (channel_mode == MODE_MONO || channel_mode == MODE_DUAL)
		tmpbitpool = 16;
	else 
		tmpbitpool = 32;

	if (bands == BANDS_8)
		tmpbitpool *= 8;
	else
		tmpbitpool *= 4;

	if (tmpbitpool > DEFAULT_MAXBPOOL)
		tmpbitpool = DEFAULT_MAXBPOOL;

	if (bitpool == 0 || tmpbitpool < bitpool)
		bitpool = tmpbitpool;

	base = event_init();

	filebuf = malloc(8192);
	if (filebuf == NULL)
		err(EXIT_FAILURE, "cannot alloc filebuf");

	if (init_server(&info) < 0)
		err(EXIT_FAILURE, "init server");

	if (verbose) {
		printf("A2DP:\n");
		printf("\tladdr: %s\n", bt_ntoa(&info.laddr, NULL));
		printf("\traddr: %s\n", bt_ntoa(&info.raddr, NULL));
	}

	event_base_loop(base, 0);
	event_base_free(base);

	return EXIT_SUCCESS;
}

static void
usage(void)
{
	fprintf(stderr,
		"usage:\t%s [-v][-d device] [-r rate] [-M mtu]\n"
		"\t\t[-2jms] [-b blocks] [-4] -a address files...\n"
		"Where:\n"
		"\t-d device    local device address\n"
		"\t-a address   remote device address\n"
		"\tfiles...     files to read from (Defaults to stdin).\n"
		"\t-v           verbose output\n"
		"\t-M mtu       MTU for transmission\n"
		"\t-B bitpool   Maximum bitpool value for encoding\n"
		"\t-2           input is dual channel\n"
		"\t-j           input is in joint stereo\n"
		"\t-m           input is mono\n"
		"\t-s           input is stereo (This is the default).\n"
		"\t-r rate      input frequency (441000 is the default).\n"
		"\t             Where rate is one of 16000, 32000, 441000\n"
		"\t             or 48000Hz.\n"
		"ENCODING OPTIONS:\n"
		"\t-b blocks    use blocks number of blocks (16 is the default)\n"
		"\t             Where blocks is one of 4, 8, 12, 16.\n"
		"\t-4           use 4 subbands (8 is the default)\n"
		"\n"
		"Without specifiying the channel the default is stereo,\n"
		"16 blocks, 8 subbands, loudness bit allocation, 441000 Hz.\n"
		, getprogname());

	exit(EXIT_FAILURE);
}

static void
do_interrupt(int fd, short ev, void *arg)
{
	static int currentFileInd = 0;
	int len;

	if (currentFileInd == 0)
		currentFileInd = startFileInd;

	len = stream(fd, sc, channel_mode, frequency, bands, blocks,
	    alloc_method, bitpool, mtu);

	if (len == -1 && currentFileInd >= numfiles -1) {
		event_del(&interrupt_ev);
		close(fd);
		fd = -1;
		exit(1);
	} else if (len == -1) {
		close(fd);
next_file:
		currentFileInd++;
		audfile = open(files2open[currentFileInd], O_RDONLY);
		if (audfile < 0) {
			warn("error opening file %s",
			    files2open[currentFileInd]);
			goto next_file;
		}
		currentAudioStream = fdopen(audfile, "r");
		setbuffer(currentAudioStream, filebuf, 8192);

		event_del(&interrupt_ev);
		event_set(&interrupt_ev, audfile, EV_READ |
		    EV_PERSIST, do_interrupt, NULL);
		if (event_add(&interrupt_ev, NULL) < 0)
			err(EXIT_FAILURE, "interrupt_ev");
	}

	if (verbose)
		printf("Streaming %d bytes\n",len);
}

/*
 * Initialise server socket
 */
static int
init_server(struct l2cap_info *myInfo)
{
	struct sockaddr_bt addr;
	int i;

	if (atexit(exit_func))
		err(EXIT_FAILURE,"atexit failed to initialize");

	client_query();

	hc = socket(PF_BLUETOOTH, SOCK_SEQPACKET, BTPROTO_L2CAP);
	if (hc < 0)
		return -1;

	memset(&addr, 0, sizeof(addr));
	addr.bt_len = sizeof(addr);
	addr.bt_family = AF_BLUETOOTH;
	addr.bt_psm = l2cap_psm;
	bdaddr_copy(&addr.bt_bdaddr, &myInfo->laddr);

	if (bind(hc, (struct sockaddr *)&addr, sizeof(addr)) < 0)
		return -1;

	bdaddr_copy(&addr.bt_bdaddr, &myInfo->raddr);
	if (connect(hc,(struct sockaddr*)&addr, sizeof(addr)) < 0)
		return -1;

	if (avdtpDiscover(hc, hc, &mySepInfo))
		err(EXIT_FAILURE, "DISCOVER FAILED");

	if (avdtpAutoConfig(hc, hc, mySepInfo.sep, frequency,
	    channel_mode, &alloc_method, &bitpool, &bands, &blocks))
		err(EXIT_FAILURE, "AUTOCONFIG FAILED");

	if (verbose)
		printf("Bitpool value = %d\n",bitpool);

	if (avdtpOpen(hc, hc, mySepInfo.sep))
		err(EXIT_FAILURE, "OPEN FAILED");
	if (numfiles == 0)
		audfile = STDIN_FILENO;
	else {
		for (i = 0; i < numfiles; i++) {
			audfile = open(files2open[i], O_RDONLY);
			if (audfile < 0)
				warn("error opening file %s",files2open[i]);
			else
				break;
		}
		startFileInd = i;
		if (startFileInd > numfiles - 1)
			errx(EXIT_FAILURE,"error opening file%s",
			    (numfiles > 1) ? "s":"");
	}
	currentAudioStream = fdopen(audfile, "r");
	setbuffer(currentAudioStream, filebuf, 8192);

	sc = socket(PF_BLUETOOTH, SOCK_SEQPACKET, BTPROTO_L2CAP);
	if (sc < 0)
		return -1;

	memset(&addr, 0, sizeof(addr));

	addr.bt_len = sizeof(addr);
	addr.bt_family = AF_BLUETOOTH;
	bdaddr_copy(&addr.bt_bdaddr, &info.laddr);
	addr.bt_psm = l2cap_psm;

	if (bind(sc, (struct sockaddr *)&addr, sizeof(addr)) < 0)
		return -1;

	bdaddr_copy(&addr.bt_bdaddr, &info.raddr);
	if (connect(sc,(struct sockaddr*)&addr, sizeof(addr)) < 0)
		return -1;

	socklen_t mtusize = sizeof(uint16_t);
	getsockopt(sc, BTPROTO_L2CAP, SO_L2CAP_OMTU, &mtu, &mtusize);

	if (userset_mtu != 0 && userset_mtu > 100 && userset_mtu < mtu)
		mtu = userset_mtu;
	else if (userset_mtu == 0 && mtu >= 500)
		mtu /= 2;

	if (avdtpStart(hc, hc, mySepInfo.sep)) 
		err(EXIT_FAILURE, "START FAILED");

	event_set(&interrupt_ev, audfile, EV_READ | EV_PERSIST, do_interrupt, NULL);
	if (event_add(&interrupt_ev, NULL) < 0)
		err(EXIT_FAILURE, "interrupt_ev");

	return 0;
}

static void
client_query(void)
{
	uint8_t buf[12];	/* enough for SSP and AIL both */
	sdp_session_t ss;
	sdp_data_t ssp, ail, rsp, rec, value, pdl, seq;
	uintmax_t psm, ver;
	uint16_t attr;
	bool rv;

	ss = sdp_open(&info.laddr, &info.raddr);
	if (ss == NULL) {
		log_err("%s: %m", service_type);
		exit(EXIT_FAILURE);
	}

	log_info("Searching for %s service at %s",
	    service_type, bt_ntoa(&info.raddr, NULL));

	seq.next = buf;
	seq.end = buf + sizeof(buf);

	/*
	 * build ServiceSearchPattern (9 bytes)
	 *
	 *	uuid16	"service_class"
	 *	uuid16	L2CAP
	 *	uuid16	AVDTP
	 */
	ssp.next = seq.next;
	sdp_put_uuid16(&seq, service_class);
	sdp_put_uuid16(&seq, SDP_UUID_PROTOCOL_L2CAP);
	sdp_put_uuid16(&seq, SDP_UUID_PROTOCOL_AVDTP);
	ssp.end = seq.next;

	/*
	 * build AttributeIDList (3 bytes)
	 *
	 *	uint16	ProtocolDescriptorList
	 */
	ail.next = seq.next;
	sdp_put_uint16(&seq, SDP_ATTR_PROTOCOL_DESCRIPTOR_LIST);
	ail.end = seq.next;

	rv = sdp_service_search_attribute(ss, &ssp, &ail, &rsp);
	if (!rv) {
		log_err("%s: %m", service_type);
		exit(EXIT_FAILURE);
	}

	/*
	 * we expect the response to contain a list of records
	 * containing a ProtocolDescriptorList. Find the first
	 * one containing L2CAP and AVDTP >= 1.0, and extract
	 * the PSM.
	 */
	rv = false;
	while (!rv && sdp_get_seq(&rsp, &rec)) {
		if (!sdp_get_attr(&rec, &attr, &value)
		    || attr != SDP_ATTR_PROTOCOL_DESCRIPTOR_LIST)
			continue;

		sdp_get_alt(&value, &value);	/* drop any alt header */
		while (!rv && sdp_get_seq(&value, &pdl)) {
			if (sdp_get_seq(&pdl, &seq) &&
			    sdp_match_uuid16(&seq, SDP_UUID_PROTOCOL_L2CAP) &&
			    sdp_get_uint(&seq, &psm) &&
			    sdp_get_seq(&pdl, &seq) &&
			    sdp_match_uuid16(&seq, SDP_UUID_PROTOCOL_AVDTP) &&
			    sdp_get_uint(&seq, &ver) && ver >= 0x0100)
				rv = true;
		}
	}

	sdp_close(ss);

	if (!rv) {
		log_err("%s query failed", service_type);
		exit(EXIT_FAILURE);
	}

	l2cap_psm = (uint16_t)psm;
	log_info("Found PSM %u for service %s", l2cap_psm, service_type);
}

static void
exit_func()
{
	avdtpAbort(hc, hc, mySepInfo.sep); 
	avdtpClose(hc, hc, mySepInfo.sep); 
	close(sc);
	close(hc);
	if (filebuf)
		free(filebuf);
	sc = hc = -1;
}

