#include <sys/types.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/disk.h>

#include <sys/disklabel.h>

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <err.h>
#include <unistd.h>
#include <util.h>

#include "rdb.h"
#include "util.h"

int debug = 0;

static void
usage()
{
	fprintf(stderr,"usage: rdbedit [-F] [-i] [-s blkno] [-e] [-t] device\n");
	exit(1);
}

static struct mitem menu_main[] = {
	{ 'r', "edit rigid disk block" },
	{ 'p', "edit partition PARTBLK" },
	{ 'c', "create partition PARTBLK NUMCYLS" },
	{ 'd', "delete partition PARTBLK" },
	{ '.', "print" },
	{ 'a', "abort" },
	{ 'q', "quit" }
};

static int
edit(struct rdb_mem *rdbm, struct diskgeometry *g)
{
	struct part_mem *pbm;
	int modr, modp, c;
	u_long blk, num;

	if (!batch) display_all(rdbm);

	modr = modp = 0;
	while ((c = MENU(menu_main)) != EOF) {
		switch (c) {
		case 'q':
			break;
		case 'a':
			modr = modp = 0;
			break;
		case '.':
			display_all(rdbm);
			break;
		case 'r':
			modr = edit_rdb(rdbm, g);
			break;
		case 'p':
			if (getuint(&blk)) {
				pbm = find_part(rdbm, blk);
				if (pbm != NULL) {
					if (edit_part(pbm, rdbm))
						modp = 1;
				} else
					printf("No PART block #%lu\n",blk);
			}
			break;
		case 'c':
			if (getuint(&blk) && getuint(&num)) {
				pbm = find_part(rdbm, blk);
				blk = find_blk(rdbm);
				if (blk != 0) {
					add_part(rdbm, blk, pbm, num);
					modp = 1;
				} else
					printf("No free RDB block found\n");
			}
			break;
		case 'd':
			if (getuint(&blk)) {
				pbm = find_part(rdbm, blk);
				if (pbm != NULL) {
					if (delete_part(rdbm, pbm))
						modp = 1;
				} else
					printf("No PART block #%lu\n",blk);
			}
			break;
		default:
			printf("invalid input: '%c'\n",c);
			break;
		}
		if (c == 'a' || c == 'q')
			break;
		c = '\0';
	}
	endmenu();

	return modr || modp;
}

static void
file2geo(int fd, struct diskgeometry *g)
{
	struct stat S, *st = &S;

	if (fstat(fd, st) == -1)
		err(1, "cannot stat");

	g->blocks    = st->st_size / 512;
	g->blocksize = 512;
	g->sectors   = 1;
	g->heads     = 1;

	while (g->sectors < 32) {
		if (g->blocks % (2*g->sectors))
			break;
		g->sectors = 2*g->sectors;
	}
	while (g->heads < 16) {
		if (g->blocks % (g->sectors * 2*g->heads))
			break;
		g->heads = 2*g->heads;
	}
}

static void
disk2geo(int fd, struct diskgeometry *g)
{
	struct disklabel DL, *dl = &DL;

	if (ioctl(fd, DIOCGDINFO, dl) == -1)
		err(1, "cannot query disklabel");

	g->blocks    = dl->d_secperunit;
	g->blocksize = dl->d_secsize;
	g->sectors   = dl->d_nsectors;
	g->heads     = dl->d_ntracks;
}

static void
printgeo(struct diskgeometry *g)
{
	printf("GEO: Blocks %u Cylinders %u Heads %u Sectors %u Bytes %u\n",
		g->blocks,
		g->blocks / (g->heads * g->sectors),
		g->heads,
		g->sectors,
		g->blocksize);
}

int main(int argc, char *argv[])
{
	static char namebuf[MAXPATHLEN + 1];
	int init_mode, edit_mode, disktab_mode, F_flag;
	char *disk;
	struct diskgeometry geo;
	int flags;

	int fd, ch, modified;
	int start;
	struct rdb_mem *rdbm;

	batch = !isatty(STDIN_FILENO);

	init_mode = 0;
	edit_mode = 0;
	disktab_mode = 0;
	F_flag    = 0;

	start = 0;

	while ((ch = getopt(argc, argv, "Fis:etd")) != -1) {
		switch (ch) {
		case 'F': F_flag=1;break;
		case 'i': init_mode=1;break;
		case 's': start=atoi(optarg); break;
		case 'e': edit_mode=1;break;
		case 't': disktab_mode=1;break;
		case 'd': ++debug;break;
		case '?': /* FALLTRHOUGH */
		default: usage();
		}
	}
	if (optind+1 != argc)
		usage();

	disk = argv[optind];
	flags = (edit_mode || init_mode) ? O_RDWR : O_RDONLY;

	if (F_flag) {
		fd = open(disk, flags, 0);
		if (fd == -1)
			err(1, "cannot open %s", disk);
		file2geo(fd, &geo);
	} else {
		fd = opendisk(disk, flags, namebuf, sizeof(namebuf), 0);
		if (fd == -1) {
			if (errno == ENODEV)
				errx(1,"%s is not a character device", namebuf);
			else
				err(1, "cannot opendisk %s", disk);
		}
		disk2geo(fd, &geo);
	}

	if (init_mode) {
		rdbm = init_rdb(start, &geo);
	} else {
		rdbm = load_rdb(fd);
	}
	if (rdbm == NULL)
		errx(1, "cannot init/load rdb");


	modified = 0;
	if (edit_mode) {
		modified = edit(rdbm, &geo);
	} else if (disktab_mode) {
		print_disktab(rdbm);
	} else {
		printgeo(&geo);
		display_all(rdbm);
	}

	if (init_mode || modified) {
		if (yesno("Write back modified RDB")) {
			if (init_mode)
				erase_rdb(fd, rdbm);
			write_rdb(fd, rdbm);
		} else
			printf("RDB not written!\n");
	}
	free_rdb(rdbm);

	close (fd);
	return 0;
}

