/* $NetBSD$ */

/*
 * Copyright (c) 2016 Jonathan A. Kollasch
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD$");

#include <sys/param.h>
#include <sys/conf.h>
#include <sys/kmem.h>
//#include <sys/systm.h>
//#include <sys/fcntl.h>
#include <sys/device.h>
//#include <sys/ioctl.h>
#include <dev/spi/spivar.h>
#include <dev/spi/spi_io.h>

static inline __unused void
hexdump(const uint8_t *buf, size_t len)
{
	size_t pos;

	for (pos = 0; pos < len; pos++) {
		if ((pos & 15) == 0)
			printf("%08zx ", pos);
		printf("%02x%c", buf[pos], (pos & 15) == 15 ? '\n' : ' ');
	}
	if ((len & 15) != 0)
		printf("\n");
}

struct spigen_softc {
	device_t sc_dev;
	struct spi_handle *sc_sh;
};

#define SPIGENUNIT(n) (minor(n))

dev_type_open(spigenopen);
dev_type_close(spigenclose);
dev_type_ioctl(spigenioctl);

const struct cdevsw spigen_cdevsw = {
	.d_open = spigenopen,
	.d_close = spigenclose,
	.d_read = noread,
	.d_write = nowrite,
	.d_ioctl = spigenioctl,
	.d_stop = nostop,
	.d_tty = notty,
	.d_poll = nopoll,
	.d_mmap = nommap,
	.d_kqfilter = nokqfilter,
	.d_discard = nodiscard,
	.d_flag = D_OTHER,
};


static int spigen_match(device_t, cfdata_t, void *);
static void spigen_attach(device_t, device_t, void *);
extern struct cfdriver spigen_cd;
CFATTACH_DECL_NEW(spigen, sizeof(struct spigen_softc), spigen_match,
    spigen_attach, NULL, NULL);

static int
spigen_match(device_t parent, cfdata_t cf, void *aux)
{
	//struct spi_attach_args * const sa = aux;

	return 1;
}

static void
spigen_attach(device_t parent, device_t self, void *aux)
{
	struct spigen_softc * const sc = device_private(self);
	struct spi_attach_args * const sa = aux;

	sc->sc_dev = self;
	sc->sc_sh = sa->sa_handle;

	aprint_normal("\n");
	aprint_naive("\n");

	if (spi_configure(sc->sc_sh, SPI_MODE_0, 12000000))
		aprint_error_dev(sc->sc_dev, "configure failed\n");
}

int
spigenopen(dev_t dev, int flag, int mode, struct lwp *l)
{
	struct spigen_softc * const sc =
	    device_lookup_private(&spigen_cd, SPIGENUNIT(dev));
	device_printf(sc->sc_dev, "%s()\n", __func__);

	return 0;
}

int
spigenclose(dev_t dev, int flag, int mode, struct lwp *l)
{
	struct spigen_softc * const sc =
	    device_lookup_private(&spigen_cd, SPIGENUNIT(dev));
	device_printf(sc->sc_dev, "%s()\n", __func__);

	return 0;
}

static int
execute_xfers(struct spigen_softc * const sc, const size_t nvec, const struct spivect * const vecs)
{
	size_t i;
	struct spi_transfer trans;
	struct spi_chunk chunk[nvec];

	spi_transfer_init(&trans);

	for (i = 0; i < nvec; i++) {
		spi_chunk_init(&chunk[i], vecs[i].len, vecs[i].tx, vecs[i].rx);
		spi_transfer_add(&trans, &chunk[i]);
	}

	spi_transfer(sc->sc_sh, &trans);
	spi_wait(&trans);

	if (trans.st_flags & SPI_F_ERROR)
		return trans.st_errno;

	return 0;
}

int
spigenioctl(dev_t dev, u_long cmd, void *data, int flag,
    struct lwp *l)
{
	struct spigen_softc * const sc =
	    device_lookup_private(&spigen_cd, SPIGENUNIT(dev));
	size_t len;
	struct spivect *sv;
	int error;

	switch (cmd) {
	case SPI_IOCTL_XFER: {
		const struct spixfer * const sx = data;
		device_printf(sc->sc_dev, "SPI_IOCTL_XFER\n");
		device_printf(sc->sc_dev, "%zu %p\n", sx->nvec, sx->vec);
		len = sizeof(struct spivect) * sx->nvec;
		sv = kmem_alloc(len, KM_SLEEP);
		error = copyin(sx->vec, sv, len);
		if (error != 0)
			device_printf(sc->sc_dev, "copyin failed %d\n", error);
		hexdump((void *)sv, len);
		//hexdump(sv[0].tx, sv[0].len);
		execute_xfers(sc, sx->nvec, sv);
		kmem_free(sv, len);
		sv = NULL;
		break;
	}
	default:
		return ENODEV;
	}

	return 0;
}
