/*	$NetBSD: cmd.c,v 1.8 2003/08/07 11:17:21 agc Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Edward Wang at The University of California, Berkeley.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
#ifndef lint
__RCSID("$NetBSD: cmd.c,v 1.8 2003/08/07 11:17:21 agc Exp $");
#endif /* not lint */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "defs.h"
#include "char.h"


static int
checkproc(ww_t *w)
{
	if (w->ww_state != WWS_HASPROC) {
		error("No process in window.");
		return -1;
	}
	return 0;
}

static ww_t *
getwin(winvars_t *winvars)
{
	int c;
	ww_t *w = 0;

	if (!winvars->terse)
		wwputs(winvars, "Which window? ", winvars->cmdwin);
	WWCURTOWIN(winvars, winvars->cmdwin);
	while ((c = WWGETC(winvars)) < 0)
		wwiomux(winvars);
	if (winvars->debug && c == 'c')
		w = winvars->cmdwin;
	else if (winvars->debug && c == 'f')
		w = winvars->framewin;
	else if (winvars->debug && c == 'b')
		w = winvars->boxwin;
	else if (c >= '1' && c < MAX_NUM_WINDOWS + '1')
		w = winvars->window[c - '1'];
	else if (c == '+')
		w = winvars->selwin;
	else if (c == '-')
		w = winvars->lastselwin;
	if (w == 0)
		wwbell();
	if (!winvars->terse)
		wwputc('\n', winvars->cmdwin);
	return w;
}

static int
getpos(winvars_t *winvars, int *row, int *col, int minrow, int mincol, int maxrow, int maxcol)
{
	static int scount;
	int count;
	int c;
	int oldrow = *row, oldcol = *col;

	while ((c = WWGETC(winvars)) >= 0) {
		switch (c) {
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			scount = scount * 10 + c - '0';
			continue;
		}
		count = scount ? scount : 1;
		scount = 0;
		switch (c) {
		case 'h':
			if ((*col -= count) < mincol)
				*col = mincol;
			break;
		case 'H':
			*col = mincol;
			break;
		case 'l':
			if ((*col += count) > maxcol)
				*col = maxcol;
			break;
		case 'L':
			*col = maxcol;
			break;
		case 'j':
			if ((*row += count) > maxrow)
				*row = maxrow;
			break;
		case 'J':
			*row = maxrow;
			break;
		case 'k':
			if ((*row -= count) < minrow)
				*row = minrow;
			break;
		case 'K':
			*row = minrow;
			break;
		case CONTROL('['):
			if (!winvars->terse)
				wwputs(winvars, "\nCancelled.  ", winvars->cmdwin);
			return 3;
		case '\r':
			return 2;
		default:
			if (!winvars->terse)
				wwputs(winvars, "\nType [hjklHJKL] to move, return to enter position, escape to cancel.", winvars->cmdwin);
			wwbell();
		}
	}
	return oldrow != *row || oldcol != *col;
}

static void		c_window(winvars_t *);
static void		c_help(winvars_t *);
static void		c_colon(winvars_t *);
static void		c_debug(winvars_t *);
static void		c_move(winvars_t *, ww_t *);
static void		c_put(winvars_t *);
static void		c_quit(winvars_t *);
static void		c_size(winvars_t *, ww_t *);
static void		c_yank(winvars_t *);

void
docmd(winvars_t *winvars)
{
	int c;
	ww_t *w;
	char out = 0;

	while (!out && !winvars->quit) {
		if ((c = WWGETC(winvars)) < 0) {
			if (winvars->terse)
				WWSETCURSOR(winvars, 0, 0);
			else {
				wwputs(winvars, "Command: ", winvars->cmdwin);
				WWCURTOWIN(winvars, winvars->cmdwin);
			}
			do
				wwiomux(winvars);
			while ((c = WWGETC(winvars)) < 0);
		}
		if (!winvars->terse)
			wwputc('\n', winvars->cmdwin);
		switch (c) {
		default:
			if (c != winvars->escapec)
				break;
			/* FALLTHROUGH */
		case 'h': case 'j': case 'k': case 'l':
		case 'y': case 'p':
		case CONTROL('y'):
		case CONTROL('e'):
		case CONTROL('u'):
		case CONTROL('d'):
		case CONTROL('b'):
		case CONTROL('f'):
		case CONTROL('s'):
		case CONTROL('q'):
		case CONTROL('['):
			if (winvars->selwin == 0) {
				error("No window.");
				continue;
			}
		}
		switch (c) {
		case '1': case '2': case '3': case '4': case '5':
		case '6': case '7': case '8': case '9':
			if ((w = winvars->window[c - '1']) == 0) {
				error("%c: No such window.", c);
				break;
			}
			setselwin(winvars, w);
			if (checkproc(winvars->selwin) >= 0)
				 out = 1;
			break;
		case '%':
			if ((w = getwin(winvars)) != 0)
				setselwin(winvars, w);
			break;
		case CONTROL('^'):
			if (winvars->lastselwin != 0) {
				setselwin(winvars, winvars->lastselwin);
				if (checkproc(winvars->selwin) >= 0)
					out = 1;
			} else
				error("No previous window.");
			break;
		case 'c':
			if ((w = getwin(winvars)) != 0)
				closewin(winvars, w);
			break;
		case 'w':
			c_window(winvars);
			break;
		case 'm':
			if ((w = getwin(winvars)) != 0)
				c_move(winvars, w);
			break;
		case 'M':
			if ((w = getwin(winvars)) != 0)
				movewin(w, w->ww_alt.t, w->ww_alt.l);
			break;
		case 's':
			if ((w = getwin(winvars)) != 0)
				c_size(winvars, w);
			break;
		case 'S':
			if ((w = getwin(winvars)) != 0)
				sizewin(w, w->ww_alt.nr, w->ww_alt.nc);
			break;
		case 'y':
			c_yank(winvars);
			break;
		case 'p':
			c_put(winvars);
			break;
		case ':':
			c_colon(winvars);
			break;
		case 'h':
			(void) wwwrite(winvars, winvars->selwin, "\b", 1);
			break;
		case 'j':
			(void) wwwrite(winvars, winvars->selwin, "\n", 1);
			break;
		case 'k':
			(void) wwwrite(winvars, winvars->selwin, "\033A", 2);
			break;
		case 'l':
			(void) wwwrite(winvars, winvars->selwin, "\033C", 2);
			break;
		case CONTROL('e'):
			wwscroll(winvars, winvars->selwin, 1);
			break;
		case CONTROL('y'):
			wwscroll(winvars, winvars->selwin, -1);
			break;
		case CONTROL('d'):
			wwscroll(winvars, winvars->selwin, winvars->selwin->ww_w.nr / 2);
			break;
		case CONTROL('u'):
			wwscroll(winvars, winvars->selwin, - winvars->selwin->ww_w.nr / 2);
			break;
		case CONTROL('f'):
			wwscroll(winvars, winvars->selwin, winvars->selwin->ww_w.nr);
			break;
		case CONTROL('b'):
			wwscroll(winvars, winvars->selwin, - winvars->selwin->ww_w.nr);
			break;
		case CONTROL('s'):
			stopwin(winvars->selwin);
			break;
		case CONTROL('q'):
			startwin(winvars->selwin);
			break;
		case CONTROL('l'):
			wwredraw(winvars);
			break;
		case '?':
			c_help(winvars);
			break;
		case CONTROL('['):
			if (checkproc(winvars->selwin) >= 0)
				out = 1;
			break;
		case CONTROL('z'):
			wwsuspend(winvars);
			break;
		case 'q':
			c_quit(winvars);
			break;
		/* debugging stuff */
		case '&':
			if (winvars->debug) {
				c_debug(winvars);
				break;
			}
			/* FALLTHROUGH */
		default:
			if (c == winvars->escapec) {
				if (checkproc(winvars->selwin) >= 0) {
					(void) write(winvars->selwin->ww_pty,
						&winvars->escapec, 1);
					out = 1;
				}
			} else {
				if (!winvars->terse)
					wwbell();
				error("Type ? for help.");
			}
		}
	}
	if (!winvars->quit)
		setcmd(winvars, 0);
}

void
setcmd(winvars_t *winvars, char new)
{
	if (new && !winvars->incmd) {
		if (!winvars->terse)
			wwadd(winvars, winvars->cmdwin, &winvars->wwhead);
		if (winvars->selwin != 0)
			wwcursor(winvars, winvars->selwin, 1);
		winvars->wwcurwin = 0;
	} else if (!new && winvars->incmd) {
		if (!winvars->terse) {
			wwdelete(winvars, winvars->cmdwin);
			reframe();
		}
		if (winvars->selwin != 0)
			wwcursor(winvars, winvars->selwin, 0);
		winvars->wwcurwin = winvars->selwin;
	}
	winvars->incmd = new;
}

void
setterse(winvars_t *winvars, char new)
{
	if (winvars->incmd) {
		if (new && !winvars->terse) {
			wwdelete(winvars, winvars->cmdwin);
			reframe();
		} else if (!new && winvars->terse)
			wwadd(winvars, winvars->cmdwin, &winvars->wwhead);
	}
	winvars->terse = new;
}

/*
 * Set the current window.
 */
void
setselwin(winvars_t *winvars, ww_t *w)
{
	if (winvars->selwin == w)
		return;
	if (winvars->selwin != 0)
		winvars->lastselwin = winvars->selwin;
	if ((winvars->selwin = w) != 0)
		front(winvars->selwin, 1);
}

static void
c_window(winvars_t *winvars)
{
	int col, row, xcol, xrow;
	int id;

	if ((id = findid(winvars)) < 0)
		return;
	if (!winvars->terse)
		wwputs(winvars, "New window (upper left corner): ", winvars->cmdwin);
	col = 0;
	row = 1;
	wwadd(winvars, winvars->boxwin, winvars->framewin->ww_back);
	for (;;) {
		wwbox(winvars, winvars->boxwin, row - 1, col - 1, 3, 3);
		WWSETCURSOR(winvars, row, col);
		while (WWPEEKC(winvars) < 0)
			wwiomux(winvars);
		switch (getpos(winvars, &row, &col, row > 1, 0,
			winvars->wwnrow - 1, winvars->wwncol - 1)) {
		case 3:
			WWUNBOX(winvars, winvars->boxwin);
			wwdelete(winvars, winvars->boxwin);
			return;
		case 2:
			WWUNBOX(winvars, winvars->boxwin);
			break;
		case 1:
			WWUNBOX(winvars, winvars->boxwin);
			/* FALLTHROUGH */
		case 0:
			continue;
		}
		break;
	}
	if (!winvars->terse)
		wwputs(winvars, "\nNew window (lower right corner): ", winvars->cmdwin);
	xcol = col;
	xrow = row;
	for (;;) {
		wwbox(winvars, winvars->boxwin, row - 1, col - 1,
			xrow - row + 3, xcol - col + 3);
		WWSETCURSOR(winvars, xrow, xcol);
		while (WWPEEKC(winvars) < 0)
			wwiomux(winvars);
		switch (getpos(winvars, &xrow, &xcol, row, col, winvars->wwnrow - 1, winvars->wwncol - 1))
		{
		case 3:
			WWUNBOX(winvars, winvars->boxwin);
			wwdelete(winvars, winvars->boxwin);
			return;
		case 2:
			WWUNBOX(winvars, winvars->boxwin);
			break;
		case 1:
			WWUNBOX(winvars, winvars->boxwin);
			/* FALLTHROUGH */
		case 0:
			continue;
		}
		break;
	}
	wwdelete(winvars, winvars->boxwin);
	if (!winvars->terse)
		wwputc('\n', winvars->cmdwin);
	WWCURTOWIN(winvars, winvars->cmdwin);
	(void) openwin(winvars, id, row, col, xrow-row+1, xcol-col+1, winvars->default_nline,
	    NULL, WWT_PTY, WWU_HASFRAME, winvars->default_shellfile,
	    winvars->default_shell);
}

const char *help_shortcmd[] = {
	"#       Select window # and return to conversation mode",
	"%#      Select window # but stay in command mode",
	"escape  Return to conversation mode without changing window",
	"^^      Return to conversation mode and change to previous window",
	"c#      Close window #",
	"w       Open a new window",
	"m#      Move window #",
	"M#      Move window # to its previous position",
	"s#      Change the size of window #",
	"S#      Change window # to its previous size",
	"^Y      Scroll up one line",
	"^E      Scroll down one line",
	"^U      Scroll up half a window",
	"^D      Scroll down half a window",
	"^B      Scroll up a full window",
	"^F      Scroll down a full window",
	"h       Move cursor left",
	"j       Move cursor down",
	"k       Move cursor up",
	"l       Move cursor right",
	"y       Yank",
	"p       Put",
	"^S      Stop output in current window",
	"^Q      Restart output in current window",
	"^L      Redraw screen",
	"^Z      Suspend",
	"q       Quit",
	":       Enter a long command",
	0
};

const char *help_longcmd[] = {
	":alias name string ...  Make `name' an alias for `string ...'",
	":alias                  Show all aliases",
	":close # ...            Close windows",
	":close all              Close all windows",
	":cursor modes           Set the cursor modes",
	":echo # string ...      Print `string ...' in window #",
	":escape c               Set escape character to `c'",
	":foreground # flag      Make # a foreground window, if `flag' is true",
	":label # string         Set label of window # to `string'",
	":list                   List all open windows",
	":default_nline lines    Set default window buffer size to `lines'",
	":default_shell string ...",
	"                        Set default shell to `string ...'",
	":default_smooth flag    Set default smooth scroll flag",
	":select #               Select window #",
	":smooth # flag          Set window # to smooth scroll mode",
	":source filename        Execute commands in `filename'",
	":terse flag             Set terse mode",
	":unalias name           Undefine `name' as an alias",
	":unset variable         Deallocate `variable'",
	":variable               List all variables",
	":window [row col nrow ncol nline label pty frame mapnl keepopen smooth shell]",
	"                        Open a window at `row', `col' of size `nrow', `ncol',",
	"                        with `nline' lines in the buffer, and `label'",
	":write # string ...     Write `string ...' to window # as input",
	0
};

static int
help_print(ww_t *w, const char *name, const char **list)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	wwprintf(w, "%s:\n\n", name);
	while (*list)
		switch (more(w, 0)) {
		case 0:
			wwputs(winvars, *list++, w);
			wwputc('\n', w);
			break;
		case 1:
			wwprintf(w, "%s: (continued)\n\n", name);
			break;
		case 2:
			return -1;
		}
	return more(w, 1) == 2 ? -1 : 0;
}

void
c_help(winvars_t *winvars)
{
	ww_t *w;

	if ((w = openiwin(winvars, winvars->wwnrow - 3, "Help")) == 0) {
		error("Can't open help window: %s.", wwerror(winvars));
		return;
	}
	wwprintf(w, "The escape character is %c.\n", winvars->escapec);
	wwprintf(w, "(# represents one of the digits from 1 to 9.)\n\n");
	if (help_print(w, "Short commands", help_shortcmd) >= 0)
		(void) help_print(w, "Long commands", help_longcmd);
	closeiwin(w);
}

static void
c_quit(winvars_t *winvars)
{
	char oldterse;

	oldterse = winvars->terse;
	setterse(winvars, 0);
	wwputs(winvars, "Really quit [yn]? ", winvars->cmdwin);
	WWCURTOWIN(winvars, winvars->cmdwin);
	while (WWPEEKC(winvars) < 0)
		wwiomux(winvars);
	if (WWGETC(winvars) == 'y') {
		wwputs(winvars, "Yes", winvars->cmdwin);
		winvars->quit++;
	} else
		wwputc('\n', winvars->cmdwin);
	setterse(winvars, !winvars->quit && oldterse);
}

void
setescape(winvars_t *winvars, char *esc)
{
	if (*esc == '^') {
		if (esc[1] != 0)
			winvars->escapec = esc[1] & 0x1f;
		else
			winvars->escapec = '^';
	} else
		winvars->escapec = *esc;
}

int
setlabel(ww_t *w, const char *label)
{
	if (w->ww_label != 0)
		free(w->ww_label);
	if ((w->ww_label = strdup(label)) == 0)
		return -1;
	return 0;
}

static void
c_colon(winvars_t *winvars)
{
	char oldterse = winvars->terse;
	char buf[512];

	setterse(winvars, 0);
	wwputc(':', winvars->cmdwin);
	wwgets(winvars, buf, winvars->wwncol - 3, winvars->cmdwin);
	wwputc('\n', winvars->cmdwin);
	WWCURTOWIN(winvars, winvars->cmdwin);
	setterse(winvars, oldterse);
	if (dolongcmd(buf, NULL, 0) < 0)
		error("Out of memory.");
}

/*
 * Window movement.
 */

void	getminmax(int, int, int, int, int *, int *, int *);

static void
c_move(winvars_t *winvars, ww_t *w)
{
	int col, row;
	int mincol, minrow;
	int maxcol, maxrow;
	int curcol, currow;

	if (!winvars->terse)
		wwputs(winvars, "New window position: ", winvars->cmdwin);
	col = w->ww_w.l;
	row = w->ww_w.t;
	wwadd(winvars, winvars->boxwin, winvars->framewin->ww_back);
	for (;;) {
		wwbox(winvars, winvars->boxwin, row - 1, col - 1, w->ww_w.nr + 2, w->ww_w.nc + 2);
		getminmax(row, w->ww_w.nr, 1, winvars->wwnrow,
			&currow, &minrow, &maxrow);
		getminmax(col, w->ww_w.nc, 0, winvars->wwncol,
			&curcol, &mincol, &maxcol);
		WWSETCURSOR(winvars, currow, curcol);
		while (WWPEEKC(winvars) < 0)
			wwiomux(winvars);
		switch (getpos(winvars, &row, &col, minrow, mincol, maxrow, maxcol)) {
		case 3:
			WWUNBOX(winvars, winvars->boxwin);
			wwdelete(winvars, winvars->boxwin);
			return;
		case 2:
			WWUNBOX(winvars, winvars->boxwin);
			break;
		case 1:
			WWUNBOX(winvars, winvars->boxwin);
			continue;
		case 0:
			continue;
		}
		break;
	}
	wwdelete(winvars, winvars->boxwin);
	if (!winvars->terse)
		wwputc('\n', winvars->cmdwin);
	WWCURTOWIN(winvars, winvars->cmdwin);
	movewin(w, row, col);
}

void
movewin(ww_t *w, int row, int col)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	ww_t *back = w->ww_back;

	w->ww_alt.t = w->ww_w.t;
	w->ww_alt.l = w->ww_w.l;
	wwdelete(winvars, w);
	wwmove(winvars, w, row, col);
	wwadd(winvars, w, back);
	reframe();
}

/*
 * Weird stufff, don't ask.
 */
void
getminmax(int x, int n, int a, int b, int *curx, int *minx, int *maxx)
{
	if (x < 0)
		*curx = x + n - 1;
	else
		*curx = x;

	if (x <= a)
		*minx = 1 - n;
	else if (x <= b - n)
		*minx = a;
	else
		*minx = b - n;

	if (x >= b - n)
		*maxx = b - 1;
	else if (x >= a)
		*maxx = b - n;
	else
		*maxx = a;
}

/*
 * Debugging commands.
 */

void	debug_str(void);

static void
c_debug(winvars_t *winvars)
{
	ww_t *w;

	if (!winvars->terse)
		wwputs(winvars, "[m(smap) n(ns) o(os) s(string) v(nvis) w(win)]? ",
		    winvars->cmdwin);
	WWCURTOWIN(winvars, winvars->cmdwin);
	while (WWPEEKC(winvars) < 0)
		wwiomux(winvars);
	if (!winvars->terse)
		wwputc('\n', winvars->cmdwin);
	switch (WWGETC(winvars)) {
	case 'm':
		wwdumpsmap(winvars);
		break;
	case 'n':
		wwdumpns(winvars);
		break;
	case 'o':
		wwdumpos(winvars);
		break;
	case 's':
		debug_str();
		break;
	case 'v':
		if ((w = getwin(winvars)) != 0)
			wwdumpnvis(w);
		break;
	case 'w':
		if ((w = getwin(winvars)) != 0)
			wwdumpwin(w);
		break;
	default:
		wwbell();
	}
}

void
debug_str(void)
{
	error("No string debugging.");
}

static void
yank_highlight_line(winvars_t *winvars, int r, int c, int cend)
{
	ww_t *w;
	char *win;

	w = winvars->selwin;
	if (r < w->ww_i.t || r >= w->ww_i.b)
		return;
	if (c < w->ww_i.l)
		c = w->ww_i.l;
	if (cend >= w->ww_i.r)
		cend = w->ww_i.r;
	for (win = w->ww_win[r] + c; c < cend; c++, win++) {
		*win ^= WWM_REV;
		if (winvars->wwsmap[r][c] == w->ww_index) {
			if (*win == 0)
				w->ww_nvis[r]++;
			else if (*win == WWM_REV)
				w->ww_nvis[r]--;
			setmodepart(&winvars->wwns[r][c], getmodepart(&winvars->wwns[r][c]) ^ WWM_REV);
			winvars->wwtouched[r] |= WWU_TOUCHED;
		}
	}
}

static void
yank_highlight(winvars_t *winvars, int row1, int col1, int row2, int col2)
{
	ww_t *w;
	int r, c;

	w = winvars->selwin;
	if ((winvars->wwavailmodes & WWM_REV) == 0)
		return;
	if (row2 < row1 || (row2 == row1 && col2 < col1)) {
		r = row1;
		c = col1;
		row1 = row2;
		col1 = col2;
		row2 = r;
		col2 = c;
	}
	c = col1;
	for (r = row1; r < row2; r++) {
		yank_highlight_line(winvars, r, c, w->ww_b.r);
		c = w->ww_b.l;
	}
	yank_highlight_line(winvars, r, c, col2);
}

static void	unyank(void);
static void	yank_line(winvars_t *, int, int, int);

/*
 * Window size.
 */
static void
c_size(winvars_t *winvars, ww_t *w)
{
	int col, row;

	if (!winvars->terse)
		wwputs(winvars, "New window size (lower right corner): ", winvars->cmdwin);
	col = MIN(w->ww_w.r, winvars->wwncol) - 1;
	row = MIN(w->ww_w.b, winvars->wwnrow) - 1;
	wwadd(winvars, winvars->boxwin, winvars->framewin->ww_back);
	for (;;) {
		wwbox(winvars, winvars->boxwin, w->ww_w.t - 1, w->ww_w.l - 1,
			row - w->ww_w.t + 3, col - w->ww_w.l + 3);
		WWSETCURSOR(winvars, row, col);
		while (WWPEEKC(winvars) < 0)
			wwiomux(winvars);
		switch (getpos(winvars, &row, &col, w->ww_w.t, w->ww_w.l,
			winvars->wwnrow - 1, winvars->wwncol - 1)) {
		case 3:
			WWUNBOX(winvars, winvars->boxwin);
			wwdelete(winvars, winvars->boxwin);
			return;
		case 2:
			WWUNBOX(winvars, winvars->boxwin);
			break;
		case 1:
			WWUNBOX(winvars, winvars->boxwin);
			continue;
		case 0:
			continue;
		}
		break;
	}
	wwdelete(winvars, winvars->boxwin);
	if (!winvars->terse)
		wwputc('\n', winvars->cmdwin);
	WWCURTOWIN(winvars, winvars->cmdwin);
	sizewin(w, row - w->ww_w.t + 1, col - w->ww_w.l + 1);
}

/*
 * Yank and put
 */

struct yb {
	char *line;
	int length;
	struct yb *link;
};
struct yb *yb_head, *yb_tail;

static void
c_yank(winvars_t *winvars)
{
	ww_t *w;
	int col1, row1;
	int col2, row2;
	int r, c;

	w = winvars->selwin;
	if (!winvars->terse)
		wwputs(winvars, "Yank starting position: ", winvars->cmdwin);
	wwcursor(winvars, w, 0);
	row1 = w->ww_cur.r;
	col1 = w->ww_cur.c;
	for (;;) {
		WWSETCURSOR(winvars, row1, col1);
		while (WWPEEKC(winvars) < 0)
			wwiomux(winvars);
		switch (getpos(winvars, &row1, &col1, w->ww_i.t, w->ww_i.l,
			       w->ww_i.b - 1, w->ww_i.r - 1)) {
		case 3:
			goto out;
		case 2:
			break;
		case 1:
		case 0:
			continue;
		}
		break;
	}
	if (!winvars->terse)
		wwputs(winvars, "\nYank ending position: ", winvars->cmdwin);
	row2 = row1;
	col2 = col1;
	for (;;) {
		WWSETCURSOR(winvars, row2, col2);
		while (WWPEEKC(winvars) < 0)
			wwiomux(winvars);
		r = row2;
		c = col2;
		switch (getpos(winvars, &row2, &col2, w->ww_i.t, w->ww_i.l,
			       w->ww_i.b - 1, w->ww_i.r - 1)) {
		case 3:
			yank_highlight(winvars, row1, col1, r, c);
			goto out;
		case 2:
			break;
		case 1:
			yank_highlight(winvars, row1, col1, r, c);
			yank_highlight(winvars, row1, col1, row2, col2);
			continue;
		case 0:
			continue;
		}
		break;
	}
	if (row2 < row1 || (row2 == row1 && col2 < col1)) {
		r = row1;
		c = col1;
		row1 = row2;
		col1 = col2;
		row2 = r;
		col2 = c;
	}
	unyank();
	c = col1;
	for (r = row1; r < row2; r++) {
		yank_line(winvars, r, c, w->ww_b.r);
		c = w->ww_b.l;
	}
	yank_line(winvars, r, c, col2);
	yank_highlight(winvars, row1, col1, row2, col2);
	if (!winvars->terse)
		wwputc('\n', winvars->cmdwin);
out:
	wwcursor(winvars, w, 1);
}

void
unyank(void)
{
	struct yb *yp, *yq;

	for (yp = yb_head; yp; yp = yq) {
		yq = yp->link;
		free(yp->line);
		free(yp);
	}
	yb_head = yb_tail = NULL;
}

void
yank_line(winvars_t *winvars, int r, int c, int cend)
{
	struct yb *yp;
	int nl = 0;
	int n;
	dispcell_t *bp;
	char *cp;

	if (c == cend)
		return;
	if ((yp = (struct yb *) malloc(sizeof *yp)) == 0)
		return;
	yp->link = 0;
	nl = cend == winvars->selwin->ww_b.r;
	bp = winvars->selwin->ww_buf[r];
	for (cend--; cend >= c; cend--)
		if (getcharpart(&bp[cend]) != ' ')
			break;
	yp->length = n = cend - c + 1;
	if (nl)
		yp->length++;
	if ((yp->line = malloc((unsigned)(yp->length + 1))) == NULL) {
		free(yp);
		return;
	}
	for (bp += c, cp = yp->line; --n >= 0;)
		*cp++ = getcharpart(bp++);
	if (nl)
		*cp++ = '\n';
	*cp = 0;
	if (yb_head)
		yb_tail = yb_tail->link = yp;
	else
		yb_head = yb_tail = yp;
}

static void
c_put(winvars_t *winvars)
{
	struct yb *yp;

	for (yp = yb_head; yp; yp = yp->link)
		(void) write(winvars->selwin->ww_pty, yp->line, (unsigned) yp->length);
}
