/*
 * Copyright  2007 Alistair Crooks.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <sys/types.h>

#include <dirent.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <fuse.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

static struct stat	vfs;		/* stat info of signature directory */
static char		*sigdir;	/* where signed binaries really are */
static int		verbose; 	/* how chatty are we? */


/* return 1 if the signature matches */
static int
verifysig(const char *path)
{
	char	 cmd[MAXPATHLEN + BUFSIZ];

	(void) snprintf(cmd, sizeof(cmd), "gpgv %s%s.sig", sigdir, path);
	return system(cmd) == 0;
}

/* perform the stat operation */
static int 
sigfs_getattr(const char *path, struct stat *st)
{
	char	name[MAXPATHLEN];

	(void) memset(st, 0x0, sizeof(*st));
	if (strcmp(path, "/") == 0) {
		st->st_mode = S_IFDIR | 0755;
		st->st_nlink = 2;
		return 0;
	}
	(void) snprintf(name, sizeof(name), "%s%s", sigdir, path);
	return stat(name, st);
}

/* readdir operation */
static int 
sigfs_readdir(const char *path, void *buf, fuse_fill_dir_t filler,
	      off_t offset, struct fuse_file_info *fi)
{
	struct dirent	*dp;
	char		name[MAXPATHLEN];
	DIR		*dirp;

	(void) fi;

	(void) snprintf(name, sizeof(name), "%s%s", sigdir, path);
	if ((dirp = opendir(name)) == NULL) {
		return -ENOENT;
	}
	while ((dp = readdir(dirp)) != NULL) {
		(void) filler(buf, dp->d_name, NULL, 0);
	}
	(void) closedir(dirp);
	return 0;
}

/* open the file in the file system */
static int 
sigfs_open(const char *path, struct fuse_file_info *fi)
{
	if ((fi->flags & 3) != O_RDONLY) {
		return -EACCES;
	}
	if (!verifysig(path)) {
		return -EACCES;
	}
	return 0;
}

/* read the file's contents in the file system */
static int 
sigfs_read(const char *path, char *buf, size_t size, off_t offset,
	   struct fuse_file_info * fi)
{
	char	name[MAXPATHLEN];
	int	fd;
	int	cc;

	(void) fi;

	if (!verifysig(path)) {
		return -EACCES;
	}
	(void) snprintf(name, sizeof(name), "%s%s", sigdir, path);
	if ((fd = open(name, O_RDONLY, 0666)) < 0) {
		return -EACCES;
	}
	if (lseek(fd, offset, SEEK_SET) < 0) {
		(void) close(fd);
		return -EBADF;
	}
	if ((cc = read(fd, buf, size)) < 0) {
		(void) close(fd);
		return -EBADF;
	}
	(void) close(fd);
	return cc;
}

/* fill in the statvfs struct */
static int
sigfs_statfs(const char *path, struct statvfs *st)
{
	(void) memset(st, 0x0, sizeof(*st));
	st->f_bsize = st->f_frsize = st->f_iosize = 512;
	st->f_owner = vfs.st_uid;
	st->f_files = 1;
	return 0;
}

/* operations struct */
static struct fuse_operations sigfs_oper = {
	.getattr = sigfs_getattr,
	.readdir = sigfs_readdir,
	.open = sigfs_open,
	.read = sigfs_read,
	.statfs = sigfs_statfs
};

int 
main(int argc, char **argv)
{
	int	i;

	while ((i = getopt(argc, argv, "v")) != -1) {
		switch(i) {
		case 'v':
			verbose = 1;
			break;
		}
	}
	if (stat(sigdir = argv[optind], &vfs) < 0) {
		err(EXIT_FAILURE, "can't stat `%s'", argv[optind]);
	}
	return fuse_main(argc, argv, &sigfs_oper);
}
