/* $Id: error.c,v 1.1.1.1 2008/09/10 09:32:57 agcrooks Exp $ */

/*
 * Copyright (c) 2005 Manuel Freire.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <errno.h>
#include <string.h>

#include "bpg.h"
#include "error.h"

static int errnum = 0;			/* Current error number. */
static char errstr[ERRSTR_MAXLEN];	/* Current error string. */

/* Raise error with code, file and line. */
void
BPG_err_full(int num, const char *file, const int line)
{
	char errtxt[ERRSTR_MAXLEN];

	errnum = num;
	err_str(num, errtxt);

	snprintf(errstr, ERRSTR_MAXLEN - 1, "%s(%d): %s.",
		 file, line, errtxt);
}

/* Get error message. */
char *
BPG_get_error_string(void)
{

	if (*errstr == 0 && errno)
		strncpy(errstr, (char *)strerror(errno), ERRSTR_MAXLEN);

	return errstr;
}

/* Obtain the string corresponding with the given code error. */
/* TODO: add system errno support and OpenSSL too. */
void
err_str(int num, char *to)
{
	switch (num) {
	case BPGERR_MALLOC:
		snprintf(to, ERRSTR_MAXLEN - 1, "Memory allocation failure");
		break;
	case BPGERR_GET_TIME:
		snprintf(to, ERRSTR_MAXLEN - 1, "Couldn't get time");
		break;
	case BPGERR_NULL_POINTER:
		snprintf(to, ERRSTR_MAXLEN - 1, "Null pointer");
		break;
	case BPGERR_INVALID_ARGUMENT:
		snprintf(to, ERRSTR_MAXLEN - 1, "Invalid argument");
		break;
	case BPGERR_OPENSSL_DIGEST:
		snprintf(to, ERRSTR_MAXLEN - 1, "OpenSSL hashing failed");
		break;
	case BPGERR_OPENSSL_ENCRYPT:
		snprintf(to, ERRSTR_MAXLEN - 1, "OpenSSL encryption failed");
		break;
	case BPGERR_OPENSSL_SIGN:
		snprintf(to, ERRSTR_MAXLEN - 1, "OpenSSL signing failed");
		break;
	case BPGERR_MPI_TOO_SHORT:
		snprintf(to, ERRSTR_MAXLEN - 1,
			"Multi precision integer is too short");
		break;
	case BPGERR_PKT_TOO_SHORT:
		snprintf(to, ERRSTR_MAXLEN - 1, "Packet is too short");
		break;
	case BPGERR_PKT_PUBKEY_TOO_SHORT:
		snprintf(to, ERRSTR_MAXLEN - 1,
			"Public key packet is too short");
		break;
	case BPGERR_PKT_SECKEY_TOO_SHORT:
		snprintf(to, ERRSTR_MAXLEN - 1,
			"Public key packet is too short");
		break;
	case BPGERR_PKT_SIG_TOO_SHORT:
		snprintf(to, ERRSTR_MAXLEN - 1,
			"Signature packet is too short");
		break;
	case BPGERR_S2K_TOO_SHORT:
		snprintf(to, ERRSTR_MAXLEN - 1, "S2K specifier is too short");
		break;
	case BPGERR_PKT_MALFORMED:
		snprintf(to, ERRSTR_MAXLEN - 1, "Malformed packet");
		break;
	case BPGERR_PKT_SIG_MALFORMED:
		snprintf(to, ERRSTR_MAXLEN - 1, "Malformed signature packet");
		break;
	case BPGERR_PKTFORMAT_UNKNOWN:
		snprintf(to, ERRSTR_MAXLEN - 1, "Unknown packet format");
		break;
	case BPGERR_ALGORITHM_UNKNOWN:
		snprintf(to, ERRSTR_MAXLEN - 1, "Unknown algorithm");
		break;
	case BPGERR_KEYVERSION_UNKNOWN:
		snprintf(to, ERRSTR_MAXLEN - 1, "Unknown key version");
		break;
	case BPGERR_PKTTYPE_UNKNOWN:
		snprintf(to, ERRSTR_MAXLEN - 1, "Unknown packet type");
		break;
	case BPGERR_PKTLENTYPE_UNKNOWN:
		snprintf(to, ERRSTR_MAXLEN - 1, "Unknown packet lenght type");
		break;
	case BPGERR_S2K_UNKNOWN:
		snprintf(to, ERRSTR_MAXLEN - 1, "Unknown S2K type.");
		break;
	case BPGERR_SIGVERSION_UNKNOWN:
		snprintf(to, ERRSTR_MAXLEN - 1, "Unknown sig version");
		break;
	case BPGERR_ALGORITHM_NOT_SUPPORTED:
		snprintf(to, ERRSTR_MAXLEN - 1, "Algorithm not supported");
		break;
	case BPGERR_KEYVERSION_2_NOT_SUPPORTED:
		snprintf(to, ERRSTR_MAXLEN - 1, "Key version 2 not supported");
		break;
	case BPGERR_SIGVERSION_4_NOT_SUPPORTED:
		snprintf(to, ERRSTR_MAXLEN - 1,
			"Signature version 4 not supported");
		break;
	case BPGERR_PKTFORMAT_NEW_NOT_SUPPORTED:
		snprintf(to, ERRSTR_MAXLEN - 1,
			"New packet format not supported");
		break;
	case BPGERR_PKTTYPE_UNEXPECTED:
		snprintf(to, ERRSTR_MAXLEN - 1, "Unexpected packet type");
		break;
	case BPGERR_COULDNT_PARSE_SECKEY:
		snprintf(to, ERRSTR_MAXLEN - 1,
			 "Couldn't parse seckey, is that passphrase correct?");
		break;
	case BPGERR_NOT_IMPLEMENTED:
		snprintf(to, ERRSTR_MAXLEN - 1, "Not implemented");
		break;
	default:
		snprintf(to, ERRSTR_MAXLEN - 1, "Unknown error (%d)", num);
	}

	to[ERRSTR_MAXLEN] = '\0';	/* Prevent buffer overflow. */
}

