/****************************************************************
Copyright (C) Lucent Technologies 1997
All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the name Lucent Technologies or any of
its entities not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.

LUCENT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
IN NO EVENT SHALL LUCENT OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
THIS SOFTWARE.
****************************************************************/

%pure_parser

%{
#include <stdio.h>
#include <string.h>

#include "awklib.h"
#include "proto.h"

#define YYPARSE_PARAM	awkp
#define YYLEX_PARAM	awkp

void checkdup(awk_t *, awknode_t *, awkcell_t *);
int yywrap(void) { return(1); }

awknode_t	*beginloc = 0;
awknode_t	*endloc = 0;
int	inloop	= 0;	/* = 1 if in while, for, do */
%}

%union {
	awknode_t	*p;
	awkcell_t	*cp;
	int	i;
	char	*s;
}

%token	<i>	FIRSTTOKEN	/* must be first */
%token	<p>	PROGRAM PASTAT PASTAT2 XBEGIN XEND
%token	<i>	NL ',' '{' '(' '|' ';' '/' ')' '}' '[' ']'
%token	<i>	ARRAY
%token	<i>	MATCH NOTMATCH MATCHOP
%token	<i>	FINAL DOT ALL CCL NCCL CHAR OR STAR QUEST PLUS EMPTYRE
%token	<i>	AND BOR APPEND EQ GE GT LE LT NE IN
%token	<i>	ARG BLTIN BREAK CLOSE CONTINUE DELETE
%token	<i>	DLALLOC DLCALL DLCLOSE DLFIELD DLFREE DLOPEN DLPROTO
%token	<i>	DO EXIT FOR FUNC 
%token	<i>	SUB GSUB IF INDEX LSUBSTR MATCHFCN NEXT NEXTFILE
%token	<i>	ADD MINUS MULT DIVIDE MOD
%token	<i>	ASSIGN ASGNOP ADDEQ SUBEQ MULTEQ DIVEQ MODEQ POWEQ
%token	<i>	PRINT PRINTF SPRINTF
%token	<p>	ELSE INTEST CONDEXPR
%token	<i>	POSTINCR PREINCR POSTDECR PREDECR
%token	<cp>	VAR IVAR VARNF CALL NUMBER STRING
%token	<s>	REGEXPR

%type	<p>	pas pattern ppattern plist pplist patlist prarg term re
%type	<p>	pa_pat pa_stat pa_stats
%type	<s>	reg_expr
%type	<p>	simple_stmt opt_simple_stmt stmt stmtlist
%type	<p>	var varname funcname varlist
%type	<p>	for if else while
%type	<i>	do st
%type	<i>	pst opt_pst lbrace rbrace rparen comma nl opt_nl and bor
%type	<i>	subop print

%right	ASGNOP
%right	'?'
%right	':'
%left	BOR
%left	AND
%left	GETLINE
%nonassoc APPEND EQ GE GT LE LT NE MATCHOP IN '|'
%left	ARG BLTIN BREAK CALL CLOSE CONTINUE DELETE
%left	DLALLOC DLCALL DLCLOSE DLFIELD DLFREE DLOPEN DLPROTO
%left	DO EXIT FOR FUNC 
%left	GENSUB GSUB IF INDEX LSUBSTR MATCHFCN NEXT NUMBER
%left	PRINT PRINTF RETURN SPLIT SPRINTF STRING SUB SUBSTR
%left	REGEXPR VAR VARNF IVAR WHILE '('
%left	CAT
%left	'+' '-'
%left	'*' '/' '%'
%left	NOT UMINUS
%right	POWER
%right	DECR INCR
%left	INDIRECT
%token	LASTTOKEN	/* must be last */

%%

program:
	  pas	{ awk_t *lawkp = (awk_t *)awkp; if (lawkp->errorflag==0)
			lawkp->winner = (awknode_t *)awklib_stat3(awkp, PROGRAM, beginloc, $1, endloc); }
	| error	{ yyclearin; awklib_bracecheck(awkp); SYNTAX(awkp, "bailing out"); }
	;

and:
	  AND | and NL
	;

bor:
	  BOR | bor NL
	;

comma:
	  ',' | comma NL
	;

do:
	  DO | do NL
	;

else:
	  ELSE | else NL
	;

for:
	  FOR '(' opt_simple_stmt ';' opt_nl pattern ';' opt_nl opt_simple_stmt rparen {inloop++;} stmt
		{ --inloop; $$ = awklib_stat4(awkp, FOR, $3, notnull(awkp, $6), $9, $12); }
	| FOR '(' opt_simple_stmt ';'  ';' opt_nl opt_simple_stmt rparen {inloop++;} stmt
		{ --inloop; $$ = awklib_stat4(awkp, FOR, $3, NIL, $7, $10); }
	| FOR '(' varname IN varname rparen {inloop++;} stmt
		{ --inloop; $$ = awklib_stat3(awkp, IN, $3, awklib_makearr(awkp, $5), $8); }
	;

funcname:
	  VAR	{ setfname(awkp, $1); }
	| CALL	{ setfname(awkp, $1); }
	;

if:
	  IF '(' pattern rparen		{ $$ = notnull(awkp, $3); }
	;

lbrace:
	  '{' | lbrace NL
	;

nl:
	  NL | nl NL
	;

opt_nl:
	  /* empty */	{ $$ = 0; }
	| nl
	;

opt_pst:
	  /* empty */	{ $$ = 0; }
	| pst
	;


opt_simple_stmt:
	  /* empty */			{ $$ = 0; }
	| simple_stmt
	;

pas:
	  opt_pst			{ $$ = 0; }
	| opt_pst pa_stats opt_pst	{ $$ = $2; }
	;

pa_pat:
	  pattern	{ $$ = notnull(awkp, $1); }
	;

pa_stat:
	  pa_pat			{ $$ = awklib_stat2(awkp, PASTAT, $1, awklib_stat2(awkp, PRINT, awklib_rectonode(awkp), NIL)); }
	| pa_pat lbrace stmtlist '}'	{ $$ = awklib_stat2(awkp, PASTAT, $1, $3); }
	| pa_pat ',' pa_pat		{ $$ = awklib_pa2stat(awkp, $1, $3, awklib_stat2(awkp, PRINT, awklib_rectonode(awkp), NIL)); }
	| pa_pat ',' pa_pat lbrace stmtlist '}'	{ $$ = awklib_pa2stat(awkp, $1, $3, $5); }
	| lbrace stmtlist '}'		{ $$ = awklib_stat2(awkp, PASTAT, NIL, $2); }
	| XBEGIN lbrace stmtlist '}'
		{ beginloc = awklib_linkum(awkp, beginloc, $3); $$ = 0; }
	| XEND lbrace stmtlist '}'
		{ endloc = awklib_linkum(awkp, endloc, $3); $$ = 0; }
	| FUNC funcname '(' varlist rparen { awk_t *lawkp = (awk_t *)awkp; lawkp->infunc++;} lbrace stmtlist '}'
		{ awk_t *lawkp = (awk_t *)awkp;lawkp->infunc--; lawkp->curfname=0; awklib_defn(awkp, (awkcell_t *)$2, $4, $8); $$ = 0; }
	;

pa_stats:
	  pa_stat
	| pa_stats opt_pst pa_stat	{ $$ = awklib_linkum(awkp, $1, $3); }
	;

patlist:
	  pattern
	| patlist comma pattern		{ $$ = awklib_linkum(awkp, $1, $3); }
	;

ppattern:
	  var ASGNOP ppattern		{ $$ = awklib_op2(awkp, $2, $1, $3); }
	| ppattern '?' ppattern ':' ppattern %prec '?'
	 	{ $$ = awklib_op3(awkp, CONDEXPR, notnull(awkp, $1), $3, $5); }
	| ppattern bor ppattern %prec BOR
		{ $$ = awklib_op2(awkp, BOR, notnull(awkp, $1), notnull(awkp, $3)); }
	| ppattern and ppattern %prec AND
		{ $$ = awklib_op2(awkp, AND, notnull(awkp, $1), notnull(awkp, $3)); }
	| ppattern MATCHOP reg_expr	{ $$ = awklib_op3(awkp, $2, NIL, $1, (awknode_t*)awklib_makedfa(awkp, $3, 0)); }
	| ppattern MATCHOP ppattern
		{ if (constnode($3))
			$$ = awklib_op3(awkp, $2, NIL, $1, (awknode_t*)awklib_makedfa(awkp, strnode($3), 0));
		  else
			$$ = awklib_op3(awkp, $2, (awknode_t *)1, $1, $3); }
	| ppattern IN varname		{ $$ = awklib_op2(awkp, INTEST, $1, awklib_makearr(awkp, $3)); }
	| '(' plist ')' IN varname	{ $$ = awklib_op2(awkp, INTEST, $2, awklib_makearr(awkp, $5)); }
	| ppattern term %prec CAT	{ $$ = awklib_op2(awkp, CAT, $1, $2); }
	| re
	| term
	;

pattern:
	  var ASGNOP pattern		{ $$ = awklib_op2(awkp, $2, $1, $3); }
	| pattern '?' pattern ':' pattern %prec '?'
	 	{ $$ = awklib_op3(awkp, CONDEXPR, notnull(awkp, $1), $3, $5); }
	| pattern bor pattern %prec BOR
		{ $$ = awklib_op2(awkp, BOR, notnull(awkp, $1), notnull(awkp, $3)); }
	| pattern and pattern %prec AND
		{ $$ = awklib_op2(awkp, AND, notnull(awkp, $1), notnull(awkp, $3)); }
	| pattern EQ pattern		{ $$ = awklib_op2(awkp, $2, $1, $3); }
	| pattern GE pattern		{ $$ = awklib_op2(awkp, $2, $1, $3); }
	| pattern GT pattern		{ $$ = awklib_op2(awkp, $2, $1, $3); }
	| pattern LE pattern		{ $$ = awklib_op2(awkp, $2, $1, $3); }
	| pattern LT pattern		{ $$ = awklib_op2(awkp, $2, $1, $3); }
	| pattern NE pattern		{ $$ = awklib_op2(awkp, $2, $1, $3); }
	| pattern MATCHOP reg_expr	{ $$ = awklib_op3(awkp, $2, NIL, $1, (awknode_t*)awklib_makedfa(awkp, $3, 0)); }
	| pattern MATCHOP pattern
		{ if (constnode($3))
			$$ = awklib_op3(awkp, $2, NIL, $1, (awknode_t*)awklib_makedfa(awkp, strnode($3), 0));
		  else
			$$ = awklib_op3(awkp, $2, (awknode_t *)1, $1, $3); }
	| pattern IN varname		{ $$ = awklib_op2(awkp, INTEST, $1, awklib_makearr(awkp, $3)); }
	| '(' plist ')' IN varname	{ $$ = awklib_op2(awkp, INTEST, $2, awklib_makearr(awkp, $5)); }
	| pattern '|' GETLINE var	{ 
			awk_t *lawkp = (awk_t *)awkp;if (lawkp->safe) SYNTAX(lawkp, "cmd | getline is unsafe");
			else $$ = awklib_op3(awkp, GETLINE, $4, awklib_itonp($2), $1); }
	| pattern '|' GETLINE		{ 
			awk_t *lawkp = (awk_t *)awkp;if (lawkp->safe) SYNTAX(lawkp, "cmd | getline is unsafe");
			else $$ = awklib_op3(awkp, GETLINE, (awknode_t*)0, awklib_itonp($2), $1); }
	| pattern term %prec CAT	{ $$ = awklib_op2(awkp, CAT, $1, $2); }
	| re
	| term
	;

plist:
	  pattern comma pattern		{ $$ = awklib_linkum(awkp, $1, $3); }
	| plist comma pattern		{ $$ = awklib_linkum(awkp, $1, $3); }
	;

pplist:
	  ppattern
	| pplist comma ppattern		{ $$ = awklib_linkum(awkp, $1, $3); }
	;

prarg:
	  /* empty */			{ $$ = awklib_rectonode(awkp); }
	| pplist
	| '(' plist ')'			{ $$ = $2; }
	;

print:
	  PRINT | PRINTF
	;

pst:
	  NL | ';' | pst NL | pst ';'
	;

rbrace:
	  '}' | rbrace NL
	;

re:
	   reg_expr
		{ $$ = awklib_op3(awkp, MATCH, NIL, awklib_rectonode(awkp), (awknode_t*)awklib_makedfa(awkp, $1, 0)); }
	| NOT re	{ $$ = awklib_op1(awkp, NOT, notnull(awkp, $2)); }
	;

reg_expr:
	  '/' {startreg(awkp);} REGEXPR '/'		{ $$ = $3; }
	;

rparen:
	  ')' | rparen NL
	;

simple_stmt:
	  print prarg '|' term		{ 
			awk_t *lawkp = (awk_t *)awkp;if (lawkp->safe) SYNTAX(lawkp, "print | is unsafe");
			else $$ = awklib_stat3(awkp, $1, $2, awklib_itonp($3), $4); }
	| print prarg APPEND term	{
			awk_t *lawkp = (awk_t *)awkp;if (lawkp->safe) SYNTAX(lawkp, "print >> is unsafe");
			else $$ = awklib_stat3(awkp, $1, $2, awklib_itonp($3), $4); }
	| print prarg GT term		{
			awk_t *lawkp = (awk_t *)awkp;if (lawkp->safe) SYNTAX(awkp, "print > is unsafe");
			else $$ = awklib_stat3(awkp, $1, $2, awklib_itonp($3), $4); }
	| print prarg			{ $$ = awklib_stat3(awkp, $1, $2, NIL, NIL); }
	| DELETE varname '[' patlist ']' { $$ = awklib_stat2(awkp, DELETE, awklib_makearr(awkp, $2), $4); }
	| DELETE varname		 { $$ = awklib_stat2(awkp, DELETE, awklib_makearr(awkp, $2), 0); }
	| pattern			{ $$ = awklib_exptostat($1); }
	| error				{ awk_t *lawkp = (awk_t *)awkp;yyclearin; SYNTAX(lawkp, "illegal statement"); }
	;

st:
	  nl
	| ';' opt_nl
	;

stmt:
	  BREAK st		{ if (!inloop) SYNTAX(awkp, "break illegal outside of loops");
				  $$ = stat1(awkp, BREAK, NIL); }
	| CONTINUE st		{  if (!inloop) SYNTAX(awkp, "continue illegal outside of loops");
				  $$ = stat1(awkp, CONTINUE, NIL); }
	| do {inloop++;} stmt {--inloop;} WHILE '(' pattern ')' st
		{ $$ = awklib_stat2(awkp, DO, $3, notnull(awkp, $7)); }
	| EXIT pattern st	{ $$ = stat1(awkp, EXIT, $2); }
	| EXIT st		{ $$ = stat1(awkp, EXIT, NIL); }
	| for
	| if stmt else stmt	{ $$ = awklib_stat3(awkp, IF, $1, $2, $4); }
	| if stmt		{ $$ = awklib_stat3(awkp, IF, $1, $2, NIL); }
	| lbrace stmtlist rbrace { $$ = $2; }
	| NEXT st	{ awk_t *lawkp = (awk_t *)awkp;if (lawkp->infunc)
				SYNTAX(lawkp, "next is illegal inside a function");
			  $$ = stat1(awkp, NEXT, NIL); }
	| NEXTFILE st	{ awk_t *lawkp = (awk_t *)awkp;if (lawkp->infunc)
				SYNTAX(lawkp, "nextfile is illegal inside a function");
			  $$ = stat1(awkp, NEXTFILE, NIL); }
	| RETURN pattern st	{ $$ = stat1(awkp, RETURN, $2); }
	| RETURN st		{ $$ = stat1(awkp, RETURN, NIL); }
	| simple_stmt st
	| while {inloop++;} stmt	{ --inloop; $$ = awklib_stat2(awkp, WHILE, $1, $3); }
	| ';' opt_nl		{ $$ = 0; }
	;

stmtlist:
	  stmt
	| stmtlist stmt		{ $$ = awklib_linkum(awkp, $1, $2); }
	;

subop:
	  SUB | GSUB
	;

term:
 	  term '/' ASGNOP term		{ $$ = awklib_op2(awkp, DIVEQ, $1, $4); }
 	| term '+' term			{ $$ = awklib_op2(awkp, ADD, $1, $3); }
	| term '-' term			{ $$ = awklib_op2(awkp, MINUS, $1, $3); }
	| term '*' term			{ $$ = awklib_op2(awkp, MULT, $1, $3); }
	| term '/' term			{ $$ = awklib_op2(awkp, DIVIDE, $1, $3); }
	| term '%' term			{ $$ = awklib_op2(awkp, MOD, $1, $3); }
	| term POWER term		{ $$ = awklib_op2(awkp, POWER, $1, $3); }
	| '-' term %prec UMINUS		{ $$ = awklib_op1(awkp, UMINUS, $2); }
	| '+' term %prec UMINUS		{ $$ = $2; }
	| NOT term %prec UMINUS		{ $$ = awklib_op1(awkp, NOT, notnull(awkp, $2)); }
	| BLTIN '(' ')'			{ $$ = awklib_op2(awkp, BLTIN, awklib_itonp($1), awklib_rectonode(awkp)); }
	| BLTIN '(' patlist ')'		{ $$ = awklib_op2(awkp, BLTIN, awklib_itonp($1), $3); }
	| BLTIN				{ $$ = awklib_op2(awkp, BLTIN, awklib_itonp($1), awklib_rectonode(awkp)); }
	| CALL '(' ')'			{ $$ = awklib_op2(awkp, CALL, awklib_celltonode(awkp, $1,CVAR), NIL); }
	| CALL '(' patlist ')'		{ $$ = awklib_op2(awkp, CALL, awklib_celltonode(awkp, $1,CVAR), $3); }
	| CLOSE term			{ $$ = awklib_op1(awkp, CLOSE, $2); }
	| DECR var			{ $$ = awklib_op1(awkp, PREDECR, $2); }
	| INCR var			{ $$ = awklib_op1(awkp, PREINCR, $2); }
	| var DECR			{ $$ = awklib_op1(awkp, POSTDECR, $1); }
	| var INCR			{ $$ = awklib_op1(awkp, POSTINCR, $1); }
	| GENSUB '(' reg_expr comma pattern comma pattern ')'
		{ $$ = awklib_op5(awkp, GENSUB, NIL, (awknode_t*)awklib_makedfa(awkp, $3, 1), $5, $7, awklib_rectonode(awkp)); }
	| GENSUB '(' pattern comma pattern comma pattern ')'
		{ if (constnode($3))
			$$ = awklib_op5(awkp, GENSUB, NIL, (awknode_t *)awklib_makedfa(awkp, strnode($3), 1), $5, $7, awklib_rectonode(awkp));
		  else
			$$ = awklib_op5(awkp, GENSUB, (awknode_t *)1, $3, $5, $7, awklib_rectonode(awkp));
		}
	| GENSUB '(' reg_expr comma pattern comma pattern comma pattern ')'
		{ $$ = awklib_op5(awkp, GENSUB, NIL, (awknode_t*)awklib_makedfa(awkp, $3, 1), $5, $7, $9); }
	| GENSUB '(' pattern comma pattern comma pattern comma pattern ')'
		{ if (constnode($3))
			$$ = awklib_op5(awkp, GENSUB, NIL, (awknode_t *)awklib_makedfa(awkp, strnode($3),1), $5,$7,$9);
		  else
			$$ = awklib_op5(awkp, GENSUB, (awknode_t *)1, $3, $5, $7, $9);
		}
	| GETLINE var LT term		{ $$ = awklib_op3(awkp, GETLINE, $2, awklib_itonp($3), $4); }
	| GETLINE LT term		{ $$ = awklib_op3(awkp, GETLINE, NIL, awklib_itonp($2), $3); }
	| GETLINE var			{ $$ = awklib_op3(awkp, GETLINE, $2, NIL, NIL); }
	| GETLINE			{ $$ = awklib_op3(awkp, GETLINE, NIL, NIL, NIL); }
	| INDEX '(' pattern comma pattern ')'
		{ $$ = awklib_op2(awkp, INDEX, $3, $5); }
	| INDEX '(' pattern comma reg_expr ')'
		{ SYNTAX(awkp, "index() doesn't permit regular expressions");
		  $$ = awklib_op2(awkp, INDEX, $3, (awknode_t*)$5); }
	| '(' pattern ')'		{ $$ = $2; }
	| MATCHFCN '(' pattern comma reg_expr ')'
		{ $$ = awklib_op3(awkp, MATCHFCN, NIL, $3, (awknode_t*)awklib_makedfa(awkp, $5, 1)); }
	| MATCHFCN '(' pattern comma pattern ')'
		{ if (constnode($5))
			$$ = awklib_op3(awkp, MATCHFCN, NIL, $3, (awknode_t*)awklib_makedfa(awkp, strnode($5), 1));
		  else
			$$ = awklib_op3(awkp, MATCHFCN, (awknode_t *)1, $3, $5); }
	| NUMBER			{ $$ = awklib_celltonode(awkp, $1, CCON); }
	| SPLIT '(' pattern comma varname comma pattern ')'     /* string */
		{ $$ = awklib_op4(awkp, SPLIT, $3, awklib_makearr(awkp, $5), $7, (awknode_t*)STRING); }
	| SPLIT '(' pattern comma varname comma reg_expr ')'    /* const /regexp/ */
		{ $$ = awklib_op4(awkp, SPLIT, $3, awklib_makearr(awkp, $5), (awknode_t*)awklib_makedfa(awkp, $7, 1), (awknode_t *)REGEXPR); }
	| SPLIT '(' pattern comma varname ')'
		{ $$ = awklib_op4(awkp, SPLIT, $3, awklib_makearr(awkp, $5), NIL, (awknode_t*)STRING); }  /* default */
	| SPRINTF '(' patlist ')'	{ $$ = awklib_op1(awkp, $1, $3); }
	| STRING	 		{ $$ = awklib_celltonode(awkp, $1, CCON); }
	| DLALLOC '(' pattern ')'	{ $$ = awklib_op1(awkp, $1, $3); }
	| DLFIELD '(' pattern comma pattern comma pattern comma pattern ')'
		{ $$ = awklib_op4(awkp, DLFIELD, $3, $5, $7, $9); }
	| DLFREE '(' pattern ')'	{ $$ = awklib_op1(awkp, $1, $3); }
	| DLOPEN '(' varname comma pattern ')'
		{ $$ = awklib_op4(awkp, DLOPEN, $5, awklib_makearr(awkp, $3), NIL, (awknode_t*)STRING); }
	| DLCALL '(' pattern comma patlist ')'
		{ $$ = awklib_op2(awkp, DLCALL, $3, (awknode_t*)$5); }
	| DLCLOSE '(' pattern ')'	{ $$ = awklib_op1(awkp, $1, $3); }
	| DLPROTO '(' varname comma pattern ')'
		{ $$ = awklib_op2(awkp, DLPROTO, $3, $5); }
	| subop '(' reg_expr comma pattern ')'
		{ $$ = awklib_op4(awkp, $1, NIL, (awknode_t*)awklib_makedfa(awkp, $3, 1), $5, awklib_rectonode(awkp)); }
	| subop '(' pattern comma pattern ')'
		{ if (constnode($3))
			$$ = awklib_op4(awkp, $1, NIL, (awknode_t*)awklib_makedfa(awkp, strnode($3), 1), $5, awklib_rectonode(awkp));
		  else
			$$ = awklib_op4(awkp, $1, (awknode_t *)1, $3, $5, awklib_rectonode(awkp)); }
	| subop '(' reg_expr comma pattern comma var ')'
		{ $$ = awklib_op4(awkp, $1, NIL, (awknode_t*)awklib_makedfa(awkp, $3, 1), $5, $7); }
	| subop '(' pattern comma pattern comma var ')'
		{ if (constnode($3))
			$$ = awklib_op4(awkp, $1, NIL, (awknode_t*)awklib_makedfa(awkp, strnode($3), 1), $5, $7);
		  else
			$$ = awklib_op4(awkp, $1, (awknode_t *)1, $3, $5, $7); }
	| SUBSTR '(' pattern comma pattern comma pattern ')'
		{ $$ = awklib_op3(awkp, SUBSTR, $3, $5, $7); }
	| SUBSTR '(' pattern comma pattern ')'
		{ $$ = awklib_op3(awkp, SUBSTR, $3, $5, NIL); }
	| var
	;

var:
	  varname
	| varname '[' patlist ']'	{ $$ = awklib_op2(awkp, ARRAY, awklib_makearr(awkp, $1), $3); }
	| IVAR				{ $$ = awklib_op1(awkp, INDIRECT, awklib_celltonode(awkp, $1, CVAR)); }
	| INDIRECT term	 		{ $$ = awklib_op1(awkp, INDIRECT, $2); }
	;	

varlist:
	  /* nothing */		{ awk_t *lawkp = (awk_t *)awkp;lawkp->arglist = $$ = 0; }
	| VAR			{ awk_t *lawkp = (awk_t *)awkp;lawkp->arglist = $$ = awklib_celltonode(awkp, $1,CVAR); }
	| varlist comma VAR	{
			awk_t *lawkp = (awk_t *)awkp;checkdup(awkp, $1, $3);
			lawkp->arglist = $$ = awklib_linkum(awkp, $1,awklib_celltonode(awkp, $3,CVAR)); }
	;

varname:
	  VAR			{ $$ = awklib_celltonode(awkp, $1, CVAR); }
	| ARG 			{ $$ = awklib_op1(awkp, ARG, awklib_itonp($1)); }
	| VARNF			{ $$ = awklib_op1(awkp, VARNF, (awknode_t *) $1); }
	;


while:
	  WHILE '(' pattern rparen	{ $$ = notnull(awkp, $3); }
	;

%%

void
setfname(awk_t *awkp, awkcell_t *p)
{
	if (ISARR(p))
		SYNTAX(awkp, "%s is an array, not a function", p->nval);
	else if (ISFCN(p))
		SYNTAX(awkp, "you can't define function %s more than once", p->nval);
	awkp->curfname = p->nval;
}

int
constnode(awknode_t *p)
{
	return ISVALUE(p) && ((awkcell_t *) (p->narg[0]))->csub == CCON;
}

char *
strnode(awknode_t *p)
{
	return ((awkcell_t *)(p->narg[0]))->sval;
}

awknode_t *
notnull(awk_t *awkp, awknode_t *n)
{
	switch (n->nobj) {
	case LE: case LT: case EQ: case NE: case GT: case GE:
	case BOR: case AND: case NOT:
		return n;
	default:
		return awklib_op2(awkp, NE, n, awkp->nullnode);
	}
}

/* check if name already in list */
void
checkdup(awk_t *awkp, awknode_t *vl, awkcell_t *cp)
{
	char *s = cp->nval;
	for ( ; vl; vl = vl->nnext) {
		if (strcmp(s, ((awkcell_t *)(vl->narg[0]))->nval) == 0) {
			SYNTAX(awkp, "duplicate argument %s", s);
			break;
		}
	}
}
