/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <ctype.h>
#include <string.h>

#include "paneld.h"

static int	hex_open(struct menu *m);
static int	hex_close(struct menu *m);
static int	hex_next(struct menu *m);
static int	hex_prev(struct menu *m);
static int	hex_inc(struct menu *m);
static int	hex_dec(struct menu *m);
static int	hex_select(struct menu *m);
static char*	hex_allocbuf(int *sz);

static int
hex_open(struct menu *m)
{
	int i;
	int len;

	if (exec_read(m) != E_OK)
		return E_IO;

	len = strlen(m->io.buf);
	if (len < m->io.len) {
		int off = m->io.len - len;
		memcpy(m->io.buf + off, m->io.buf, len);
		memset(m->io.buf, '0', off);
		m->io.buf[m->io.len] = 0;
	}

	for (i = 0; m->io.buf[i] != 0; i++) {
		if (!isdigit(m->io.buf[i])) {
			if ((m->io.buf[i] >= 'a') &&
					(m->io.buf[i] <= 'f'))
				continue;
			else if ((m->io.buf[i] >= 'A') &&
					(m->io.buf[i] <= 'F'))
				m->io.buf[i] = tolower(m->io.buf[i]);
			else
				return E_IO;
		}
	}

	return E_OK;
}

static int
hex_close(struct menu *m)
{
	util_intstrip(m->io.buf);
	return exec_write(m);
}

static int
hex_next(struct menu *m)
{
	return window->edit()->right();
}

static int
hex_prev(struct menu *m)
{
	return window->edit()->left();
}

static int
hex_inc(struct menu *m)
{
	int c;
	if (window->edit()->getchr(&c) == E_OK) {
		if (c == '9')
			return window->edit()->putchr('a');
		else if (c == 'f')
			return window->edit()->putchr('0');
		else
			return window->edit()->putchr(++c);
	}
	return E_IO;
}

static int
hex_dec(struct menu *m)
{
	int c;
	if (window->edit()->getchr(&c) == E_OK) {
		if (c == '0')
			return window->edit()->putchr('f');
		else if (c == 'a')
			return window->edit()->putchr('9');
		else
			return window->edit()->putchr(--c);
	}
	return E_IO;
}

static int
hex_select(struct menu *m)
{
	if (hex_open(m) == E_OK) {
		window->set_title(m->nm);
		window->edit()->setbuf(m->io.buf, ALIGN_ANY);

		if (enter_menu(m) == E_SELECT)
			return hex_close(m);
	}

	return E_IO;
}

static char*
hex_allocbuf(int *sz)
{
	return (char*)cf_malloc(*sz + 1);
}

struct menu_ops*
get_hex_ops(void)
{
	static struct menu_ops hex_ops = {
		hex_next,
		hex_prev,
		hex_inc,
		hex_dec,
		hex_select,
		hex_allocbuf
	};
	return &hex_ops;
}
