/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/wait.h>

#include "paneld.h"

#define MAX_ARG	255

/*
 * Parse command arguments and store them into an array
 * suitable for exec() calls
 */
int
parse_args(char *buf, char ***r_args)
{
	int i;

	char *ptr = buf;
	char *arr[MAX_ARG];

	for (i = 0; i < MAX_ARG; ) {
		arr[i++] = ptr;

		ptr = strchr(ptr, ' ');
		if (ptr == NULL) {
			char **args = (char **)malloc(sizeof(char*) * (i + 1));
			if (args == NULL)
				break;

			memcpy(args, arr, sizeof(char*) * i);
			args[i] = NULL;

			*r_args = args;
			return E_OK;
		}

		*ptr++ = 0;
	}

	return E_IO;
}

/*
 * Execute external command and return its output in a supplied
 * buffer
 */
int
readconf(char *args[], char *buf, int *sz)
{
	int fd[2];
	pid_t pid;
	int size = (sz != NULL) ? *sz : 0;

	if (pipe(fd) < 0)
		return E_IO;

	if ( (pid = fork()) < 0)
		return E_IO;
	else if (pid > 0) {	/* parent */
		int rd = 0;
		int cnt = 0;
		int status;

		close(fd[1]);

		if ((buf != NULL) && (size > 0)) {
			int i;
			while ( (cnt = read(fd[0], buf + rd, size)) != 0) {
				rd += cnt;
				size -= cnt;
				if (size <= 0) {
					rd = *sz - 1;
					break;
				}
			}

			for (i = 0; i < rd; i++) {
				if ((buf[i] == '\n') || (buf[i] == '\r')) {
					buf[i] = 0;
					break;
				}
			}

			buf[rd] = 0;
		}

		close(fd[0]);

		if (waitpid(pid, &status, 0) == pid) {
			util_trace(LOG_INFO,
					"cmd: %s, ifexited: %d, "
					"status: %d, size: %d",
					args[0], WIFEXITED(status),
					WEXITSTATUS(status), rd);
			if (WEXITSTATUS(status) == 0) {
				if (sz != NULL) {
					*sz = rd;
				}
				return E_OK;
			}
		}

		util_trace(LOG_WARNING, "cmd %s failed", args[0]);
	} else {		/* child  */
		close(fd[0]);
		if (fd[1] != STDOUT_FILENO)
			if (dup2(fd[1], STDOUT_FILENO) != STDOUT_FILENO)
			exit(1);

		if (fd[1] != STDERR_FILENO)
			if (dup2(fd[1], STDERR_FILENO) != STDERR_FILENO)
			exit(1);
		close(fd[1]);

		execv(args[0], args);
		exit(1);
	}

	return E_IO;
}

/*
 * Execute .read argument
 */
int
exec_read(struct menu *m)
{
	int size = m->io.len;

	util_trace(LOG_DEBUG, "menu [%s], read()", m->nm);

	if (m->io.rargs == NULL)
		return E_OK;
	if ((m->io.len == 0) || (m->io.buf == NULL))
		return E_IO;
	return readconf(m->io.rargs, m->io.buf, &size);
}

/*
 * Execute .write argument
 */
int
exec_write(struct menu *m)
{
	util_trace(LOG_DEBUG, "menu [%s], write()", m->nm);

	if (m->io.wargs == NULL)
		return E_OK;
	return readconf(m->io.wargs, NULL, NULL);
}
