/* $NetBSD$ */

%{
/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <err.h>
#include <stdio.h>

#include "paneld.h"
#include "cf-lex.h"

static struct menu*	cf_pop_menu(struct menu*);
static struct menu*	cf_push_menu(struct menu*);

static void		cf_root_menu(struct menu*);
static struct menu*	cf_add_child(struct menu*);
static struct menu*	cf_add_sibling(struct menu*, struct menu*);
static struct menu*	cf_create_menu(void);
static void		cf_create_xlate(void);
static struct lcd_xlate* cf_lookup_xlate(const char*);

static char*		cf_resolve_ref(struct menu*, char*);
static void		cf_link_menu(struct menu*, struct menu*);
static void		cf_resolve_args(struct menu*, struct menu*, char**);
static char*		get_default_buf(int *sz);

/*#define YYDEBUG	1*/
#define MAX_BANNER_SIZE	255
%}
%union {
	int		num;
	char		c;
	char		*str;
	struct menu	*menu;
	struct menu_ops *ops;
}

%type	<ops> type_expr
%type	<num> text_expr
%type	<str> id_expr
%type	<c> inc_expr dec_expr next_expr prev_expr enter_expr
%type	<menu> menu menu_start stmts stmt

%token	<c> TK_CHAR
%token	<num> TK_INTEGER
%token	<str> TK_STRING TK_INET TK_INET6 TK_TEXT TK_INT TK_HEX
%token	TK_MENU TK_TITLE TK_READ TK_WRITE TK_START
%token	TK_RESIST TK_REPEAT TK_IDENTIFIER TK_UNKNOWN
%token	TK_DEVNAME TK_DEVNODE TK_EXIT TK_QUIT TK_TYPE TK_LEAF
%token	TK_SELECT TK_SEQUENCE TK_CONFIRM TK_MENUID
%token	TK_NONE TK_XLATE TK_INC TK_DEC TK_NEXT TK_PREV TK_ENTER
%token	TK_XTBL TK_BANNER TK_KEYRATE

%%
file:
	stmts
		{ cf_root_menu($1); }
	;

stmts:
	/* empty */
		{ $$ = NULL; }
	| stmt stmts
		{ $$ = $2; }
	| menu stmts
		{ $$ = cf_add_sibling($1, $2); }
	;

stmt:
	expr ';'
		{ $$ = NULL; }
	| xlate_stmt
		{ $$ = NULL; }
	;

menu:
	menu_start stmts '}'
		{ cf_add_child($2); $$ = cf_pop_menu($1); }
	;

menu_start:
	check_root TK_MENU '=' '{'
		{ $$ = cf_push_menu(cf_create_menu()); }
	;

check_root:
	{ if (!root && $<menu>0) yyerror("too many root menus"); }
	;

local:
	{ if (!root) yyerror("expression is not allowed in global scope"); }
	;

global:
	{ if (root) yyerror("expression is not allowed in local scope"); }
	;

expr:
	local name_expr
	| local write_expr
	| local read_expr
	| local id_expr
		{ root->id = $2; }
	| local type_expr
		{ root->ops = $2; }
	| local xlateid_expr
	| global resist_expr
	| global repeat_expr
	| global keyrate_expr
	| global device_expr
	| global banner_expr
	| global start_expr
	;

id_expr:
	TK_MENUID '=' TK_STRING
		{ $$ = cf_strdup($3); }
	;

type_expr:
	type_start TK_EXIT
		{ $$ = get_exit_ops(); }
	| type_start TK_QUIT
		{ $$ = get_quit_ops(); }
	| type_start TK_NONE
		{ $$ = get_none_ops(); }
	| type_start TK_SELECT
		{ $$ = get_select_ops(); }
	| type_start TK_SEQUENCE
		{ $$ = get_sequence_ops(); }
	| type_start TK_CONFIRM
		{ $$ = get_confirm_ops(); }
	| type_start text_expr
		{ $$ = get_text_ops(); }
	| type_start TK_INET
		{ $$ = get_inet_ops(); }
	| type_start TK_INET6
		{ $$ = get_inet6_ops(); }
	| type_start int_expr
		{ $$ = get_int_ops(); }
	| type_start hex_expr
		{ $$ = get_hex_ops(); }
	;

xlateid_expr:
	TK_XLATE '=' TK_STRING
		{ root->xlate = cf_lookup_xlate($3); }
	;

type_start:
	TK_TYPE '='
	;

name_expr:
	TK_TITLE '=' TK_STRING
		{ root->nm = cf_strdup($3); }
	;

size_expr:
	'[' TK_INTEGER ']'
		{ root->io.len = $2; }
	;

int_expr:
	TK_INT size_expr
	;

hex_expr:
	TK_HEX size_expr
	;

text_expr:
	TK_TEXT size_expr
	;

read_expr:
	TK_READ '=' TK_STRING
		{ root->io.read = cf_strdup($3); }
	;

write_expr:
	TK_WRITE '=' TK_STRING
		{ root->io.write = cf_strdup($3); }
	;

device_expr:
	TK_DEVNAME '=' TK_STRING
		{ globals->dev_name = cf_strdup($3); }
	| TK_DEVNODE '=' TK_STRING
		{ globals->dev_node = cf_strdup($3); }
	;

resist_expr:
	TK_RESIST '=' TK_INTEGER
		{ globals->devcap.resist = $3; }
	;

repeat_expr:
	TK_REPEAT '=' TK_INTEGER
		{ globals->devcap.repeat = $3; }
	;

keyrate_expr:
	TK_KEYRATE '=' TK_INTEGER
		{ globals->devcap.keyrate = $3; }

xlate_stmt:
	xlate_start xlate_expr_list '}'
	;

xlate_expr_list:
	/* empty */
	| xlate_expr ';' xlate_expr_list
	;

xlate_expr:
	id_expr
		{ xlate->id = $1; }
	| inc_expr
		{ xlate->x_inc = $1; }
	| dec_expr
		{ xlate->x_dec = $1; }
	| next_expr
		{ xlate->x_next = $1; }
	| prev_expr
		{ xlate->x_prev = $1; }
	| enter_expr
		{ xlate->x_enter = $1; }
	;

inc_expr:
	TK_INC '=' TK_CHAR
		{ $$ = $3; }
	;

dec_expr:
	TK_DEC '=' TK_CHAR
		{ $$ = $3; }
	;

next_expr:
	TK_NEXT '=' TK_CHAR
		{ $$ = $3; }
	;

prev_expr:
	TK_PREV '=' TK_CHAR
		{ $$ = $3; }
	;

enter_expr:
	TK_ENTER '=' TK_CHAR
		{ $$ = $3; }
	;

banner_expr:
	TK_BANNER '=' TK_STRING
		{
			globals->banner_cmd = cf_strdup($3);
			if (parse_args(globals->banner_cmd,
					&globals->banner_args) != E_OK)
				yyerror("malformed banner command");
			globals->buf_size = MAX_BANNER_SIZE;
			globals->banner_buf
				= (char*)cf_malloc(MAX_BANNER_SIZE);
		}
	;

start_expr:
	TK_START '=' TK_CHAR
		{ globals->x_enter = $3; }

xlate_start:
	TK_XTBL '=' '{'
		{ cf_create_xlate(); }
	;
%%
#define DEFAULT_BUFSZ	128

static char*
get_default_buf(int *sz)
{
	*sz = DEFAULT_BUFSZ;
	return (char*)cf_malloc(DEFAULT_BUFSZ);
}

/*
 * Try to resolve a reference to another menu from external
 * command's argument list
 */
static char*
cf_resolve_ref(struct menu *m, char *id)
{
	for (; m != NULL; m = m->prev) {
		if ((m->id != NULL) && !strcmp(m->id, id)) {
			if (m->io.buf == NULL)
				m->io.buf = m->ops->allocbuf(&m->io.len);
			if (m->io.buf == NULL)
				m->io.buf = get_default_buf(&m->io.len);
			return m->io.buf;
		}
	}
	return NULL;
}

/*
 * Resolve all references to the menu ids
 */
static void
cf_resolve_args(struct menu *p, struct menu *m, char **args)
{
	int i;
	int seq = (p->ops == get_sequence_ops());
	for (i = 0; args[i] != NULL; i++) {
		if (*args[i] == '$') {
			char *buf;
			if (!seq)
				errx(EXIT_SUCCESS,
					"'%s' must have sequential type to "
					"allow argument references in '%s'",
					p->nm, m->nm);

			buf = cf_resolve_ref(m, args[i] + 1);
			if (buf == NULL)
				errx(EXIT_SUCCESS,
					"cannot resolve symbol '%s' "
					"referenced in '%s'",
					args[i] + 1, m->nm);
			args[i] = buf;
		}
	}
}

/*
 * Initialize argument arrays for the specified actions, if any
 */
static void
cf_link_menu(struct menu *parent, struct menu *child)
{
	int err;

	for (; child != NULL; child = child->next) {
		if (child->io.buf == NULL)
			child->io.buf = child->ops->allocbuf(&child->io.len);
		if (child->io.read != NULL) {
			err = parse_args(child->io.read, &child->io.rargs);
			if (err != E_OK)
				errx(EXIT_SUCCESS,
					"cannot parse .read in menu '%s'",
					child->nm);
		}
		if (child->io.write != NULL) {
			err = parse_args(child->io.write, &child->io.wargs);
			if (err != E_OK)
				errx(EXIT_SUCCESS,
					"cannot parse .write in menu '%s'",
					child->nm);
		}

		if (child->io.rargs != NULL)
			cf_resolve_args(parent, child, child->io.rargs);
		if (child->io.wargs != NULL)
			cf_resolve_args(parent, child, child->io.wargs);
	}
}

/*
 * Pop menu off the stack
 */
static struct menu*
cf_pop_menu(struct menu* m)
{
	struct menu *prev = root;

	if (root->nm == NULL)
		root->nm = "<no name>";

	if (root->ops == NULL)
		yyerror("menu has no type");

	cf_link_menu(root, root->child);

	root = m;
	return prev;
}

/*
 * Push menu on the stack
 */
static struct menu*
cf_push_menu(struct menu *m)
{
	struct menu *prev = root;
	root = m;
	return prev;
}

/*
 * Set root menu
 */
static void
cf_root_menu(struct menu* m)
{
	root = m;
}

/*
 * Add childs to a parent menu
 */
static struct menu*
cf_add_child(struct menu* child)
{
	if (root != NULL)
		root->child = child;
	return root;
}

/*
 * Add sibling to a menu
 */
static struct menu*
cf_add_sibling(struct menu* root, struct menu* m)
{
	root->next = m;
	if (m != NULL)
		m->prev = root;

	return root;
}

/*
 * Allocate menu structure
 */
static struct menu*
cf_create_menu(void)
{
	struct menu *m;

	m = (struct menu*)cf_malloc(sizeof(struct menu));
	memset(m, 0, sizeof(struct menu));

	return m;
}

/*
 * Allocate translation table
 */
static void
cf_create_xlate(void)
{
	struct lcd_xlate *x;

	x = (struct lcd_xlate*)cf_malloc(sizeof(struct lcd_xlate));
	memset(x, 0, sizeof(struct lcd_xlate));

	x->next = xlate;
	xlate = x;
}

/*
 * Lookup translation table by identifier
 */
static struct lcd_xlate*
cf_lookup_xlate(const char *x_nm)
{
	struct lcd_xlate *x = xlate;

	if (x_nm == NULL)
		return NULL;

	for (; x != NULL; x = x->next)
		if (!strcmp(x->id, x_nm))
		break;

	if (x == NULL)
		yyerror("unknown xlate id");

	return x;
}

/*
 * Parse the configuration file
 */
void
cf_parse(const char *cfile) {
/*	yydebug=1;*/

	root = NULL;
	xlate = NULL;

	globals = (struct lcd_globals*)cf_malloc(sizeof(struct lcd_globals));
	memset(globals, 0, sizeof(struct lcd_globals));

	yylex_init(cfile);
	(void)yyparse();
	yylex_fini();
}
